#!/usr/bin/perl

use Direct::Modern;
use my_inc '../..';
use List::Util qw/pairs/;
use List::MoreUtils qw/uniq/;

use Yandex::DBTools;
use Yandex::DBShards;

use Settings;
use ScriptHelper;
use PrimitivesIds;
use User ();
use VCards;
use Campaign::Types qw/camp_kind_in/;

=head1 SYNOPSIS

Скрипт для удаления дублей клиентов по шардам
Копия с direct-client.pl

=cut

my %SHARD_BY_DUPLICATED_CLIENTS = (
    37428085 => 14,
    37452349 => 16,
    48881736 => 21,
    59773411 => 21,
    60820537 => 18,
    63211367 => 6
);

$log->out('START');

main();

$log->out('FINISH');

sub main {
    for my $ClientID (keys %SHARD_BY_DUPLICATED_CLIENTS) {
        my $shard = $SHARD_BY_DUPLICATED_CLIENTS{$ClientID};
        my $g = $log->msg_prefix_guard("[shard=$shard, client_id=$ClientID]");
        
        my $uids = get_uids_on_shard($shard, $ClientID);

        for my $camp (@{get_all_sql(PPC(shard => $shard), [ "SELECT cid, uid FROM campaigns", WHERE => { uid => $uids } ])}) {
            $log->out("delete campaign $camp->{cid} on shard $shard");
            delete_campaign_from_specific_shard($shard, $ClientID, $camp->{cid}, $camp->{uid});
        }

        for my $uid (@$uids) {
            $log->out("delete user $uid on shard $shard");
            delete_user_from_specific_shard($shard, $uid, $ClientID);
        }

        # uid-ы у дублирующихся клиентов дублируются, поэтому не реализовавывал удаление из new_users_data, как и delete_from_rbac
    }
}

sub get_uids_on_shard {
    my ($shard, $ClientID) = @_;
    my $uids = get_one_column_sql(PPC(shard => $shard), "SELECT uid FROM users WHERE ClientID = ?", $ClientID);
    return [uniq @$uids];
}

# Копия User::delete_user_from_db, но с удалением с определенного шарда
sub delete_user_from_specific_shard {
    my ($shard, $uid, $ClientID) = @_;

    die "delete_user_from_db() require uid" unless $uid;

    my $login = get_login(uid => $uid);

    my $rows_count = 0;
    do_in_transaction {
        for my $users_table_name (keys %User::USER_TABLES) {
            $rows_count += do_delete_from_table(PPC(shard => $shard), $users_table_name, where => {uid => $uid});
        }

        if (!get_one_field_sql(PPC(shard => $shard), "SELECT 1 FROM users WHERE ClientID = ? LIMIT 1", $ClientID)) {
            for my $table (@User::TABLES_TO_CLEAN_BY_CLIENTID) {
                $rows_count += do_delete_from_table(PPC(shard => $shard), $table, where => {ClientID => $ClientID});
            }
        }
    };

    return $rows_count;
}

use constant DELETE_RULES => (
    # delete banners
    bid => [qw/banner_images banners_mobile_content banners_performance banners_additions banner_display_hrefs images banner_resources
               banners_minus_geo aggregator_domains moderate_banner_pages banner_moderation_versions
               banner_measurers banners_tns banner_permalinks banner_phones banner_additional_hrefs banner_turbolandings/],
    cid => [qw/banners/],
    # delete show conditions
    cid => [qw/bids_href_params bids_manual_prices bids_arc bids bids_base/],
    pid => [qw/bids_retargeting bids_dynamic dynamic_conditions bids_performance/],
    # delete adgroups
    pid => [qw/group_params adgroups_internal adgroup_additional_targetings adgroups_mobile_content adgroups_performance adgroups_dynamic adgroups_cpm_banner adgroups_cpm_video adgroup_project_params adgroup_page_targets adgroups_minus_words video_segment_goals phrases/],
    # delete mediaplans
    mbid => [qw/mediaplan_bids_retargeting/],
    cid => [qw/mediaplan_bids mediaplan_banners_original mediaplan_banners autobudget_forecast/],
    bid => [[qw/redirect_check_queue/], object_id => {object_type => 'banner'}],

    bid => [['mod_object_version'], obj_id => {obj_type => [qw/banner contactinfo sitelinks_set image display_href/]}],
    pid => [['mod_object_version'], obj_id => {obj_type => ['phrases']}],

    # delete dialogs links
    cid => [qw/camp_dialogs/],
);

# Копия Campaign::del_camp, но с удалением с определенного шарда
sub delete_campaign_from_specific_shard {
    my ($shard, $ClientID, $cid, $uid) = @_;

    # привязки менеджеров не удаляются, ибо у кампаний для текущего one-shot-а ManagerUID отсутствуют
    # RBACDirect::rbac_delete_campaign($rbac, $cid, $client_uid);

    return unless ( $cid =~ m/^\d+$/ && $uid );

    my $camp = get_one_line_sql(
        PPC(shard => $shard), "
        SELECT IFNULL(type, 'text') AS mediaType
        FROM campaigns
        WHERE cid = ?
        ", $cid
    );

    # не снимаем кампанию с обслуживания в Балансе, ибо у кампаний для текущего one-shot-а ManagerUID и AgencyID отсутствуют
    # Campaign::unservice_orders_in_balance($shard, [$cid]);

    # аналогично, не останавливаем АВ-эксперименты
    # Experiments::stop_experiment_on_delete_campaign($ClientID, $cid);

    my $vcards_to_delete = get_one_column_sql(PPC(shard => $shard), 'SELECT DISTINCT vcard_id FROM vcards WHERE cid = ?', $cid) || [];

    my @delete_by_cid = qw/
        campaigns_mobile_content
        camp_options
        user_campaigns_favorite
        camp_metrika_counters
        metrika_counters
        campaigns_performance
        campaigns_internal
        warn_pay
        camp_payments_info
        camp_promocodes
        bs_export_specials
        campaigns
        campaigns_deals
        campaign_permalinks
        campaign_phones
        camp_additional_data
        camp_secondary_options
    /;
    for my $table (@delete_by_cid) {
        do_delete_from_table(PPC(shard => $shard), $table, where => { cid => $cid });
    }

    # Удаляем метки;
    my $tag_ids = get_one_column_sql(PPC(shard => $shard), 'SELECT tag_id FROM tag_campaign_list WHERE cid = ?', $cid) || [];

    if (scalar(@{$tag_ids})) {
        do_delete_from_table(PPC(shard => $shard), 'tag_campaign_list', where => { cid => $cid });
        do_delete_from_table(PPC(shard => $shard), 'tag_group', where => { tag_id => $tag_ids });
        delete_shard(tag_id => $tag_ids);
    }

    # Удаляем результаты модерации
    do_delete_from_table(PPC(shard => $shard), 'mod_reasons', where => {clientid => $ClientID, cid => $cid});

    if (camp_kind_in(type => $camp->{mediaType}, "web_edit_base")) {
        my $pids = get_one_column_sql(PPC(shard => $shard), "SELECT pid FROM phrases WHERE cid = ?", $cid);
        my $bids = get_one_column_sql(PPC(shard => $shard), "SELECT bid FROM banners WHERE cid = ?", $cid);
        my $mbids = get_one_column_sql(PPC(shard => $shard), "SELECT mbid FROM mediaplan_banners WHERE cid = ?", $cid);

        my $now = Yandex::DateTime->now();
        do_mass_insert_sql(PPC(shard => $shard), "REPLACE INTO deleted_banners (bid, deleteTime) VALUES %s",
                           [map {[$_, $now]} @$bids]);

        my %ids_type = (
            bid => $bids,
            pid => $pids,
            cid => $cid,
            mbid => $mbids
        );
        for my $pair (pairs DELETE_RULES) {
            my $rule = $pair->value;
            next unless @$rule;

            my $key = $pair->key;
            my ($tables, $table_key, %where);
            if (ref $rule->[0] eq 'ARRAY') {
                $table_key = $rule->[1];
                $tables = $rule->[0];
                %where = %{$rule->[2] || {}};
            } else {
                $table_key = $key;
                $tables = $rule;
            }

            for my $table (@$tables) {
                do_delete_from_table(PPC(shard => $shard), $table, where => {$table_key => $ids_type{$key}, %where});
            }
        }

        delete_camp_hierarchical_multipliers_from_specific_shard($shard, $cid);

        delete_shard(bid => $bids);
        delete_shard(pid => $pids);
        delete_shard(mediaplan_bid => $mbids) if @$mbids;
    } elsif(camp_kind_in(type => $camp->{mediaType}, "media")) {
        do_sql(PPC(shard => $shard), qq!
                DELETE mcb_phrases
                  FROM mcb_phrases, media_groups
                 WHERE mcb_phrases.mgid = media_groups.mgid
                   AND media_groups.cid = ?
            !, $cid);

        do_sql(PPC(shard => $shard), qq!
                DELETE media_banners, media_groups
                  FROM media_banners, media_groups
                 WHERE media_banners.mgid = media_groups.mgid
                   AND media_groups.cid = ?
            !, $cid);
    }

    # удаляем визитки вместе с адресами.
    VCards::delete_vcard_from_db($vcards_to_delete, skip_org_details => 1);

    # спорный момент - орг-детали вместе с визитками подчистить может не получится, т.к. там uid определяется из визитки (может не совпасть)
    my $empty_orgs = get_one_column_sql(PPC(shard => $shard), ['
            SELECT od.org_details_id
              FROM org_details od
                   LEFT JOIN vcards vc ON vc.uid = od.uid
                                          AND vc.org_details_id = od.org_details_id
        ', where => {
            'od.uid' => $uid,
            'vc.org_details_id__is_null' => 1,
        }
    ]) || [];
    if (@$empty_orgs) {
        do_delete_from_table(PPC(shard => $shard), 'org_details', where => {org_details_id => $empty_orgs});
        delete_shard(org_details_id => $empty_orgs);
    }

    delete_shard(cid => $cid);

    return 1;
}

sub delete_camp_hierarchical_multipliers_from_specific_shard {
    my ($shard, $cid) = @_;
    do_sql(PPC(shard => $shard), [
        "delete h, r from hierarchical_multipliers h join ab_segment_multiplier_values r using(hierarchical_multiplier_id)",
        where => { 'h.cid' => $cid, 'h.type' => 'ab_segment_multiplier' },
    ]);

    do_sql(PPC(shard => $shard), [
        "delete h, r from hierarchical_multipliers h join demography_multiplier_values r using(hierarchical_multiplier_id)",
        where => { 'h.cid' => $cid, 'h.type' => 'demography_multiplier' },
    ]);

    do_delete_from_table(PPC(shard => $shard), 'hierarchical_multipliers', where => { cid => $cid, type => 'desktop_multiplier'});

    do_sql(PPC(shard => $shard), [
        "delete h, r from hierarchical_multipliers h left join expression_multiplier_values r using(hierarchical_multiplier_id)",
        where => { 'h.cid' => $cid, 'h.type' => 'express_traffic_multiplier' },
    ]);

    do_sql(PPC(shard => $shard), [
        'DELETE h, g FROM hierarchical_multipliers h JOIN geo_multiplier_values g ON (g.hierarchical_multiplier_id = h.hierarchical_multiplier_id )',
        WHERE => { 'h.cid' => $cid, 'h.type' => 'geo_multiplier' },
    ]);

    do_sql(PPC(shard => $shard), [
        "delete h, r from hierarchical_multipliers h left join mobile_multiplier_values r using(hierarchical_multiplier_id)",
        where => { 'h.cid' => $cid, 'h.type' => 'mobile_multiplier' },
    ]);

    do_delete_from_table(PPC(shard => $shard), 'hierarchical_multipliers', where => { cid => $cid, type => 'performance_tgo_multiplier'});

    do_sql(PPC(shard => $shard), [
        "delete h, r from hierarchical_multipliers h join retargeting_multiplier_values r using(hierarchical_multiplier_id)",
        where => { 'h.cid' => $cid, 'h.type' => 'retargeting_multiplier' },
    ]);

    do_delete_from_table(PPC(shard => $shard), 'hierarchical_multipliers', where => { cid => $cid, type => 'video_multiplier'});
}
