#!/usr/bin/perl

=head1 NAME

    delete_invalid_retargeting_conditions.pl

=head1 SYNOPSIS

    /var/www/ppc.yandex.ru/protected/one-shot/delete_invalid_retargeting_conditions.pl --dry-run
    cat /var/www/ppc.yandex.ru/protected/logs/delete_invalid_retargeting_conditions.log | less
    /var/www/ppc.yandex.ru/protected/one-shot/delete_invalid_retargeting_conditions.pl

=head1 DESCRIPTION

    Скрипт находит в базе невалидные условия (содержащие цели без номера) по таблице
    retargeting_conditions, помечает их как удаленные (is_deleted = 1) и удаляет данные
    об этих целях из всех соответствующих таблиц: bids_retargeting, retargeting_goals,
    mediaplan_bids_retargeting. Перед модификаицей/удалением сохраняет все данные в лог.

    Параметры командной строки:
        --dry-run  - ничего не изменять/удалять (только запись в лог)

=cut

use warnings;
use strict;
use utf8;

use List::MoreUtils qw/uniq/;

use Yandex::DBTools;

use my_inc '../..';

use Settings;
use ScriptHelper;

use ShardingTools;

my ($DRY_RUN);
extract_script_params('dry-run!' => \$DRY_RUN);

my $res = 0;

$log->out('START');

for my $shard (ppc_shards()) {
    $log->msg_prefix(sprintf('[%sshard: %d]', ($DRY_RUN ? 'dry run, ' : ''), $shard));

    $log->out('fetch invalid conditions from db');
    my $invalid_conditions = get_all_sql(PPC(shard => $shard),'SELECT * FROM retargeting_conditions WHERE condition_json LIKE "%\"goal_id\":null%"');
    $log->out({table => 'retargeting_conditions', backup_data => $invalid_conditions});
    next unless @$invalid_conditions;

    # retargeting_conditions
    my @ret_cond_ids = map { $_->{ret_cond_id} } @$invalid_conditions;
    $log->out('mark conditions as deleted with ret_cond_id: ' . join(',', @ret_cond_ids));
    unless ($DRY_RUN) {
        $res = do_update_table(PPC(shard => $shard),
                               'retargeting_conditions',
                               { is_deleted => 1 },
                               where => {ret_cond_id => \@ret_cond_ids, is_deleted => 0},
                               );
    }
    $log->out("retargeting_conditions updated, affected $res rows");

    # bids_retargeting
    $log->out('fetch data from bids_retargeting');
    my $bids_retargeting = get_all_sql(PPC(shard => $shard), [
                                            'SELECT * FROM bids_retargeting',
                                            where => {ret_cond_id => \@ret_cond_ids},
                                       ]);
    $log->out({table => 'bids_retargeting', backup_data => $bids_retargeting});
    if (@$bids_retargeting) {
        my @ids = map { $_->{ret_id} } @$bids_retargeting;
        $log->out('delete data from bids_retargeting with ret_id: ' . join(',', @ids));
        unless ($DRY_RUN) {
            $res = do_delete_from_table(PPC(shard => $shard),
                                        'bids_retargeting',
                                        where => { ret_id => \@ids },
                                        );
        }
        $log->out("deleted $res rows from bids_retargeting");
        
        my @pids = uniq(map { $_->{pid} } @$bids_retargeting);
        $log->out('mark phrases as BS-unsynced with pid: ' . join(',', @pids));
        unless ($DRY_RUN) {
            $res = do_update_table(PPC(shard => $shard),
                                   'phrases',
                                   {statusBsSynced => 'No'},
                                   where => {pid => \@pids},
                                   );
        }
        $log->out("phrases updated, affected $res rows");
    }

    # retargeting_goals
    $log->out('fetch data from retargeting_goals');
    my $retargeting_goals = get_all_sql(PPC(shard => $shard), [
                                            'SELECT * from retargeting_goals',
                                            where => { ret_cond_id => \@ret_cond_ids },
                                        ]);
    $log->out({table => 'retargeting_goals', backup_data => $retargeting_goals});
    if (@$retargeting_goals) {
        my @ids = uniq(map { $_->{ret_cond_id} } @$retargeting_goals);
        $log->out('delete data from retargeting_goals with ret_cond_id: ' . join(',', @ids));
        unless ($DRY_RUN) {
            $res = do_delete_from_table(PPC(shard => $shard),
                                        'retargeting_goals',
                                        where => { ret_cond_id => \@ids },
                                        );
        }
        $log->out("deleted $res rows from retargeting_goals");
    }

    # mediaplan_bids_retargeting
    $log->out('fetch data from mediaplan_bids_retargeting');
    my $mediaplan_bids_retargeting = get_all_sql(PPC(shard => $shard), [
                                                    'SELECT * FROM mediaplan_bids_retargeting',
                                                    where => { ret_cond_id => \@ret_cond_ids },
                                                 ]);
    $log->out({table => 'mediaplan_bids_retargeting', backup_data => $mediaplan_bids_retargeting});
    if (@$mediaplan_bids_retargeting) {
        my @ids = uniq(map { $_->{ret_cond_id} } @$mediaplan_bids_retargeting);
        $log->out('delete data from mediaplan_bids_retargeting with ret_cond_id: ' . join(',', @ids));
        unless ($DRY_RUN) {
            $res = do_delete_from_table(PPC(shard => $shard),
                                        'mediaplan_bids_retargeting',
                                        where => { ret_cond_id => \@ret_cond_ids },
                                        );
        }
        $log->out("deleted $res rows from mediaplan_bids_retargeting");
    }
}

$log->out('FINISH');
