#!/usr/bin/perl

=head2 NAME

    fix-banner_images-status-moderate.pl

=head2 SYNOPSYS

    /var/www/ppc.yandex.ru/protected/one-shot/fix-banner_images-status-moderate.pl

=head2 DESCRIPTION

    Исправляем значение поля banner_images.statusModerate на значение статуса, по данным Модерации для заданных картинок
    (в отдельном файле указаны bid, image_id и image_hash) при условии, что картинка не удалена (banner_images.statusShow = "Yes"),
    текущий статус модерации у нее New и баннер промодерирован

=cut

use Direct::Modern;

use my_inc '../..';

use Settings;
use ScriptHelper;

use Yandex::DBTools;
use Yandex::DBShards qw/foreach_shard_parallel/;
use Yandex::ListUtils qw/chunks/;
use Yandex::Retry qw/relaxed/;

use Path::Tiny;

my $CHUNK_SIZE = 10000;
my $CONDITIONS_CHUNK_SIZE = 1000;

extract_script_params(
    'i=s' => \my $input_file,
);

die "No input file; stop"  if !$input_file;

$log->out('START');

my %image_data_by_status_moderate;
my $fh = path($input_file)->filehandle;
while (my $line = $fh->getline()) {
    chomp $line;
    my ($bid, $image_id, $image_hash, $status_moderate) = split /\t/, $line; # image_id не используем
    $image_data_by_status_moderate{$status_moderate}{$bid} = $image_hash;
}

foreach my $status_moderate (keys %image_data_by_status_moderate) {
    foreach my $bids (chunks([keys $image_data_by_status_moderate{$status_moderate}], $CHUNK_SIZE)) {
        process_chunk($bids, $image_data_by_status_moderate{$status_moderate}, $status_moderate);
    }
}

$log->out('FINISH');

sub process_chunk {
    my ($bids, $image_hash_by_bid, $status_moderate) = @_;

    foreach_shard_parallel bid => $bids, chunk_size => $CONDITIONS_CHUNK_SIZE, with_undef_shard => 1, sub {
        my ($shard, $bids_chunk) = @_;

        if (!$shard) {
            $log->out(($shard // 'undefined') .' shard for bids: '. join(', ', @$bids_chunk));
            return;
        }
        my $prefix_guard = $log->msg_prefix_guard("[shard $shard]");

        my $chunk_conditions = prepare_chunk_conditions($image_hash_by_bid, $bids_chunk);

        relaxed times => 1, sub {
            my $found_bids = get_one_column_sql(PPC(shard => $shard), [
                "SELECT bi.bid FROM banner_images bi FORCE INDEX (bid)
                JOIN banners b FORCE INDEX (PRIMARY) ON b.bid = bi.bid",
                WHERE => {
                    'bi.statusModerate' => 'New',
                    'bi.statusShow' => 'Yes',
                    'b.statusModerate' => 'Yes',
                    _OR => $chunk_conditions
                }
            ]);

            $log->out('found '.(scalar @$found_bids).' of '.(scalar @$bids_chunk).' images');

            return if !@$found_bids;

            $log->out({status_moderate => $status_moderate, found_bids => $found_bids});

            $chunk_conditions = prepare_chunk_conditions($image_hash_by_bid, $found_bids);
            $log->out('changing statusModerate New to '.$status_moderate);
            my $updated_rows = do_sql( PPC(shard => $shard), [
                'UPDATE banner_images bi FORCE INDEX (bid) JOIN banners b FORCE INDEX (PRIMARY) ON b.bid = bi.bid SET bi.statusModerate=?',
                WHERE => {
                    'bi.statusModerate' => 'New',
                    'bi.statusShow' => 'Yes',
                    'b.statusModerate' => 'Yes',
                    _OR => $chunk_conditions
                }
            ], $status_moderate );
            $log->out( { changed => scalar @$found_bids == $updated_rows + 0 ? 'all' : 'not all', found_ids => scalar @$found_bids, updated_ids => $updated_rows + 0 } );
        }
    }
}

sub prepare_chunk_conditions {
    my ($image_hash_by_bid, $bids_chunk) = @_;

    my %bids_by_image_hash;
    foreach my $bid (@$bids_chunk) {
        push @{$bids_by_image_hash{$image_hash_by_bid->{$bid}}}, $bid;
    }

    my @conditions;
    foreach my $image_hash (keys %bids_by_image_hash) {
        push @conditions, _AND => { 'bi.image_hash' => $image_hash, 'bi.bid' => $bids_by_image_hash{$image_hash} };
    }

    return \@conditions;
}
