#!/usr/bin/perl

=head1 DESCRIPTION

    Исправляет phrases.bid в базе.
    Заменяет bid=NULL, несуществующие и баннеры из других кампаний на минимальный bid из группы.
    При отсутствии в группе баннеров прописывает bid=NULL.

=cut

use my_inc '../..';
use lib::abs '..';

use warnings;
use strict;
use utf8;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::ListUtils qw(chunks);

use Settings;
use ScriptHelper;
use ShardingTools;

$log->out('START');

foreach_shard shard => [ppc_shards()], sub {
    my ($shard) = @_;
    $log->msg_prefix("[shard $shard]");

    $log->out('Fetching data to fix');
    my $data = get_all_sql(PPC(shard => $shard), ['
        SELECT p.pid, p.bid AS wrong_bid, MIN(br.bid) AS right_bid
        FROM phrases p
        LEFT JOIN banners bw ON p.bid = bw.bid
        LEFT JOIN banners br ON p.pid = br.pid
        WHERE
               (p.bid IS NOT NULL AND bw.bid IS NOT NULL AND p.cid != bw.cid) -- ссылается на баннер из чужой кампании
            OR (p.bid IS NOT NULL AND bw.bid IS NULL) -- ссылается на несуществующий баннер
            OR (p.bid IS NULL AND br.bid IS NOT NULL) -- ссылки нет, а баннеры в группе есть
        GROUP BY p.pid
    ']) || [];
    $log->out('Got ' . scalar(@$data) . ' groups to fix');

    for my $data_chunk (chunks $data, 500) {
        $log->out('Processing chunk:', $data_chunk);
        my $pids_chunk = [ map { $_->{pid} } @$data_chunk ];
        my $pid2wrongbid_chunk = { map { $_->{pid} => $_->{wrong_bid} } @$data_chunk };
        my $pid2rightbid_chunk = { map { $_->{pid} => $_->{right_bid} } @$data_chunk };

        # для групп без баннеров записываем phrases.bid = NULL
        my $new_bid_sql_case = sql_case('pid', $pid2rightbid_chunk, default => undef);
        my $old_bid_sql_case = sql_case('pid', $pid2wrongbid_chunk);
        my $affected_cnt = do_update_table(PPC(shard => $shard), 'phrases',
            {bid__dont_quote => $new_bid_sql_case, LastChange__dont_quote => 'LastChange'},
            where => {pid => $pids_chunk, bid__dont_quote => $old_bid_sql_case});
        $log->out("Updated $affected_cnt rows");
    }
};
$log->msg_prefix('');

$log->out('FINISH');
