#!/usr/bin/perl
use Direct::Modern;

=pod

=encoding utf8

    $Id:$

=head1 NAME

fix-sparse-data-in-sitelink-sets.pl

=head1 DESCRIPTION

Оказывается, в таблице sitelinks_set_to_link есть записи с sl_id = 0. Почему
они там появились, непонятно, в любом случае этот скрипт идёт по базе и их чинит.
Как чинит: такие записи удаляет, остальным записям расставляет order_num
по порядку от 0 до какого-то N без пропусков. Запускать этот скрипт надо
вручную в релизной миграции, например.

=head1 SYNOPSIS

    # посмотреть, что поломано и как бы починить:
    protected/one-shot/fix-sparse-data-in-sitelink-sets.pl --dry

    # починить:
    protected/one-shot/fix-sparse-data-in-sitelink-sets.pl

=cut

use Yandex::DBTools;
use Yandex::ListUtils;

use my_inc '../..', for => 'protected';
use ScriptHelper;
use Settings;
use ShardingTools;

my ($DRY_RUN);
extract_script_params(
    'dry-run' => \$DRY_RUN,
);

$DRY_RUN //= 0;

$log->out('start');
$log->out("dry_run=$DRY_RUN");

for my $shard ( ppc_shards() ) {
    my $shard_msg_guard = $log->msg_prefix_guard("[shard=$shard]");
    $log->out('start');

    my $sl_set_ids = get_one_column_sql( PPC( shard => $shard ), 'select distinct sitelinks_set_id from sitelinks_set_to_link where sl_id = 0' ) // [];
    for my $sl_set_id (@$sl_set_ids) {
        my $sl_set_msg_guard = $log->msg_prefix_guard("[shard=$shard, sitelinks_set_id=$sl_set_id]");

        do_in_transaction {
            my $sl_set_data = get_all_sql( PPC( shard => $shard ),
                'select order_num, sl_id from sitelinks_set_to_link where sitelinks_set_id = ? order by order_num for update',
                $sl_set_id );

            for my $row_before (@$sl_set_data) {
                $log->out("row before: order_num=$row_before->{order_num}, sl_id=$row_before->{sl_id}");
            }

            my @order_nums_to_delete;

            # order_num в данных до миграции => какой order_num поставить
            my %order_num_map;

            # order_num начинаются с нуля
            my $next_new_order_num = 0;
            for my $row_before (@$sl_set_data) {
                if ( $row_before->{sl_id} == 0 ) {
                    push @order_nums_to_delete, $row_before->{order_num};
                    $log->out("deleting this row: order_num=$row_before->{order_num}, sl_id=$row_before->{sl_id}");
                    next;
                }

                my $new_order_num = $next_new_order_num++;

                if ( $row_before->{order_num} != $new_order_num ) {
                    $log->out( "changing order_num for this row: order_num=$row_before->{order_num}, " .
                        "sl_id=$row_before->{sl_id}, new order_num=$new_order_num" );

                    $order_num_map{ $row_before->{order_num} } = $new_order_num;
                } else {
                    $log->out("keeping this row intact: order_num=$row_before->{order_num}, sl_id=$row_before->{sl_id}");
                }
            }

            if ($DRY_RUN) {
                $log->out("not doing any changes because it's a dry run");
                return;
            }

            for my $order_num (@order_nums_to_delete) {
                $log->out("actually deleting order_num=$order_num");
                my $deleted = int do_delete_from_table( PPC( shard => $shard ),
                    'sitelinks_set_to_link',
                    where => {
                        sitelinks_set_id => $sl_set_id,
                        order_num => $order_num,
                    } );

                $log->out("... deleted $deleted");
            }

            for my $old_order_num ( nsort keys %order_num_map ) {
                my $new_order_num = $order_num_map{$old_order_num};
                $log->out("actually changing order_num=$old_order_num to order_num=$new_order_num");

                my $updated = int do_update_table( PPC(shard => $shard ),
                    'sitelinks_set_to_link',
                    { order_num => $new_order_num },
                    where => {
                        sitelinks_set_id => $sl_set_id,
                        order_num => $old_order_num,
                    } );

                $log->out("... updated $updated");
            }
        };
    }

    $log->out('finish');
}

$log->out('finish');
