#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use my_inc "../..";

use ScriptHelper;
use Settings;
use ShardingTools;
use Currencies;

use Yandex::DBTools;

use List::MoreUtils qw(uniq);

my %table2key = (
    bids => {key => 'id'},
    bids_arc => {key => 'cid'},
    bids_manual_prices => {key => 'cid', no_statusBsSynced => 1, no_strategy_check => 1},
);

my $CHUNK_SIZE = 100;

$log->out('START');

my @currencies = keys %Currencies::_CURRENCY_DESCRIPTION;
my %currency2minprice = map { $_ => get_currency_constant($_, 'MIN_PRICE') } @currencies;
my $minprice_case = sql_case('CURRENCY_PLACEHOLDER', \%currency2minprice, default => 0);
$minprice_case =~ s/`CURRENCY_PLACEHOLDER`/IFNULL(c.currency, 'YND_FIXED')/;

my $total_rows_cnt = 0;
for my $shard (ppc_shards()) {
    while (my ($table, $table_desc) = each %table2key) {
        my $key_field = $table_desc->{key};
        $log->msg_prefix("shard=$shard,table=$table");
        my $last_key = 0;
        my $fetched_rows_cnt;
        my @fromjoin_sql = ("
            $table bi
            INNER JOIN campaigns c ON bi.cid = c.cid
        ");
        my (@where, @set);
        if ($table_desc->{no_strategy_check}) {
            # раз не проверяем стратегию, то фиксим только явные нули в обоих полях (могло быть отдельное размещение до включения автобюджета)
            @where = ({
                'bi.price__lt__dont_quote' => $minprice_case,
                'bi.price_context__lt__dont_quote' => $minprice_case,
            });
            @set = (
                  "bi.price = $minprice_case",
                ", bi.price_context = $minprice_case",
            );
        } else {
            push @fromjoin_sql, "LEFT JOIN camp_options co ON bi.cid = co.cid";
            my $wrong_price_cond = "(bi.price < $minprice_case AND NOT (IFNULL(co.strategy, '') = 'different_places' AND platform = 'context')";
            my $wrong_price_context_cond = "(bi.price_context < $minprice_case AND IFNULL(co.strategy, '') = 'different_places')";
            @where = (
                {
                    'c.autobudget' => 'No',
                },
                "AND ($wrong_price_cond OR $wrong_price_context_cond)",
            );
            @set = (
                  "bi.price = IF($wrong_price_cond, $minprice_case, bi.price)",
                ", bi.price_context = IF($wrong_price_context_cond, $minprice_case, bi.price_context)",
            );
        }
        if (!$table_desc->{no_statusBsSynced}) {
            # целиком на условие сбрасываем, т.к. в БК может не быть такой фразы, если ранее получили ошибку при отправке с кривыми ценами
            push @fromjoin_sql, "LEFT JOIN phrases p ON bi.pid = p.pid";
            push @set, ",", { 'p.statusBsSynced' => 'No' };
        }
        do {
            $log->out("Fetching bad rows beginning from $key_field >= $last_key");
            my $rows = get_all_sql(PPC(shard => $shard), ["
                SELECT bi.*
                FROM", @fromjoin_sql, "
                WHERE", @where, 'AND', {"bi.${key_field}__ge" => $last_key}, "
                ORDER BY $key_field
                LIMIT $CHUNK_SIZE
            "]);
            $fetched_rows_cnt = scalar(@$rows);
            if ($fetched_rows_cnt > 0) {
                $last_key = $rows->[-1]->{$key_field};
                $log->out("Fetched $fetched_rows_cnt rows, last $key_field is $last_key:", @$rows);

                my @keys_to_fix = uniq map { $_->{$key_field} } @$rows;
                $log->out("Fixing ${key_field}'s:", \@keys_to_fix);
                my $updated_cnt = do_sql(PPC(shard => $shard), ["
                    UPDATE", @fromjoin_sql, "
                    SET", @set,
                    "WHERE", @where, 'AND', {"bi.${key_field}" => \@keys_to_fix},
                ]);
                $log->out("Updated $updated_cnt rows");
                $total_rows_cnt += $updated_cnt // 0;
            } else {
                $log->out("Fetched $fetched_rows_cnt rows");
            }
        } until ($fetched_rows_cnt < $CHUNK_SIZE);
        $log->msg_prefix('');
    }
}

$log->out("Total fixed $total_rows_cnt rows");
$log->out('FINISH');
