#!/usr/bin/perl

use my_inc '../..';

=head1 DESCRIPTION

Исправляем geo по списку.

    protected/one-shot/fix_geo_by_list.pl --data filename

filename - tsv-файл с полями
type    id      geo_from    geo_to

type - один из cid, pid, mbid, mgid

=cut

use Direct::Modern;

use Settings;
use ScriptHelper;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Retry;
use BS::ResyncQueue;

my $SLEEP_COEF = 1;
my $READ_CHUNK_SIZE = 10000;
my $PROCESS_CHUNK_SIZE = 1000;

extract_script_params(
    'sleep-coef=f' => \$SLEEP_COEF,
    'data=s' => \my $file,
);




$log->out('START');

open my $in, '<', $file  or $log->die("Cannot open $file, STOP");

my %buf;
while (my $line = readline $in) {
    chomp $line;
    my ($type, $id, $geo_from, $geo_to) = split /\t/ => $line;
    my $cur_buf = $buf{$type} ||= [];
    push @$cur_buf, { $type => $id, geo_from => $geo_from, geo_to => $geo_to };

    if (@$cur_buf >= $READ_CHUNK_SIZE) {
        _process_big_chunk($type => $cur_buf);
        delete $buf{$type};
    }
}
_process_big_chunk($_ => $buf{$_})  for keys %buf;

close $in;

$log->out('FINISH');


sub _process_big_chunk {
    my ($type, $big_chunk) = @_;

    $log->out(sprintf "processing $type chunk: %d items", scalar @$big_chunk);

    my $sharded_by = $type;
    my %chunk_param = (chunk_size => $PROCESS_CHUNK_SIZE);
    if ($type eq 'mgid') {
        my $cid_by_mgid = get_hash_sql(PPC(shard => 'all'), [
                'SELECT mgid, cid FROM media_groups',
                WHERE => {mgid => [map { $_->{$type} } @$big_chunk]},
            ]);
        $_->{cid} = $cid_by_mgid->{$_->{mgid}} || 0  for @$big_chunk;
        $sharded_by = 'cid';            
    }
    elsif ($type eq 'mbid') {
        $sharded_by = 'mediaplan_bid';
        $chunk_param{by} = 'mbid';
    }

    for my $chunk (sharded_chunks($sharded_by => $big_chunk, %chunk_param)) {
        _process_chunk($chunk->{shard}, $type => $chunk->{$sharded_by});
    }
    return;
}



sub _process_chunk {
    my ($shard, $type, $items) = @_;

    my $relax = relaxed_guard(times => $SLEEP_COEF);

    my $current = _get_current_items($shard, $type, $items);

    my @items_to_fix = grep {$_->{geo_from} eq ($current->{$_->{$type}}->{geo} || '')} @$items;
    return if !@items_to_fix;

    for my $item (@items_to_fix) {
        $log->out("updating geo for $type $item->{$type} from $item->{geo_from} to $item->{geo_to}");
    }
    $log->out(sprintf "writing %d values in shard #$shard", scalar @items_to_fix);

    my $affected = _update_values($shard, $type, \@items_to_fix, $current);
    $log->out("$affected $type values updated");

    return;
}


sub _get_current_items {
    my ($shard, $type, $items) = @_;

    my @item_ids = map { $_->{$type} } @$items;
    if ($type eq 'pid') {
        return get_hashes_hash_sql(PPC(shard => $shard), [
                'SELECT pid, cid, geo  FROM phrases',
                WHERE => { pid => \@item_ids },
            ]);
    }
    elsif ($type eq 'cid') {
        return get_hashes_hash_sql(PPC(shard => $shard), [
                'SELECT cid, geo  FROM campaigns',
                WHERE => { cid => \@item_ids },
            ]);
    }
    elsif ($type eq 'mbid') {
        return get_hashes_hash_sql(PPC(shard => $shard), [
                'SELECT mbid, geo FROM mediaplan_banners',
                WHERE => { mbid => \@item_ids },
            ]);
    }
    elsif ($type eq 'mgid') {
        return get_hashes_hash_sql(PPC(shard => $shard), [
                'SELECT mgid, geo FROM media_groups',
                WHERE => { mgid => \@item_ids },
            ]);
    }
    else {
        croak "Unsupported type $type";
    }
}


sub _update_values {
    my ($shard, $type, $items, $current) = @_;

    my @item_ids = map { $_->{$type} } @$items;
    my $geo_from_sql = sql_case($type => {map { $_->{$type} => $_->{geo_from} } @$items});
    my $geo_to_sql = sql_case($type => {map { $_->{$type} => $_->{geo_to} } @$items});

    my $affected;
    if ($type eq 'pid') {
        $affected = do_update_table(PPC(shard => $shard), 'phrases',
            {geo__dont_quote => $geo_to_sql},
            where => {
                pid => \@item_ids,
                geo__dont_quote => $geo_from_sql,
            },
        );

        my @resync = map {{cid => $current->{$_}->{cid}, pid => $_, priority => -30}} @item_ids;
        bs_resync(\@resync);
    }
    elsif ($type eq 'cid') {
        $affected = do_update_table(PPC(shard => $shard), 'campaigns',
            {geo__dont_quote => $geo_to_sql},
            where => {
                cid => \@item_ids,
                geo__dont_quote => $geo_from_sql,
            },
        );
    }
    elsif ($type eq 'mbid') {
        $affected = do_update_table(PPC(shard => $shard), 'mediaplan_banners',
            {geo__dont_quote => $geo_to_sql},
            where => {
                mbid => \@item_ids,
                geo__dont_quote => $geo_from_sql,
            },
        );
    }
    elsif ($type eq 'mgid') {
        $affected = do_update_table(PPC(shard => $shard), 'media_groups', {
                geo__dont_quote => $geo_to_sql,
                statusBsSynced => 'No',
            },
            where => {
                mgid => \@item_ids,
                geo__dont_quote => $geo_from_sql,
            },
        );
    }
    else {
        croak "Unsupported type $type";
    }

    return $affected;
}



