#!/usr/bin/env perl

use my_inc "../..";

=head1 DESCRIPTION

    Исправляем последствия выдачи дублирующихся айдишников базой PPCDICT за 2014-11-14

    LOG_TEE=1 ./protected/one-shot/fix_ids_14nov2014.pl

    Параметры:

        dry-run    -- запуск без внесения изменений в БД
        use-clh    -- использовать ClickHouse в качестве источника данных

=cut

use Direct::Modern;
use English qw/ -no_match_vars/;
use open ':std' => ':utf8';

use LWP::UserAgent;
use Yandex::HTTP;
use JSON;
use List::MoreUtils qw/all/;

use Settings;
use ScriptHelper;

use Yandex::DBTools;
use Yandex::DBShards;

use ShardingTools;

use RBAC2::Extended;
use Campaign qw//;
use PrimitivesIds qw//;

use Data::Dumper;

$LIST_SEPARATOR = ',';

my $USE_CLH;
my $CLH_URL = "http://ppchouse01f.yandex.ru:8123/";

my $DRY_RUN;
extract_script_params(
    'dry-run'       => \$DRY_RUN,
    'use-clh'       => \$USE_CLH,
);

$log->out('START');

sub clh_query {
    my $query = shift;

    my $ua = LWP::UserAgent->new();
    $ua->timeout(10*3600);
    my $resp = $ua->post(Yandex::HTTP::make_url($CLH_URL, {query => "$query format JSON"}));
    die $resp->content."\n".$resp->status_line unless $resp->is_success;
    return decode_json($resp->content);
}

# Заберем данные из clickhouse'а по выданным идентификаторам
my %key2dup_ids;
if ($USE_CLH) {
    my $clh_res = clh_query("SELECT key, ids FROM dbshards_ids WHERE /*key = 'cid' AND*/ log_date = toDate('2014-11-14')");
    for my $row (@{$clh_res->{data}}) {
        $key2dup_ids{ $row->{key} }->{$_}++ for @{$row->{ids}};
    }
    for my $key (keys %key2dup_ids) {
        delete $key2dup_ids{$key}->{$_} for grep { $key2dup_ids{$key}->{$_} == 1 } keys %{$key2dup_ids{$key}};
        delete $key2dup_ids{$key} unless keys %{$key2dup_ids{$key}};
    }
} else {
    %key2dup_ids = %{decode_json(join '', <DATA>)};
}

my $rbac;
eval { $rbac = RBAC2::Extended->get_singleton(1); 1; } or die "Cannot connect to rbac: $@";

my $get_shard_multi_orig = \&Yandex::DBShards::get_shard_multi;
my $delete_shard_orig = \&Yandex::DBShards::delete_shard;

my @REPAIR_SUBS = (
    #cid                   => \&fix_campaigns,
    pid                   => \&fix_adgroups,
    phid                  => \&fix_keywords,
    bid                   => \&fix_banners,
    #sitelinks_set_id      => \&fix_sitelinks_sets,
    #sl_id                 => \&fix_sitelinks,
    #mw_id                 => \&fix_minus_words,
    #org_details_id        => \&fix_org_details,
    #users_notification_id => \&fix_users_notifications,
    #opt_camp_postpone_id  => \&fix_opt_camp_postpone,
    #banner_images_pool_id => \&fix_banner_images_pool,
);

# Вызывать процедуры восстановления будем в заданном порядке
for (my $i = 0; $i < @REPAIR_SUBS; $i += 2) {
    my ($key, $sub) = @REPAIR_SUBS[$i, $i + 1];
    next unless exists $key2dup_ids{$key};
    $sub->([keys %{$key2dup_ids{$key}}]);
}

$log->out('FINISH');

sub fix_campaigns {
    my ($cids) = @_;

    $log->out("Running fix_campaigns()");
    $log->out("cids: @$cids");

    my $metabase_shard = get_shard_multi(cid => $cids);
    my $cids_data = {};
    for my $shard (ppc_shards()) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT cid, OrderID FROM campaigns}, WHERE => {cid => $cids}]);
        $cids_data->{$_->{cid}}->{$shard} = $_->{OrderID} for @$rows;
    }

    local *delete_campaign = sub {
        my ($cid, $force_shard) = @_;

        no warnings 'redefine';
        local *Yandex::DBShards::get_shard_multi = sub {
            my ($key, $vals, $chain_key) = @_;
            return {map { $_ => $force_shard } @$vals};
        };
        # Т.к. функция delete_shard уже импортирована в Campaign, то заменять ссылку на функцию нужно там
        # get_shard_multi, похоже, используется только внутри DBShards, внутри вызова get_shard; поэтому заменяем там
        local *Campaign::delete_shard = sub {
            my ($key, $vals, %OPT) = @_;
            return if $key eq 'cid' && $vals eq $cid;
            $delete_shard_orig->(@_);
        };

        my $client_uid = get_one_field_sql(PPC(cid => $cid), q{SELECT uid FROM campaigns WHERE cid = ?}, $cid);
        return Campaign::del_camp($rbac, $cid, $client_uid, force => 1, ignore_rbac_errors => 1);
    };

    for my $cid (@$cids) {
        my $m_shard = $metabase_shard->{$cid};
        my $cid_data = $cids_data->{$cid};
        my @cid_shards = keys %$cid_data;

        next if (!$m_shard && !@cid_shards) || ($m_shard && @cid_shards == 1 && $m_shard == $cid_shards[0]);

        if (!@cid_shards) {
            # Кампании нет ни в одном шарде. Удалим из метабазы.
            $log->out("[case0] Cannot find campaign#${cid} in any shard. Deleting from metabase.");
            delete_shard(cid => $cid) if !$DRY_RUN;
        } elsif (@cid_shards == 1) {
            # Кампания в одном шарде
            my $shard = $cid_shards[0];

            if (
                (($m_shard && $m_shard != $shard) || !$m_shard) && !$cid_data->{$shard}
            ) {
                # Кампания находится в шарде, отличным от метабазы (или не записана там). Не была в БК. Удалим.
                $log->out("[case1.1] Deleting campaign#${cid} in shard#${shard}.");
                delete_campaign($cid, $shard) if !$DRY_RUN;
                delete_shard(cid => $cid) if $m_shard && !$DRY_RUN;
            } elsif (
                (($m_shard && $m_shard != $shard) || !$m_shard) && $cid_data->{$shard}
            ) {
                # Кампания находится в шарде, отличным от метабазы (или не записана там). Была в БК. Перепропишем шард.
                $log->out("[case1.2] Setting shard#${shard} for campaign#${cid}");
                delete_shard(cid => $cid) if $m_shard && !$DRY_RUN;
                my $client_id = get_one_field_sql(PPC(shard => $shard), ["select u.ClientID from campaigns c join users u on u.uid = c.uid", where => {'c.cid' => $cid}]);
                save_shard(cid => $cid, ClientID => $client_id) if !$DRY_RUN;
            } else {
                $log->out(Dumper({m_shard => $m_shard, $cid => $cid_data}));
                $log->die("[case1.3] Unknown case");
            }
        } else {
            # Кампания в нескольких шардах
            my @other_shards = grep { !$m_shard || $m_shard != $_ } @cid_shards;

            if ($m_shard && exists $cid_data->{$m_shard} && all { !$cid_data->{$_} } @other_shards) {
                # Кампания есть в шарде метабазы. В других шардах не была в БК. Удалим из остальных шардов.
                for my $shard (@other_shards) {
                    $log->out("[case2.1] Deleting campaign#${cid} in shard#${shard}.");
                    delete_campaign($cid, $shard) if !$DRY_RUN;
                }
            } else {
                $log->out(Dumper({m_shard => $m_shard, $cid => $cid_data}));
                $log->die("[case2.2] Unknown case");
            }
        }
    }
}

sub fix_adgroups {
    my ($pids) = @_;

    $log->out("Running fix_adgroups()");
    $log->out("pids: @$pids");

    my $metabase_shard = get_shard_multi(pid => $pids);
    my $pids_data = {};
    for my $shard (ppc_shards()) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT pid, PriorityID FROM phrases}, WHERE => {pid => $pids}]);
        $pids_data->{$_->{pid}}->{$shard} = $_->{PriorityID} for @$rows;
    }

    local *delete_adgroup = sub {
        my ($pid, $force_shard) = @_;

        no warnings 'redefine';
        local *Yandex::DBShards::get_shard_multi = sub {
            my ($key, $vals, $chain_key) = @_;
            return {map { $_ => $force_shard } @$vals};
        };

        my $cid = PrimitivesIds::get_cid(pid => $pid);
        my $bids = PrimitivesIds::get_bids(pid => $pid);

        do_delete_from_table(PPC(shard => $force_shard), 'redirect_check_queue', where => {object_id => $bids, object_type => 'banner'});
        do_delete_from_table(PPC(shard => $force_shard), 'mod_reasons', where => {id => $bids, type => [qw/banner contactinfo sitelinks_set image/]});
        do_update_table(PPC(shard => $force_shard), 'mediaplan_banners', {source_bid => 0}, where => {source_bid => $bids, cid => $cid});
        for my $tbl (qw/banner_images banners_mobile_content banners_performance
                        catalogia_banners_rubrics banners_mod_doc_queue
                        banners/
        ) {
            do_delete_from_table(PPC(shard => $force_shard), $tbl, where => {bid => $bids});
        }

        # При удалении баннеров из метабазы пропустим те, которые прописаны в других шардах
        my $shard_by_bid = $get_shard_multi_orig->(bid => $bids);
        my @bids_to_delete = grep { ($shard_by_bid->{$_} // 0) == $force_shard } @$bids;
        delete_shard(bid => \@bids_to_delete) if @bids_to_delete;

        for my $tbl (qw/bids bids_retargeting tag_group group_params bids_dynamic 
                        dynamic_conditions bids_performance adgroups_mobile_content adgroups_performance adgroups_dynamic
                        phrases/
        ) {
            do_delete_from_table(PPC(shard => $force_shard), $tbl, where => {pid => $pid});
        }

        return;
    };

    for my $pid (@$pids) {
        my $m_shard = $metabase_shard->{$pid};
        my $pid_data = $pids_data->{$pid};
        my @pid_shards = keys %$pid_data;

        next if (!$m_shard && !@pid_shards) || ($m_shard && @pid_shards == 1 && $m_shard == $pid_shards[0]);

        if (!@pid_shards) {
            # Группы нет ни в одном шарде. Удалим из метабазы.
            $log->out("[case0] Cannot find adgroup#${pid} in any shard. Deleting from metabase.");
            delete_shard(pid => $pid) if !$DRY_RUN;
        } elsif (@pid_shards == 1) {
            # Группа в одном шарде
            my $shard = $pid_shards[0];

            if (
                (($m_shard && $m_shard != $shard) || !$m_shard) && !$pid_data->{$shard}
            ) {
                # Группа находится в шарде, отличным от метабазы (или не записана там). Не была в БК. Удалим.
                $log->out("[case1.1] Deleting adgroup#${pid} in shard#${shard}.");
                delete_adgroup($pid, $shard) if !$DRY_RUN;
                delete_shard(pid => $pid) if $m_shard && !$DRY_RUN;
            } elsif (
                (($m_shard && $m_shard != $shard) || !$m_shard) && $pid_data->{$shard}
            ) {
                # Группа находится в шарде, отличным от метабазы (или не записана там). Была в БК. Перепропишем шард.
                $log->out("[case1.2] Setting shard#${shard} for adgroup#${pid}");
                delete_shard(pid => $pid) if $m_shard && !$DRY_RUN;
                my $client_id = get_one_field_sql(PPC(shard => $shard), ["select u.ClientID from phrases p join campaigns c on c.cid = p.cid join users u on u.uid = c.uid", where => {'p.pid' => $pid}]);
                save_shard(pid => $pid, ClientID => $client_id) if !$DRY_RUN;
            } else {
                $log->out(Dumper({m_shard => $m_shard, $pid => $pid_data}));
                $log->die("[case1.3] Unknown case");
            }
        } else {
            # Группа в нескольких шардах
            my @other_shards = grep { !$m_shard || $m_shard != $_ } @pid_shards;

            if ($m_shard && exists $pid_data->{$m_shard} && all { !$pid_data->{$_} } @other_shards) {
                # Группа есть в шарде метабазы. В других шардах не была в БК. Удалим из остальных шардов.
                for my $shard (@other_shards) {
                    $log->out("[case2.1] Deleting adgroup#${pid} in shard#${shard}.");
                    delete_adgroup($pid, $shard) if !$DRY_RUN;
                }
            } else {
                $log->out(Dumper({m_shard => $m_shard, $pid => $pid_data}));
                $log->die("[case2.2] Unknown case");
            }
        }
    }
}

sub fix_keywords {
    my ($kwids) = @_;

    $log->out("Running fix_keywords()");
    $log->out("ids: @$kwids");

    my $kwids_data = {};
    for my $shard (ppc_shards()) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT id FROM bids}, WHERE => {id => $kwids}]);
        $kwids_data->{$_->{id}}->{$shard} = 1 for @$rows;
    }

    for my $kwid (@$kwids) {
        my $kwid_data = $kwids_data->{$kwid};
        my @kwid_shards = keys %$kwid_data;

        next unless @kwid_shards > 1;

        my $shard_where_update = $kwid_shards[1];
        my $new_kwid = !$DRY_RUN ? get_new_id('phid') : q{`get_new_id('phid')`};
        my ($cid, $pid) = get_one_line_array_sql(PPC(shard => $shard_where_update), q{SELECT c.cid, p.pid FROM campaigns c JOIN phrases p ON (c.cid = p.cid) JOIN bids kw ON (kw.pid = p.pid) WHERE kw.id = ?}, $kwid);

        $log->out("Changing keyword_id#${kwid} (cid=$cid;pid=$pid) in shard#${shard_where_update} to $new_kwid");

        # Заменяем id ключевого слова
        if (!$DRY_RUN) {
            do_update_table(PPC(shard => $shard_where_update), 'bids_href_params', {id => $new_kwid}, where => {id => $kwid, cid => $cid});
            do_update_table(PPC(shard => $shard_where_update), 'bids_manual_prices', {id => $new_kwid}, where => {id => $kwid, cid => $cid});
            do_update_table(PPC(shard => $shard_where_update), 'bids_phraseid_history', {id => $new_kwid}, where => {id => $kwid, cid => $cid});
            do_update_table(PPC(shard => $shard_where_update), 'bids', {id => $new_kwid}, where => {id => $kwid});
        }
    }
}

sub fix_banners {
    my ($bids) = @_;

    $log->out("Running fix_banners()");
    $log->out("bids: @$bids");

    my $metabase_shard = get_shard_multi(bid => $bids);
    my $bids_data = {};
    for my $shard (ppc_shards()) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT bid, BannerID FROM banners}, WHERE => {bid => $bids}]);
        $bids_data->{$_->{bid}}->{$shard} = $_->{BannerID} for @$rows;
    }

    local *delete_banner = sub {
        my ($bid, $force_shard) = @_;

        no warnings 'redefine';
        local *Yandex::DBShards::get_shard_multi = sub {
            my ($key, $vals, $chain_key) = @_;
            return {map { $_ => $force_shard } @$vals};
        };

        my $cid = PrimitivesIds::get_cid(bid => $bid);

        do_delete_from_table(PPC(shard => $force_shard), 'redirect_check_queue', where => {object_id => $bid, object_type => 'banner'});
        do_delete_from_table(PPC(shard => $force_shard), 'mod_reasons', where => {id => $bid, type => [qw/banner contactinfo sitelinks_set image/]});
        do_update_table(PPC(shard => $force_shard), 'mediaplan_banners', {source_bid => 0}, where => {source_bid => $bid, cid => $cid});
        for my $tbl (qw/banner_images banners_mobile_content banners_performance
                        catalogia_banners_rubrics banners_mod_doc_queue
                        banners/
        ) {
            do_delete_from_table(PPC(shard => $force_shard), $tbl, where => {bid => $bid});
        }


        return;
    };

    for my $bid (@$bids) {
        my $m_shard = $metabase_shard->{$bid};
        my $bid_data = $bids_data->{$bid};
        my @bid_shards = keys %$bid_data;

        next if (!$m_shard && !@bid_shards) || ($m_shard && @bid_shards == 1 && $m_shard == $bid_shards[0]);

        if (!@bid_shards) {
            # Баннера нет ни в одном шарде. Удалим из метабазы.
            $log->out("[case0] Cannot find banner#${bid} in any shard. Deleting from metabase.");
            delete_shard(bid => $bid) if !$DRY_RUN;
        } elsif (@bid_shards == 1) {
            # Баннер в одном шарде
            my $shard = $bid_shards[0];

            if (
                (($m_shard && $m_shard != $shard) || !$m_shard) && !$bid_data->{$shard}
            ) {
                # Баннер находится в шарде, отличным от метабазы (или не записан там). Не был в БК. Удалим.
                $log->out("[case1.1] Deleting banner#${bid} in shard#${shard}.");
                delete_banner($bid, $shard) if !$DRY_RUN;
                delete_shard(bid => $bid) if $m_shard && !$DRY_RUN;
            } elsif (
                (($m_shard && $m_shard != $shard) || !$m_shard) && $bid_data->{$shard}
            ) {
                # Баннер находится в шарде, отличным от метабазы (или не записан там). Был в БК. Перепропишем шард.
                $log->out("[case1.2] Setting shard#${shard} for banner#${bid}");
                delete_shard(bid => $bid) if $m_shard && !$DRY_RUN;
                my $client_id = get_one_field_sql(PPC(shard => $shard), ["select u.ClientID from banners b join campaigns c on c.cid = b.cid join users u on u.uid = c.uid", where => {'b.bid' => $bid}]);
                save_shard(bid => $bid, ClientID => $client_id) if !$DRY_RUN;
            } else {
                $log->out(Dumper({m_shard => $m_shard, $bid => $bid_data}));
                $log->die("[case1.3] Unknown case");
            }
        } else {
            # Баннер в нескольких шардах
            my @other_shards = grep { !$m_shard || $m_shard != $_ } @bid_shards;

            if ($m_shard && exists $bid_data->{$m_shard} && all { !$bid_data->{$_} } @other_shards) {
                # Баннер есть в шарде метабазы. В других шардах не был в БК. Удалим из остальных шардов.
                for my $shard (@other_shards) {
                    $log->out("[case2.1] Deleting banner#${bid} in shard#${shard}.");
                    delete_banner($bid, $shard) if !$DRY_RUN;
                }
            } else {
                $log->out(Dumper({m_shard => $m_shard, $bid => $bid_data}));
                $log->die("[case2.2] Unknown case");
            }
        }
    }
}

sub fix_sitelinks_sets {
    my ($ssids) = @_;

    $log->out("Running fix_sitelinks_sets()");
    $log->out("sitelinks_set_ids: @$ssids");

    my $metabase_shard = get_shard_multi(sitelinks_set_id => $ssids);
    my $ssids_data = {};
    for my $shard (ppc_shards()) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT sitelinks_set_id, ClientID FROM sitelinks_sets}, WHERE => {sitelinks_set_id => $ssids}]);
        $ssids_data->{$_->{sitelinks_set_id}}->{$shard} = $_->{ClientID} for @$rows;
    }

    for my $ssid (@$ssids) {
        my $m_shard = $metabase_shard->{$ssid};
        my $ssid_data = $ssids_data->{$ssid};
        my @ssid_shards = keys %$ssid_data;

        next if (!$m_shard && !@ssid_shards) || ($m_shard && @ssid_shards == 1 && $m_shard == $ssid_shards[0]);

        if (!@ssid_shards) {
            # Сайтлинк-сета нет ни в одном шарде. Удалим из метабазы.
            $log->out("[case0] Cannot find sitelinks_set_id#${ssid} in any shard. Deleting from metabase.");
            delete_shard(sitelinks_set_id => $ssid) if !$DRY_RUN;
        } elsif (@ssid_shards == 1) {
            # Сайтлинк-сет находится в шарде, отличным от метабазы (или не записан там). Перепропишем шард.
            my $shard = $ssid_shards[0];
            $log->out("[case1] Setting shard#${shard} for sitelinks_set_id#${ssid}");
            delete_shard(sitelinks_set_id => $ssid) if $m_shard && !$DRY_RUN;
            save_shard(sitelinks_set_id => $ssid, ClientID => $ssid_data->{$shard}) if !$DRY_RUN;
        } else {
            # Сайтлинк-сет в нескольких шардах
            my @other_shards = grep { !$m_shard || $m_shard != $_ } @ssid_shards;

            if (!$m_shard || !exists $ssid_data->{$m_shard}) {
                my $shard = shift @other_shards;
                $log->out("[case2.1] Setting shard#${shard} for sitelinks_set_id#${ssid}");
                delete_shard(sitelinks_set_id => $ssid) if $m_shard && !$DRY_RUN;
                save_shard(sitelinks_set_id => $ssid, ClientID => $ssid_data->{$shard}) if !$DRY_RUN;
            }

            for my $shard (@other_shards) {
                my $new_ssid = !$DRY_RUN ? get_new_id('sitelinks_set_id', ClientID => $ssid_data->{$shard}) : q{`get_new_id('sitelinks_set_id', ClientID => ...)`};
                $log->out("[case2.2] Changing sitelinks_set_id#${ssid} in shard#${shard} to $new_ssid");

                # Заменяем id сайтлинк-сета
                if (!$DRY_RUN) {
                    do_update_table(PPC(shard => $shard), 'banners', {sitelinks_set_id => $new_ssid}, where => {sitelinks_set_id => $ssid});
                    do_update_table(PPC(shard => $shard), 'sitelinks_set_to_link', {sitelinks_set_id => $new_ssid}, where => {sitelinks_set_id => $ssid});
                    do_update_table(PPC(shard => $shard), 'sitelinks_sets', {sitelinks_set_id => $new_ssid}, where => {sitelinks_set_id => $ssid});
                }
            }
        }
    }
}

sub fix_sitelinks {
    my ($slids) = @_;

    $log->out("Running fix_sitelinks()");
    $log->out("sitelinks_ids: @$slids");

    my $slids_data = {};
    for my $shard (ppc_shards()) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT sl_id FROM sitelinks_links}, WHERE => {sl_id => $slids}]);
        $slids_data->{$_->{sl_id}}->{$shard} = 1 for @$rows;
    }

    for my $slid (@$slids) {
        my $slid_data = $slids_data->{$slid};
        my @slid_shards = keys %$slid_data;

        next unless @slid_shards > 1;

        my $shard_where_update = $slid_shards[1];
        my $new_slid = !$DRY_RUN ? get_new_id('sl_id') : q{`get_new_id('sl_id')`};

        $log->out("Changing sitelink_id#${slid} in shard#${shard_where_update} to $new_slid");

        # Заменяем id сайтлинка
        if (!$DRY_RUN) {
            do_update_table(PPC(shard => $shard_where_update), 'sitelinks_set_to_link', {sl_id => $new_slid}, where => {sl_id => $slid});
            do_update_table(PPC(shard => $shard_where_update), 'sitelinks_links', {sl_id => $new_slid}, where => {sl_id => $slid});
        }
    }
}

sub fix_minus_words {
    my ($mwids) = @_;

    $log->out("Running fix_minus_words()");
    $log->out("mw_ids: @$mwids");

    my $mwids_data = {};
    for my $shard (ppc_shards()) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT mw_id FROM minus_words}, WHERE => {mw_id => $mwids}]);
        $mwids_data->{$_->{mw_id}}->{$shard} = 1 for @$rows;
    }

    # Т.к. индекса по phrases.mw_id нет, выберем нужные данные из phrases для обновления
    my $pids_by_mwid = {};
    $pids_by_mwid->{(sort keys %{$mwids_data->{$_}})[1]}->{$_} = [] for grep { scalar(keys %{$mwids_data->{$_}}) > 1 } @$mwids;
    for my $shard (keys %$pids_by_mwid) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT pid, mw_id FROM phrases}, WHERE => {mw_id => [keys %{$pids_by_mwid->{$shard}}]}]);
        push @{$pids_by_mwid->{$shard}->{$_->{mw_id}}}, $_->{pid} for @$rows;
    }

    for my $mwid (@$mwids) {
        my $mwid_data = $mwids_data->{$mwid};
        my @mwid_shards = sort keys %$mwid_data;

        next unless @mwid_shards > 1;

        my $shard_where_update = $mwid_shards[1];
        my $new_mwid = !$DRY_RUN ? get_new_id('mw_id') : q{`get_new_id('mw_id')`};
        my $pids = $pids_by_mwid->{$shard_where_update}->{$mwid};

        $log->out("Changing mw_id#${mwid} (pids: @$pids) in shard#${shard_where_update} to $new_mwid");

        # Заменяем id минус-слов
        if (!$DRY_RUN) {
            do_update_table(PPC(shard => $shard_where_update), 'phrases', {mw_id => $new_mwid}, where => {mw_id => $mwid, pid => $pids});
            do_update_table(PPC(shard => $shard_where_update), 'minus_words', {mw_id => $new_mwid}, where => {mw_id => $mwid});
        }
    }
}

sub fix_org_details {
    my ($odids) = @_;

    $log->out("Running fix_org_details()");
    $log->out("org_details_ids: @$odids");

    my $odids_data = {};
    for my $shard (ppc_shards()) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT org_details_id FROM org_details}, WHERE => {org_details_id => $odids}]);
        $odids_data->{$_->{org_details_id}}->{$shard} = 1 for @$rows;
    }

    for my $odid (@$odids) {
        my $odid_data = $odids_data->{$odid};
        my @odid_shards = keys %$odid_data;

        next unless @odid_shards > 1;

        my $shard_where_update = $odid_shards[1];
        my $new_odid = !$DRY_RUN ? get_new_id('org_details_id') : q{`get_new_id('org_details_id')`};

        $log->out("Changing org_details_id#${odid} in shard#${shard_where_update} to $new_odid");

        # Заменяем id данных организации
        if (!$DRY_RUN) {
            do_update_table(PPC(shard => $shard_where_update), 'vcards', {org_details_id => $new_odid}, where => {org_details_id => $odid});
            do_update_table(PPC(shard => $shard_where_update), 'org_details', {org_details_id => $new_odid}, where => {org_details_id => $odid});
        }
    }
}

sub fix_users_notifications {
    my ($unids) = @_;

    $log->out("Running fix_users_notifications()");
    $log->out("notification_ids: @$unids");

    my $unids_data = {};
    for my $shard (ppc_shards()) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT notification_id FROM users_notifications}, WHERE => {notification_id => $unids}]);
        $unids_data->{$_->{notification_id}}->{$shard} = 1 for @$rows;
    }

    for my $unid (@$unids) {
        my $unid_data = $unids_data->{$unid};
        my @unid_shards = keys %$unid_data;

        next unless @unid_shards > 1;

        my $shard_where_update = $unid_shards[1];
        my $new_unid = !$DRY_RUN ? get_new_id('users_notification_id') : q{`get_new_id('users_notification_id')`};

        $log->out("Changing notification_id#${unid} in shard#${shard_where_update} to $new_unid");

        if (!$DRY_RUN) {
            do_update_table(PPC(shard => $shard_where_update), 'users_notifications_details', {notification_id => $new_unid}, where => {notification_id => $unid});
            do_update_table(PPC(shard => $shard_where_update), 'push_notifications_process', {notification_id => $new_unid}, where => {notification_id => $unid});
            do_update_table(PPC(shard => $shard_where_update), 'users_notifications', {notification_id => $new_unid}, where => {notification_id => $unid});
        }
    }
}

sub fix_opt_camp_postpone {
    my ($ocpids) = @_;

    $log->out("Running fix_opt_camp_postpone()");
    $log->out("opt_camp_postpone_ids: @$ocpids");

    my $ocpids_data = {};
    for my $shard (ppc_shards()) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT id FROM optimizing_campaign_postpone}, WHERE => {id => $ocpids}]);
        $ocpids_data->{$_->{id}}->{$shard} = 1 for @$rows;
    }

    for my $ocpid (@$ocpids) {
        my $ocpid_data = $ocpids_data->{$ocpid};
        my @ocpid_shards = keys %$ocpid_data;

        next unless @ocpid_shards > 1;

        # Nothing to do
        $log->die("[case0] Unknown case");
    }
}

sub fix_banner_images_pool {
    my ($impids) = @_;

    $log->out("Running fix_banner_images_pool()");
    $log->out("banner_images_pool_ids: @$impids");

    my $metabase_shard = get_shard_multi(banner_images_pool_id => $impids);
    my $impids_data = {};
    for my $shard (ppc_shards()) {
        my $rows = get_all_sql(PPC(shard => $shard), [q{SELECT imp_id, ClientID FROM banner_images_pool}, WHERE => {imp_id => $impids}]);
        $impids_data->{$_->{imp_id}}->{$shard} = $_->{ClientID} for @$rows;
    }

    for my $impid (@$impids) {
        my $m_shard = $metabase_shard->{$impid};
        my $impid_data = $impids_data->{$impid};
        my @impid_shards = keys %$impid_data;

        next if (!$m_shard && !@impid_shards) || ($m_shard && @impid_shards == 1 && $m_shard == $impid_shards[0]);

        if (!@impid_shards) {
            # Пула картинок нет ни в одном шарде. Удалим из метабазы.
            $log->out("[case0] Cannot find banner_images_pool_id#${impid} in any shard. Deleting from metabase.");
            delete_shard(banner_images_pool_id => $impid) if !$DRY_RUN;
        } elsif (@impid_shards == 1) {
            # Пул картинок находится в шарде, отличным от метабазы (или не записан там). Перепропишем шард.
            my $shard = $impid_shards[0];
            $log->out("[case1] Setting shard#${shard} for banner_images_pool_id#${impid}");
            delete_shard(banner_images_pool_id => $impid) if $m_shard && !$DRY_RUN;
            save_shard(banner_images_pool_id => $impid, ClientID => $impid_data->{$shard}) if !$DRY_RUN;
        } else {
            # Пул картинок в нескольких шардах
            my @other_shards = grep { !$m_shard || $m_shard != $_ } @impid_shards;

            if (!$m_shard || !exists $impid_data->{$m_shard}) {
                my $shard = shift @other_shards;
                $log->out("[case2.1] Setting shard#${shard} for banner_images_pool_id#${impid}");
                delete_shard(banner_images_pool_id => $impid) if $m_shard && !$DRY_RUN;
                save_shard(banner_images_pool_id => $impid, ClientID => $impid_data->{$shard}) if !$DRY_RUN;
            }

            for my $shard (@other_shards) {
                my $new_impid = !$DRY_RUN ? get_new_id('banner_images_pool_id', ClientID => $impid_data->{$shard}) : q{`get_new_id('banner_images_pool_id', ClientID => ...)`};
                $log->out("[case2.2] Changing banner_images_pool_id#${impid} in shard#${shard} to $new_impid");

                # Заменяем id пула картинок
                if (!$DRY_RUN) {
                    do_update_table(PPC(shard => $shard), 'banner_images_pool', {imp_id => $new_impid}, where => {imp_id => $impid});
                }
            }
        }
    }
}

__DATA__
{
   "bid" : {
      "1215471816" : 1,
      "1215471164" : 1,
      "1215472300" : 1,
      "1215472738" : 1,
      "1215471986" : 1,
      "1215472068" : 1,
      "1215471681" : 1,
      "1214572314" : 1,
      "1215471347" : 1,
      "1215472657" : 1,
      "1215472700" : 1,
      "1215472676" : 1,
      "1215472415" : 1,
      "1215471433" : 1,
      "1215472413" : 1,
      "1215472716" : 1,
      "1215471813" : 1,
      "1215472281" : 1,
      "1215472280" : 1,
      "1215471811" : 1,
      "1215472728" : 1,
      "1215471686" : 1,
      "1215471776" : 1,
      "1215472059" : 1,
      "1215472677" : 1,
      "1215471929" : 1,
      "1215472678" : 1,
      "1215471830" : 1,
      "1215472712" : 1,
      "1215472180" : 1,
      "1215472733" : 1,
      "1215471435" : 1,
      "1215471985" : 1,
      "1215472186" : 1,
      "1215471080" : 1,
      "1215472278" : 1,
      "1215471437" : 1,
      "1215471077" : 1,
      "1215471772" : 1,
      "1215471726" : 1,
      "1215471593" : 1,
      "1215471815" : 1,
      "1215472662" : 1,
      "1215471875" : 1,
      "1215472279" : 1,
      "1215472653" : 1,
      "1215471168" : 1,
      "1215471818" : 1,
      "1215471446" : 1,
      "1215472710" : 1,
      "1215471884" : 1,
      "1215472183" : 1,
      "1215472609" : 1,
      "1215471778" : 1,
      "1215472697" : 1,
      "1215471881" : 1,
      "1215472181" : 1,
      "1215472474" : 1,
      "1215471586" : 1,
      "1215471886" : 1,
      "1215472740" : 1,
      "1215472302" : 1,
      "1215472178" : 1,
      "1215471078" : 1,
      "1215471688" : 1,
      "1215472418" : 1,
      "1215471765" : 1,
      "1215471809" : 1,
      "1215472681" : 1,
      "1215472301" : 1,
      "1215471447" : 1,
      "1215472282" : 1,
      "1215471592" : 1,
      "1215471503" : 1,
      "1215471438" : 1,
      "1215471428" : 1,
      "1215472287" : 1,
      "1215471442" : 1,
      "1215472289" : 1,
      "1215472067" : 1,
      "1215472702" : 1,
      "1215471439" : 1,
      "1215472719" : 1,
      "1215471826" : 1,
      "1215472722" : 1,
      "1215472175" : 1,
      "1215472475" : 1,
      "1215471766" : 1,
      "1215472184" : 1,
      "1215472066" : 1,
      "1215472277" : 1,
      "1215472182" : 1,
      "1215472134" : 1,
      "1215471825" : 1,
      "1215471877" : 1,
      "1215472179" : 1,
      "1215471587" : 1,
      "1215472292" : 1,
      "1215471082" : 1,
      "1215472669" : 1,
      "1215471831" : 1,
      "1215471343" : 1,
      "1215472707" : 1,
      "1215471584" : 1,
      "1215471987" : 1,
      "1215471692" : 1,
      "1215472061" : 1,
      "1215471989" : 1,
      "1215472290" : 1,
      "1215472001" : 1,
      "1215471764" : 1,
      "1215472673" : 1,
      "1215472730" : 1,
      "1215471723" : 1,
      "1215471690" : 1,
      "1215472744" : 1,
      "1215472556" : 1,
      "1215472063" : 1,
      "1215472684" : 1,
      "1215472675" : 1,
      "1214563864" : 1,
      "1215472416" : 1,
      "1215471769" : 1,
      "1215471774" : 1,
      "1215472062" : 1,
      "1215472128" : 1,
      "1215472706" : 1,
      "1215472176" : 1,
      "1215471079" : 1,
      "1215471829" : 1,
      "1215471691" : 1,
      "1215472126" : 1,
      "1215472671" : 1,
      "1215471995" : 1,
      "1215471341" : 1,
      "1215472057" : 1,
      "1215472741" : 1,
      "1215471500" : 1,
      "1215471450" : 1,
      "1215471814" : 1,
      "1215472709" : 1,
      "1215472345" : 1,
      "1215472273" : 1,
      "1215472002" : 1,
      "1215472303" : 1,
      "1215472665" : 1,
      "1215472714" : 1,
      "1215471595" : 1,
      "1215471722" : 1,
      "1215471580" : 1,
      "1215471777" : 1,
      "1215471430" : 1,
      "1215472674" : 1,
      "1215472692" : 1,
      "1215471581" : 1,
      "1215471497" : 1,
      "1215472274" : 1,
      "1215472687" : 1,
      "1215472065" : 1,
      "1215471827" : 1,
      "1215472476" : 1,
      "1215128785" : 1,
      "1215472727" : 1,
      "1215471819" : 1,
      "1215472412" : 1,
      "1215472225" : 1,
      "1215472682" : 1,
      "1215472680" : 1,
      "1215472275" : 1,
      "1215472473" : 1,
      "1215472129" : 1,
      "1215472666" : 1,
      "1215471930" : 1,
      "1215471431" : 1,
      "1215471763" : 1,
      "1215471444" : 1,
      "1215471993" : 1,
      "1215471689" : 1,
      "1215471494" : 1,
      "1215472651" : 1,
      "1215471998" : 1,
      "1215471585" : 1,
      "1215471451" : 1,
      "1215472747" : 1,
      "1215471725" : 1,
      "1215471879" : 1,
      "1215472185" : 1,
      "1215471429" : 1,
      "1215472723" : 1,
      "1215471824" : 1,
      "1215471449" : 1,
      "1215471775" : 1,
      "1215472718" : 1,
      "1215472554" : 1,
      "1215471342" : 1,
      "1215472291" : 1,
      "1215471501" : 1,
      "1215472654" : 1,
      "1215472690" : 1,
      "1215472685" : 1,
      "1215472748" : 1,
      "1215472344" : 1,
      "1215472296" : 1,
      "1215471767" : 1,
      "1215471823" : 1,
      "1215472294" : 1,
      "1215472660" : 1,
      "1215471810" : 1,
      "1215471348" : 1,
      "1215472679" : 1,
      "1215472689" : 1,
      "1215472725" : 1,
      "1215471885" : 1,
      "1215471687" : 1,
      "1215471821" : 1,
      "1215472663" : 1,
      "1215472417" : 1,
      "1215471583" : 1,
      "1215471876" : 1,
      "1215471590" : 1,
      "1215471349" : 1,
      "1215471768" : 1,
      "1215472720" : 1,
      "1215472299" : 1,
      "1215472739" : 1,
      "1215471432" : 1,
      "1215471507" : 1,
      "1215472746" : 1,
      "1215472715" : 1,
      "1215471505" : 1,
      "1215471165" : 1,
      "1215471685" : 1,
      "1215471990" : 1,
      "1215472734" : 1,
      "1215472729" : 1,
      "1215472069" : 1,
      "1214563865" : 1,
      "1215472691" : 1,
      "1215471344" : 1,
      "1215472698" : 1,
      "1215472297" : 1,
      "1214590761" : 1,
      "1215472705" : 1,
      "1215471448" : 1,
      "1215472120" : 1,
      "1215472342" : 1,
      "1215472127" : 1,
      "1215471817" : 1,
      "1215471878" : 1,
      "1215471988" : 1,
      "1215471682" : 1,
      "1215471241" : 1,
      "1215472226" : 1,
      "1215471984" : 1,
      "1215472659" : 1,
      "1215472119" : 1,
      "1215472658" : 1,
      "1215471167" : 1,
      "1215472672" : 1,
      "1215471506" : 1,
      "1215472742" : 1,
      "1215471589" : 1,
      "1215471994" : 1,
      "1215471992" : 1,
      "1215471761" : 1,
      "1215471591" : 1,
      "1215472696" : 1,
      "1215471882" : 1,
      "1215472661" : 1,
      "1215472477" : 1,
      "1215472737" : 1,
      "1215472694" : 1,
      "1215472717" : 1,
      "1215471997" : 1,
      "1215472735" : 1,
      "1215472000" : 1,
      "1215472655" : 1,
      "1215471440" : 1,
      "1214585641" : 1,
      "1215472133" : 1,
      "1215472745" : 1,
      "1215472732" : 1,
      "1214565839" : 1,
      "1215472177" : 1,
      "1215472132" : 1,
      "1215472726" : 1,
      "1215472003" : 1,
      "1215472124" : 1,
      "1215472060" : 1,
      "1215471434" : 1,
      "1215471166" : 1,
      "1215472131" : 1,
      "1215472652" : 1,
      "1215471820" : 1,
      "1215472668" : 1,
      "1215472283" : 1,
      "1215471441" : 1,
      "1215471504" : 1,
      "1215472736" : 1,
      "1215471828" : 1,
      "1215472670" : 1,
      "1215472688" : 1,
      "1215471771" : 1,
      "1215472298" : 1,
      "1215472058" : 1,
      "1215471340" : 1,
      "1214566465" : 1,
      "1215471629" : 1,
      "1215472708" : 1,
      "1215471496" : 1,
      "1215471991" : 1,
      "1215471680" : 1,
      "1215472123" : 1,
      "1215471436" : 1,
      "1215471928" : 1,
      "1215471443" : 1,
      "1215471773" : 1,
      "1215472286" : 1,
      "1215472064" : 1,
      "1215472695" : 1,
      "1215472284" : 1,
      "1215471594" : 1,
      "1214563815" : 1,
      "1215471346" : 1,
      "1215472224" : 1,
      "1215471887" : 1,
      "1215471339" : 1,
      "1215471588" : 1,
      "1214565838" : 1,
      "1215472555" : 1,
      "1215472130" : 1,
      "1215472285" : 1,
      "1215471883" : 1,
      "1215472704" : 1,
      "1215472683" : 1,
      "1215472743" : 1,
      "1215472656" : 1,
      "1215472686" : 1,
      "1215471762" : 1,
      "1215471812" : 1,
      "1215471724" : 1,
      "1214567422" : 1,
      "1215471345" : 1,
      "1215472699" : 1,
      "1214567320" : 1,
      "1215472223" : 1,
      "1215472731" : 1,
      "1215472125" : 1,
      "1215471999" : 1,
      "1215472711" : 1,
      "1215471888" : 1,
      "1215472276" : 1,
      "1214572285" : 1,
      "1215472667" : 1,
      "1215471880" : 1,
      "1215471499" : 1,
      "1215472343" : 1,
      "1215471582" : 1,
      "1215471822" : 1,
      "1215472721" : 1,
      "1215472293" : 1,
      "1214564289" : 1,
      "1215471495" : 1,
      "1214566483" : 1,
      "1215472121" : 1,
      "1215472724" : 1,
      "1215471684" : 1,
      "1215472419" : 1,
      "1215471996" : 1,
      "1215472122" : 1,
      "1215472693" : 1,
      "1215298154" : 1,
      "1215472608" : 1,
      "1215472701" : 1,
      "1215472713" : 1,
      "1215472288" : 1,
      "1215472346" : 1,
      "1215471502" : 1,
      "1215298155" : 1,
      "1215471498" : 1,
      "1215472187" : 1,
      "1215471081" : 1,
      "1215472414" : 1,
      "1214563816" : 1,
      "1214591130" : 1,
      "1215471770" : 1,
      "1215471445" : 1,
      "1215472703" : 1,
      "1215472295" : 1,
      "1215471727" : 1,
      "1215472664" : 1,
      "1215471683" : 1
   },
   "pid" : {
      "905568272" : 1,
      "905565949" : 1,
      "905422850" : 1,
      "905403600" : 1,
      "905532301" : 1,
      "905366311" : 1,
      "905565767" : 1,
      "905567680" : 1,
      "905565991" : 1,
      "905565224" : 1,
      "905366354" : 1,
      "0" : 1,
      "905564745" : 1,
      "905289212" : 1,
      "905403804" : 1,
      "905366366" : 1,
      "905566110" : 1,
      "905446050" : 1,
      "905366460" : 1,
      "905289253" : 1,
      "905532143" : 1,
      "905366339" : 1,
      "905289454" : 1,
      "905289173" : 1,
      "905366342" : 1,
      "905564692" : 1,
      "905565287" : 1,
      "905567686" : 1,
      "905289256" : 1,
      "905565705" : 1,
      "905422947" : 1,
      "905565704" : 1,
      "905366352" : 1,
      "905568043" : 1,
      "905422890" : 1,
      "905532195" : 1,
      "905532303" : 1,
      "905422861" : 1,
      "905366341" : 1,
      "905475492" : 1,
      "905567347" : 1,
      "905566054" : 1,
      "905403321" : 1,
      "905532312" : 1,
      "905446077" : 1,
      "905475497" : 1,
      "905289332" : 1,
      "905475473" : 1,
      "905565342" : 1,
      "905475501" : 1,
      "905567982" : 1,
      "905568158" : 1,
      "905289207" : 1,
      "905445916" : 1,
      "905475437" : 1,
      "905532308" : 1,
      "905564688" : 1,
      "905366319" : 1,
      "905366307" : 1,
      "905289208" : 1,
      "905568279" : 1,
      "905475446" : 1,
      "905422858" : 1,
      "905568049" : 1,
      "905289168" : 1,
      "905565295" : 1,
      "905564633" : 1,
      "905565290" : 1,
      "905565702" : 1,
      "905567681" : 1,
      "905532302" : 1,
      "905532313" : 1,
      "905567979" : 1,
      "905565770" : 1,
      "905532103" : 1,
      "905567788" : 1,
      "905568053" : 1,
      "905532300" : 1,
      "905532299" : 1,
      "905446052" : 1,
      "905565279" : 1,
      "905403691" : 1,
      "905564626" : 1,
      "905289393" : 1,
      "905567674" : 1,
      "905568155" : 1,
      "905366345" : 1,
      "905289211" : 1,
      "905564684" : 1,
      "905422844" : 1,
      "905422927" : 1,
      "905475488" : 1,
      "905403631" : 1,
      "905532309" : 1,
      "905568269" : 1,
      "905475484" : 1,
      "905446055" : 1,
      "905366349" : 1,
      "905567977" : 1,
      "905366323" : 1,
      "905532221" : 1,
      "905564680" : 1,
      "905403487" : 1,
      "905532281" : 1,
      "905567340" : 1,
      "905565635" : 1,
      "905564560" : 1,
      "905289329" : 1,
      "905475445" : 1,
      "905568211" : 1,
      "905445963" : 1,
      "905366399" : 1,
      "905422854" : 1,
      "905565995" : 1,
      "905475435" : 1,
      "905422843" : 1,
      "905475470" : 1,
      "905403322" : 1,
      "905422839" : 1,
      "905567980" : 1,
      "905422685" : 1,
      "905567984" : 1,
      "905475502" : 1,
      "905366298" : 1,
      "905446076" : 1,
      "905568275" : 1,
      "905289164" : 1,
      "905366467" : 1,
      "905568056" : 1,
      "905475443" : 1,
      "905403219" : 1,
      "905445875" : 1,
      "905475472" : 1,
      "905567677" : 1,
      "905565773" : 1,
      "905567603" : 1,
      "905403482" : 1,
      "905475485" : 1,
      "905564624" : 1,
      "905567691" : 1,
      "905445874" : 1,
      "905565993" : 1,
      "905565703" : 1,
      "905475447" : 1,
      "905567841" : 1,
      "905366427" : 1,
      "905568213" : 1,
      "905289251" : 1,
      "905366318" : 1,
      "905564635" : 1,
      "905567986" : 1,
      "905289214" : 1,
      "905567685" : 1,
      "905366468" : 1,
      "905422948" : 1,
      "905567548" : 1,
      "905289160" : 1,
      "905567981" : 1,
      "905289336" : 1,
      "905475448" : 1,
      "905566053" : 1,
      "905475480" : 1,
      "905403468" : 1,
      "905289335" : 1,
      "905532145" : 1,
      "905289205" : 1,
      "905366464" : 1,
      "905422930" : 1,
      "905422863" : 1,
      "905567790" : 1,
      "905289456" : 1,
      "905403224" : 1,
      "905289453" : 1,
      "905446054" : 1,
      "905567360" : 1,
      "905567698" : 1,
      "905402996" : 1,
      "905564561" : 1,
      "905403601" : 1,
      "905532138" : 1,
      "905564689" : 1,
      "905568278" : 1,
      "905403599" : 1,
      "905565278" : 1,
      "905289338" : 1,
      "905366317" : 1,
      "905564743" : 1,
      "905564742" : 1,
      "905564627" : 1,
      "905532142" : 1,
      "905568214" : 1,
      "905403581" : 1,
      "905403488" : 1,
      "905567693" : 1,
      "905567896" : 1,
      "905568099" : 1,
      "905289249" : 1,
      "905366465" : 1,
      "905289213" : 1,
      "905567684" : 1,
      "905422950" : 1,
      "905568048" : 1,
      "905532282" : 1,
      "905565633" : 1,
      "905403805" : 1,
      "905366302" : 1,
      "905565774" : 1,
      "905475490" : 1,
      "905565771" : 1,
      "905564628" : 1,
      "905568215" : 1,
      "905568045" : 1,
      "905366296" : 1,
      "905567787" : 1,
      "905565918" : 1,
      "905366472" : 1,
      "905422849" : 1,
      "905289167" : 1,
      "905289452" : 1,
      "905289461" : 1,
      "905475452" : 1,
      "905566051" : 1,
      "905289210" : 1,
      "905289457" : 1,
      "905366461" : 1,
      "905567694" : 1,
      "905289250" : 1,
      "905475436" : 1,
      "905568052" : 1,
      "905568051" : 1,
      "905567897" : 1,
      "905289206" : 1,
      "905422840" : 1,
      "905564636" : 1,
      "905366310" : 1,
      "905366324" : 1,
      "905475434" : 1,
      "905532307" : 1,
      "905366344" : 1,
      "905422841" : 1,
      "905289455" : 1,
      "905289449" : 1,
      "905532305" : 1,
      "905366303" : 1,
      "905532141" : 1,
      "905565950" : 1,
      "905567289" : 1,
      "905564504" : 1,
      "905567690" : 1,
      "905445915" : 1,
      "905475499" : 1,
      "905565768" : 1,
      "905564741" : 1,
      "905422706" : 1,
      "905564630" : 1,
      "905403719" : 1,
      "905565992" : 1,
      "905289394" : 1,
      "905475451" : 1,
      "905532283" : 1,
      "905475509" : 1,
      "905567350" : 1,
      "905567395" : 1,
      "905403323" : 1,
      "905366340" : 1,
      "905422707" : 1,
      "905568273" : 1,
      "905289450" : 1,
      "905475479" : 1,
      "905567352" : 1,
      "905564623" : 1,
      "905366456" : 1,
      "905567673" : 1,
      "905567351" : 1,
      "905422952" : 1,
      "905567983" : 1,
      "905289154" : 1,
      "905422729" : 1,
      "905289391" : 1,
      "905422929" : 1,
      "905366453" : 1,
      "905366353" : 1,
      "905567489" : 1,
      "905567355" : 1,
      "905445962" : 1,
      "905289155" : 1,
      "905568059" : 1,
      "905475486" : 1,
      "905567699" : 1,
      "905289257" : 1,
      "905403696" : 1,
      "905366470" : 1,
      "905567342" : 1,
      "905475487" : 1,
      "905289464" : 1,
      "905366337" : 1,
      "905565923" : 1,
      "905475441" : 1,
      "905567357" : 1,
      "905532140" : 1,
      "905567898" : 1,
      "905564685" : 1,
      "905475449" : 1,
      "905567789" : 1,
      "905422852" : 1,
      "905475440" : 1,
      "905403039" : 1,
      "905422682" : 1,
      "905567549" : 1,
      "905366343" : 1,
      "905475478" : 1,
      "905565340" : 1,
      "905565291" : 1,
      "905475500" : 1,
      "905568160" : 1,
      "905366350" : 1,
      "905567692" : 1,
      "905564632" : 1,
      "905475476" : 1,
      "905446049" : 1,
      "905565766" : 1,
      "905475489" : 1,
      "905422891" : 1,
      "905422773" : 1,
      "905532311" : 1,
      "905403282" : 1,
      "905568041" : 1,
      "905289331" : 1,
      "905568268" : 1,
      "905289330" : 1,
      "905567349" : 1,
      "905366365" : 1,
      "905565294" : 1,
      "905565223" : 1,
      "905366309" : 1,
      "905403483" : 1,
      "905568054" : 1,
      "905567545" : 1,
      "905532304" : 1,
      "905289209" : 1,
      "905565915" : 1,
      "905532310" : 1,
      "905564681" : 1,
      "905568057" : 1,
      "905475508" : 1,
      "905475495" : 1,
      "905366458" : 1,
      "905422864" : 1,
      "905475450" : 1,
      "905402993" : 1,
      "905567683" : 1,
      "905565341" : 1,
      "905403692" : 1,
      "905567444" : 1,
      "905568040" : 1,
      "905567490" : 1,
      "905422853" : 1,
      "905403630" : 1,
      "905366462" : 1,
      "905567689" : 1,
      "905403598" : 1,
      "905568209" : 1,
      "905289448" : 1,
      "905565996" : 1,
      "905289392" : 1,
      "905567697" : 1,
      "905446048" : 1,
      "905422855" : 1,
      "905446013" : 1,
      "905564683" : 1,
      "905565920" : 1,
      "905446053" : 1,
      "905289158" : 1,
      "905289156" : 1,
      "905475474" : 1,
      "905564739" : 1,
      "905564687" : 1,
      "905289447" : 1,
      "905289333" : 1,
      "905565344" : 1,
      "905568154" : 1,
      "905567734" : 1,
      "905422842" : 1,
      "905532319" : 1,
      "905567604" : 1,
      "905567978" : 1,
      "905567290" : 1,
      "905565994" : 1,
      "905289165" : 1,
      "905568207" : 1,
      "905475507" : 1,
      "905564690" : 1,
      "905475494" : 1,
      "905566487" : 1,
      "905568042" : 1,
      "905568046" : 1,
      "905568217" : 1,
      "905565292" : 1,
      "905567696" : 1,
      "905289451" : 1,
      "905403040" : 1,
      "905567221" : 1,
      "905422949" : 1,
      "905564622" : 1,
      "905366351" : 1,
      "905289157" : 1,
      "905289459" : 1,
      "905403037" : 1,
      "905568157" : 1,
      "905568274" : 1,
      "905403118" : 1,
      "905403467" : 1,
      "905289215" : 1,
      "905564631" : 1,
      "905403429" : 1,
      "905475491" : 1,
      "905422951" : 1,
      "905289171" : 1,
      "905422931" : 1,
      "905422944" : 1,
      "905567346" : 1,
      "905568271" : 1,
      "905289463" : 1,
      "905402994" : 1,
      "905565776" : 1,
      "905366370" : 1,
      "905366301" : 1,
      "905366325" : 1,
      "905568156" : 1,
      "905567348" : 1,
      "905568159" : 1,
      "905366473" : 1,
      "905366428" : 1,
      "905564682" : 1,
      "905565282" : 1,
      "905532146" : 1,
      "905366463" : 1,
      "905422684" : 1,
      "905565637" : 1,
      "905565775" : 1,
      "905289252" : 1,
      "905422860" : 1,
      "905566109" : 1,
      "905475439" : 1,
      "905475475" : 1,
      "905564506" : 1,
      "905366459" : 1,
      "905475481" : 1,
      "905532306" : 1,
      "905565281" : 1,
      "905567359" : 1,
      "905567546" : 1,
      "905565922" : 1,
      "905532318" : 1,
      "905567679" : 1,
      "905366367" : 1,
      "905422859" : 1,
      "905567687" : 1,
      "905565277" : 1,
      "905403375" : 1,
      "905565219" : 1,
      "905567396" : 1,
      "905422934" : 1,
      "905567733" : 1,
      "905366308" : 1,
      "905568276" : 1,
      "905366457" : 1,
      "905446012" : 1,
      "905565220" : 1,
      "905422838" : 1,
      "905475438" : 1,
      "905567840" : 1,
      "905567695" : 1,
      "905565951" : 1,
      "905366454" : 1,
      "905564505" : 1,
      "905446051" : 1,
      "905366306" : 1,
      "905567287" : 1,
      "905567900" : 1,
      "905568216" : 1,
      "905422836" : 1,
      "905567547" : 1,
      "905568212" : 1,
      "905366297" : 1,
      "905366316" : 1,
      "905565833" : 1,
      "905422945" : 1,
      "905567700" : 1,
      "905403720" : 1,
      "905422946" : 1,
      "905422851" : 1,
      "905568050" : 1,
      "905568044" : 1,
      "905289462" : 1,
      "905567682" : 1,
      "905567341" : 1,
      "905289172" : 1,
      "905366336" : 1,
      "905564691" : 1,
      "905567339" : 1,
      "905289161" : 1,
      "905366335" : 1,
      "905289170" : 1,
      "905422837" : 1,
      "905568047" : 1,
      "905289254" : 1,
      "905532298" : 1,
      "905475496" : 1,
      "905289258" : 1,
      "905564625" : 1,
      "905403223" : 1,
      "905568270" : 1,
      "905422683" : 1,
      "905564740" : 1,
      "905564629" : 1,
      "905366471" : 1,
      "905366300" : 1,
      "905568210" : 1,
      "905565289" : 1,
      "905403486" : 1,
      "905565221" : 1,
      "905475511" : 1,
      "905366304" : 1,
      "905366305" : 1,
      "905422932" : 1,
      "905366321" : 1,
      "905567985" : 1,
      "905567675" : 1,
      "905567398" : 1,
      "905475442" : 1,
      "905567358" : 1,
      "905403695" : 1,
      "905565285" : 1,
      "905568277" : 1,
      "905366338" : 1,
      "905565777" : 1,
      "905403466" : 1,
      "905565343" : 1,
      "905403629" : 1,
      "905289169" : 1,
      "905366398" : 1,
      "905565293" : 1,
      "905565634" : 1,
      "905289458" : 1,
      "905403324" : 1,
      "905422933" : 1,
      "905403038" : 1,
      "905422928" : 1,
      "905366322" : 1,
      "905403545" : 1,
      "905366368" : 1,
      "905366400" : 1,
      "905366369" : 1,
      "905565283" : 1,
      "905564686" : 1,
      "905532223" : 1,
      "905403221" : 1,
      "905403080" : 1,
      "905445839" : 1,
      "905475471" : 1,
      "905532315" : 1,
      "905366312" : 1,
      "905403690" : 1,
      "905289166" : 1,
      "905289163" : 1,
      "905403320" : 1,
      "905403222" : 1,
      "905403594" : 1,
      "905567344" : 1,
      "905565222" : 1,
      "905366466" : 1,
      "905366469" : 1,
      "905565921" : 1,
      "905567678" : 1,
      "905567688" : 1,
      "905567356" : 1,
      "905568152" : 1,
      "905366346" : 1,
      "905565917" : 1,
      "905366299" : 1,
      "905403632" : 1,
      "905567397" : 1,
      "905565765" : 1,
      "905475512" : 1,
      "905565225" : 1,
      "905532167" : 1,
      "905422862" : 1,
      "905403595" : 1,
      "905532251" : 1,
      "905403481" : 1,
      "905568153" : 1,
      "905366320" : 1,
      "905532166" : 1,
      "905289255" : 1,
      "905568055" : 1,
      "905366313" : 1,
      "905567343" : 1,
      "905289216" : 1,
      "905565914" : 1,
      "905446014" : 1,
      "905289260" : 1,
      "905402995" : 1,
      "905566052" : 1,
      "905567354" : 1,
      "905567676" : 1,
      "905422856" : 1,
      "905289259" : 1,
      "905532317" : 1,
      "905403597" : 1,
      "905567345" : 1,
      "905289460" : 1,
      "905532250" : 1,
      "905532316" : 1,
      "905565288" : 1,
      "905567288" : 1,
      "905564744" : 1,
      "905567899" : 1,
      "905403041" : 1,
      "905403758" : 1,
      "905532139" : 1,
      "905403485" : 1,
      "905289162" : 1,
      "905564634" : 1,
      "905565772" : 1,
      "905289159" : 1,
      "905565286" : 1,
      "905568058" : 1,
      "905403220" : 1,
      "905366315" : 1,
      "905403697" : 1,
      "905532165" : 1,
      "905568218" : 1,
      "905568208" : 1,
      "905403698" : 1,
      "905475493" : 1,
      "905532144" : 1,
      "905475477" : 1,
      "905475444" : 1,
      "905403693" : 1,
      "905567987" : 1,
      "905475483" : 1,
      "905403628" : 1,
      "905289334" : 1,
      "905475510" : 1,
      "905446056" : 1,
      "905565769" : 1,
      "905568280" : 1,
      "905532222" : 1,
      "905366348" : 1,
      "905565916" : 1,
      "905403596" : 1,
      "905403546" : 1,
      "905366455" : 1,
      "905475482" : 1,
      "905289217" : 1,
      "905475498" : 1,
      "905565284" : 1,
      "905289337" : 1,
      "905422857" : 1,
      "905403484" : 1,
      "905403694" : 1,
      "905565280" : 1,
      "905366347" : 1,
      "905532314" : 1,
      "905565636" : 1,
      "905567399" : 1,
      "905565919" : 1,
      "905567353" : 1,
      "905366314" : 1
   },
   "phid" : {
      "4211374847" : 1,
      "4211406266" : 1,
      "4211399436" : 1,
      "4211368326" : 1,
      "4212498693" : 1,
      "4211399468" : 1,
      "4211397007" : 1,
      "4211406132" : 1,
      "4211406018" : 1,
      "4212498789" : 1,
      "4211406013" : 1,
      "4211375861" : 1,
      "4211396826" : 1,
      "4211405996" : 1,
      "4211399054" : 1,
      "4211407714" : 1,
      "4211394094" : 1,
      "4211405907" : 1,
      "4211405679" : 1,
      "4212498768" : 1,
      "4211406028" : 1,
      "4211396855" : 1,
      "4211407563" : 1,
      "4211407254" : 1,
      "4211405972" : 1,
      "4211405902" : 1,
      "4211373394" : 1,
      "4211397170" : 1,
      "4211406173" : 1,
      "4211406154" : 1,
      "4211406375" : 1,
      "4211407798" : 1,
      "4211388688" : 1,
      "4211407755" : 1,
      "4211405900" : 1,
      "4211394085" : 1,
      "4211406139" : 1,
      "4212498727" : 1,
      "4211405899" : 1,
      "4211406277" : 1,
      "4212498710" : 1,
      "4211406534" : 1,
      "4211397700" : 1,
      "4211408969" : 1,
      "4211399439" : 1,
      "4211406159" : 1,
      "4211406172" : 1,
      "4212848159" : 1,
      "4211394086" : 1,
      "4211406524" : 1,
      "4211405699" : 1,
      "4211406337" : 1,
      "4211369769" : 1,
      "4211385173" : 1,
      "4211406336" : 1,
      "4211405974" : 1,
      "4211376357" : 1,
      "4211406157" : 1,
      "4211397707" : 1,
      "4211399070" : 1,
      "4211390935" : 1,
      "4211407709" : 1,
      "4211407760" : 1,
      "4211387235" : 1,
      "4211407145" : 1,
      "4211407165" : 1,
      "4211406126" : 1,
      "4211379922" : 1,
      "4211406351" : 1,
      "4212498772" : 1,
      "4212848158" : 1,
      "4211407363" : 1,
      "4211407732" : 1,
      "4211374900" : 1,
      "4211407238" : 1,
      "4211408059" : 1,
      "4211378895" : 1,
      "4211406467" : 1,
      "4211406791" : 1,
      "4211385187" : 1,
      "4211380446" : 1,
      "4211403450" : 1,
      "4211392889" : 1,
      "4211407712" : 1,
      "4211406020" : 1,
      "4211407785" : 1,
      "4211405691" : 1,
      "4211406011" : 1,
      "4211407031" : 1,
      "4211407228" : 1,
      "4211402795" : 1,
      "4211375627" : 1,
      "4211405988" : 1,
      "4211406479" : 1,
      "4211399067" : 1,
      "4212498730" : 1,
      "4211407810" : 1,
      "4211405768" : 1,
      "4212498723" : 1,
      "4211375110" : 1,
      "4211405689" : 1,
      "4211405825" : 1,
      "4211407799" : 1,
      "4211407559" : 1,
      "4211395319" : 1,
      "4211399443" : 1,
      "4211407174" : 1,
      "4211407234" : 1,
      "4212848166" : 1,
      "4211406901" : 1,
      "4211406378" : 1,
      "4212498699" : 1,
      "4211397705" : 1,
      "4211405830" : 1,
      "4212498698" : 1,
      "4211406327" : 1,
      "4211406905" : 1,
      "4211394089" : 1,
      "4211407769" : 1,
      "4211397121" : 1,
      "4211405696" : 1,
      "4211407801" : 1,
      "4211405898" : 1,
      "4211407187" : 1,
      "4211406655" : 1,
      "4211399077" : 1,
      "4211405762" : 1,
      "4211385288" : 1,
      "4211406273" : 1,
      "4212498787" : 1,
      "4211399431" : 1,
      "4211397365" : 1,
      "4211406262" : 1,
      "4211399469" : 1,
      "4211385193" : 1,
      "4211405826" : 1,
      "4211396856" : 1,
      "4211385143" : 1,
      "4211399441" : 1,
      "4211406651" : 1,
      "4211406149" : 1,
      "4211405754" : 1,
      "4211387757" : 1,
      "4211407556" : 1,
      "4211406518" : 1,
      "4211405838" : 1,
      "4211369768" : 1,
      "4211405847" : 1,
      "4211398145" : 1,
      "4211407362" : 1,
      "4211407774" : 1,
      "4211406681" : 1,
      "4211407803" : 1,
      "4211374743" : 1,
      "4212498703" : 1,
      "4211371847" : 1,
      "4211391970" : 1,
      "4211398914" : 1,
      "4211407237" : 1,
      "4211406004" : 1,
      "4211387232" : 1,
      "4211385157" : 1,
      "4211397353" : 1,
      "4211406276" : 1,
      "4211371841" : 1,
      "4211387360" : 1,
      "4211389594" : 1,
      "4211406443" : 1,
      "4211382702" : 1,
      "4212498792" : 1,
      "4211396819" : 1,
      "4211407791" : 1,
      "4211406146" : 1,
      "4211406889" : 1,
      "4211397504" : 1,
      "4211407726" : 1,
      "4211407765" : 1,
      "4212848161" : 1,
      "4211399056" : 1,
      "4212498748" : 1,
      "4211407032" : 1,
      "4211406466" : 1,
      "4211406328" : 1,
      "4211385139" : 1,
      "4211397699" : 1,
      "4211406151" : 1,
      "4211407560" : 1,
      "4211387234" : 1,
      "4211399450" : 1,
      "4211406342" : 1,
      "4211407777" : 1,
      "4211406656" : 1,
      "4211405973" : 1,
      "4211389691" : 1,
      "4211407236" : 1,
      "4211394093" : 1,
      "4211407557" : 1,
      "4211397704" : 1,
      "4211387876" : 1,
      "4211399076" : 1,
      "4211405978" : 1,
      "4211389119" : 1,
      "4211407266" : 1,
      "4211406315" : 1,
      "4211407778" : 1,
      "4211397708" : 1,
      "4211387521" : 1,
      "4212498689" : 1,
      "4211407195" : 1,
      "4211405828" : 1,
      "4211399925" : 1,
      "4211407739" : 1,
      "4211406899" : 1,
      "4212848163" : 1,
      "4211405688" : 1,
      "4211406349" : 1,
      "4211407261" : 1,
      "4211368581" : 1,
      "4211406478" : 1,
      "4211405831" : 1,
      "4211405975" : 1,
      "4211368695" : 1,
      "4211407193" : 1,
      "4211385292" : 1,
      "4211397703" : 1,
      "4211387639" : 1,
      "4211397850" : 1,
      "4211397689" : 1,
      "4211406033" : 1,
      "4211407186" : 1,
      "4211385164" : 1,
      "4211368638" : 1,
      "4211388044" : 1,
      "4211385113" : 1,
      "4211397371" : 1,
      "4212498688" : 1,
      "4212498716" : 1,
      "4211397710" : 1,
      "4211406331" : 1,
      "4212498751" : 1,
      "4211405809" : 1,
      "4211399452" : 1,
      "4211407719" : 1,
      "4211371800" : 1,
      "4211389593" : 1,
      "4211405682" : 1,
      "4212498718" : 1,
      "4211385176" : 1,
      "4211406282" : 1,
      "4211406131" : 1,
      "4211401846" : 1,
      "4211405764" : 1,
      "4211406893" : 1,
      "4211393419" : 1,
      "4211406526" : 1,
      "4212498762" : 1,
      "4212498788" : 1,
      "4211385196" : 1,
      "4211413191" : 1,
      "4211385432" : 1,
      "4211407359" : 1,
      "4211406668" : 1,
      "4211399425" : 1,
      "4211406152" : 1,
      "4211407787" : 1,
      "4211406002" : 1,
      "4212498686" : 1,
      "4211406906" : 1,
      "4211385190" : 1,
      "4211397712" : 1,
      "4211397686" : 1,
      "4211397564" : 1,
      "4211405760" : 1,
      "4211408060" : 1,
      "4211403067" : 1,
      "4211405997" : 1,
      "4211385161" : 1,
      "4212498717" : 1,
      "4211405993" : 1,
      "4211406030" : 1,
      "4211407711" : 1,
      "4211399426" : 1,
      "4211405763" : 1,
      "4211374899" : 1,
      "4211399072" : 1,
      "4211405905" : 1,
      "4211405677" : 1,
      "4211405848" : 1,
      "4211399709" : 1,
      "4211399433" : 1,
      "4212498777" : 1,
      "4211396824" : 1,
      "4211407149" : 1,
      "4211397358" : 1,
      "4211406908" : 1,
      "4211406521" : 1,
      "4211407270" : 1,
      "4211375168" : 1,
      "4211405827" : 1,
      "4211397470" : 1,
      "4211405779" : 1,
      "4211385166" : 1,
      "4211371494" : 1,
      "4211399432" : 1,
      "4211385169" : 1,
      "4211399058" : 1,
      "4211406354" : 1,
      "4211406169" : 1,
      "4211406679" : 1,
      "4212498695" : 1,
      "4211407233" : 1,
      "4211406334" : 1,
      "4212498769" : 1,
      "4211407718" : 1,
      "4211385178" : 1,
      "4211385170" : 1,
      "4211407804" : 1,
      "4211407708" : 1,
      "4211406163" : 1,
      "4211405843" : 1,
      "4211399081" : 1,
      "4211406166" : 1,
      "4211387725" : 1,
      "4211405784" : 1,
      "4211405761" : 1,
      "4211406900" : 1,
      "4211406891" : 1,
      "4211399451" : 1,
      "4211405853" : 1,
      "4211388908" : 1,
      "4211406015" : 1,
      "4211407229" : 1,
      "4211389033" : 1,
      "4211397563" : 1,
      "4211405777" : 1,
      "4211399059" : 1,
      "4211398913" : 1,
      "4211406174" : 1,
      "4211407771" : 1,
      "4211405995" : 1,
      "4211406125" : 1,
      "4211407724" : 1,
      "4211406729" : 1,
      "4211397690" : 1,
      "4211388740" : 1,
      "4211407265" : 1,
      "4211387454" : 1,
      "4211406726" : 1,
      "4212498682" : 1,
      "4211407172" : 1,
      "4211385168" : 1,
      "4212848164" : 1,
      "4211407797" : 1,
      "4211406670" : 1,
      "4211405676" : 1,
      "4212498774" : 1,
      "4211407190" : 1,
      "4211399462" : 1,
      "4211405983" : 1,
      "4211397200" : 1,
      "4211406372" : 1,
      "4211371746" : 1,
      "4211394090" : 1,
      "4211385194" : 1,
      "4211406793" : 1,
      "4211385155" : 1,
      "4211407253" : 1,
      "4211406470" : 1,
      "4211407164" : 1,
      "4211406008" : 1,
      "4211406519" : 1,
      "4212498740" : 1,
      "4211406153" : 1,
      "4211407232" : 1,
      "4211408061" : 1,
      "4211385082" : 1,
      "4211408965" : 1,
      "4211406647" : 1,
      "4211385154" : 1,
      "4211405812" : 1,
      "4211407243" : 1,
      "4211388043" : 1,
      "4211397694" : 1,
      "4211385197" : 1,
      "4211407775" : 1,
      "4212498761" : 1,
      "4212498746" : 1,
      "4211385192" : 1,
      "4211405904" : 1,
      "4211407731" : 1,
      "4211397198" : 1,
      "4211406345" : 1,
      "4211406161" : 1,
      "4211406674" : 1,
      "4212498735" : 1,
      "4211374845" : 1,
      "4211397693" : 1,
      "4211397702" : 1,
      "4211407268" : 1,
      "4211397360" : 1,
      "4211375937" : 1,
      "4211401297" : 1,
      "4211403207" : 1,
      "4211403122" : 1,
      "4211405757" : 1,
      "4211385167" : 1,
      "4211399438" : 1,
      "4211385186" : 1,
      "4211406321" : 1,
      "4211394092" : 1,
      "4212498794" : 1,
      "4211399066" : 1,
      "4211397691" : 1,
      "4212498733" : 1,
      "4211406673" : 1,
      "4211406029" : 1,
      "4211396823" : 1,
      "4211405839" : 1,
      "4211388783" : 1,
      "4212848162" : 1,
      "4212498726" : 1,
      "4212498760" : 1,
      "4211406168" : 1,
      "4211388689" : 1,
      "4211407163" : 1,
      "4211397195" : 1,
      "4211407263" : 1,
      "4211388000" : 1,
      "4211374846" : 1,
      "4211406652" : 1,
      "4211386836" : 1,
      "4211378894" : 1,
      "4211374739" : 1,
      "4211397194" : 1,
      "4211399061" : 1,
      "4211406171" : 1,
      "4211385152" : 1,
      "4211406014" : 1,
      "4211385117" : 1,
      "4211405999" : 1,
      "4211405780" : 1,
      "4211407177" : 1,
      "4211385172" : 1,
      "4211405998" : 1,
      "4211389095" : 1,
      "4211387407" : 1,
      "4211401394" : 1,
      "4212498771" : 1,
      "4211407758" : 1,
      "4212498791" : 1,
      "4211406649" : 1,
      "4211406329" : 1,
      "4211407802" : 1,
      "4212498707" : 1,
      "4211396821" : 1,
      "4211396817" : 1,
      "4211406001" : 1,
      "4212498696" : 1,
      "4211399465" : 1,
      "4211408052" : 1,
      "4211410639" : 1,
      "4211407784" : 1,
      "4211371469" : 1,
      "4211406444" : 1,
      "4211399457" : 1,
      "4211406007" : 1,
      "4211408966" : 1,
      "4211407192" : 1,
      "4211406343" : 1,
      "4212498765" : 1,
      "4211407780" : 1,
      "4212498715" : 1,
      "4211406352" : 1,
      "4211374841" : 1,
      "4211405845" : 1,
      "4211406666" : 1,
      "4211405842" : 1,
      "4211408968" : 1,
      "4212498697" : 1,
      "4211407766" : 1,
      "4211374843" : 1,
      "4212498744" : 1,
      "4211371743" : 1,
      "4211406314" : 1,
      "4211385081" : 1,
      "4211399464" : 1,
      "4211399055" : 1,
      "4211406725" : 1,
      "4211407244" : 1,
      "4211407167" : 1,
      "4211407194" : 1,
      "4211399954" : 1,
      "4211405785" : 1,
      "4211407246" : 1,
      "4211407168" : 1,
      "4211407730" : 1,
      "4211405994" : 1,
      "4211399923" : 1,
      "4211408175" : 1,
      "4211406269" : 1,
      "4211407759" : 1,
      "4211405771" : 1,
      "4211397357" : 1,
      "4211377213" : 1,
      "4211405990" : 1,
      "4211374844" : 1,
      "4211405856" : 1,
      "4211406332" : 1,
      "4211397561" : 1,
      "4211399071" : 1,
      "4211396857" : 1,
      "4211406794" : 1,
      "4211406160" : 1,
      "4211406027" : 1,
      "4211405758" : 1,
      "4211406128" : 1,
      "4211407148" : 1,
      "4211405897" : 1,
      "4211385483" : 1,
      "4211406346" : 1,
      "4211385158" : 1,
      "4211406320" : 1,
      "4212498702" : 1,
      "4211399075" : 1,
      "4211390940" : 1,
      "4211378977" : 1,
      "4211397361" : 1,
      "4211406686" : 1,
      "4211399064" : 1,
      "4211386258" : 1,
      "4211398315" : 1,
      "4211399350" : 1,
      "4211406135" : 1,
      "4211407734" : 1,
      "4211399063" : 1,
      "4211407183" : 1,
      "4211410640" : 1,
      "4211406584" : 1,
      "4211399442" : 1,
      "4211371692" : 1,
      "4211407735" : 1,
      "4211397697" : 1,
      "4211397366" : 1,
      "4211400015" : 1,
      "4211396859" : 1,
      "4211406527" : 1,
      "4211385191" : 1,
      "4211385189" : 1,
      "4211405854" : 1,
      "4211407770" : 1,
      "4211406650" : 1,
      "4211390250" : 1,
      "4211385195" : 1,
      "4211371364" : 1,
      "4211406165" : 1,
      "4211397015" : 1,
      "4211385163" : 1,
      "4211408050" : 1,
      "4211405765" : 1,
      "4212498781" : 1,
      "4211385180" : 1,
      "4211406338" : 1,
      "4211406530" : 1,
      "4211385171" : 1,
      "4212498683" : 1,
      "4211413188" : 1,
      "4211407257" : 1,
      "4211394487" : 1,
      "4211405782" : 1,
      "4211406684" : 1,
      "4211403188" : 1,
      "4211407788" : 1,
      "4211407807" : 1,
      "4211407269" : 1,
      "4211405759" : 1,
      "4211397709" : 1,
      "4211407241" : 1,
      "4211406474" : 1,
      "4211406268" : 1,
      "4211399074" : 1,
      "4211399446" : 1,
      "4211406468" : 1,
      "4211407251" : 1,
      "4211405976" : 1,
      "4211397714" : 1,
      "4212498766" : 1,
      "4211398316" : 1,
      "4211406034" : 1,
      "4211407772" : 1,
      "4211406629" : 1,
      "4211387313" : 1,
      "4211397683" : 1,
      "4211406010" : 1,
      "4211406016" : 1,
      "4211399466" : 1,
      "4211406730" : 1,
      "4211397692" : 1,
      "4211407710" : 1,
      "4211407182" : 1,
      "4211385184" : 1,
      "4211401527" : 1,
      "4211397696" : 1,
      "4211407728" : 1,
      "4211407806" : 1,
      "4211406270" : 1,
      "4211406477" : 1,
      "4212498709" : 1,
      "4211398758" : 1,
      "4211387520" : 1,
      "4211405808" : 1,
      "4211397202" : 1,
      "4211399051" : 1,
      "4211406026" : 1,
      "4211399434" : 1,
      "4211406680" : 1,
      "4211385115" : 1,
      "4211399259" : 1,
      "4211385175" : 1,
      "4212498714" : 1,
      "4211399065" : 1,
      "4211407188" : 1,
      "4212498677" : 1,
      "4211406339" : 1,
      "4211385116" : 1,
      "4211385162" : 1,
      "4211405833" : 1,
      "4211406274" : 1,
      "4211406344" : 1,
      "4211406136" : 1,
      "4212498741" : 1,
      "4212498773" : 1,
      "4211388388" : 1,
      "4211396822" : 1,
      "4211407715" : 1,
      "4212498700" : 1,
      "4211406654" : 1,
      "4211405912" : 1,
      "4211405687" : 1,
      "4212498755" : 1,
      "4211407725" : 1,
      "4211402625" : 1,
      "4211388766" : 1,
      "4211371695" : 1,
      "4211399705" : 1,
      "4211407800" : 1,
      "4211388629" : 1,
      "4211406017" : 1,
      "4211385077" : 1,
      "4211406330" : 1,
      "4211406148" : 1,
      "4211397235" : 1,
      "4211399707" : 1,
      "4211406644" : 1,
      "4211407764" : 1,
      "4211408174" : 1,
      "4211406317" : 1,
      "4211397355" : 1,
      "4211406646" : 1,
      "4211399711" : 1,
      "4211387288" : 1,
      "4211407247" : 1,
      "4212498783" : 1,
      "4211397354" : 1,
      "4211413190" : 1,
      "4211408051" : 1,
      "4211405896" : 1,
      "4211407357" : 1,
      "4211394091" : 1,
      "4211406145" : 1,
      "4211385290" : 1,
      "4211375628" : 1,
      "4211405686" : 1,
      "4211406005" : 1,
      "4211407052" : 1,
      "4211401968" : 1,
      "4211405769" : 1,
      "4211373417" : 1,
      "4211406025" : 1,
      "4211406158" : 1,
      "4211375629" : 1,
      "4211388787" : 1,
      "4211405991" : 1,
      "4211399068" : 1,
      "4211407383" : 1,
      "4211399444" : 1,
      "4211388342" : 1,
      "4211406272" : 1,
      "4211394454" : 1,
      "4211368715" : 1,
      "4211407184" : 1,
      "4211399427" : 1,
      "4211399460" : 1,
      "4211407762" : 1,
      "4211405992" : 1,
      "4211406374" : 1,
      "4212498757" : 1,
      "4211397684" : 1,
      "4211379540" : 1,
      "4211406683" : 1,
      "4211371793" : 1,
      "4211406671" : 1,
      "4211368566" : 1,
      "4211371523" : 1,
      "4211406353" : 1,
      "4212498734" : 1,
      "4211371896" : 1,
      "4211407185" : 1,
      "4211406267" : 1,
      "4211407795" : 1,
      "4211397364" : 1,
      "4211407252" : 1,
      "4211406628" : 1,
      "4211406275" : 1,
      "4211385183" : 1,
      "4211407245" : 1,
      "4211396820" : 1,
      "4212498763" : 1,
      "4212498736" : 1,
      "4211408967" : 1,
      "4211387335" : 1,
      "4211407767" : 1,
      "4211408058" : 1,
      "4211405680" : 1,
      "4211388909" : 1,
      "4212498719" : 1,
      "4211399453" : 1,
      "4211399461" : 1,
      "4211406279" : 1,
      "4211408970" : 1,
      "4212498713" : 1,
      "4212498753" : 1,
      "4211371427" : 1,
      "4211405755" : 1,
      "4211399108" : 1,
      "4211406162" : 1,
      "4211374901" : 1,
      "4211405989" : 1,
      "4211407258" : 1,
      "4211405698" : 1,
      "4211405694" : 1,
      "4211406350" : 1,
      "4211406482" : 1,
      "4211405690" : 1,
      "4211407790" : 1,
      "4211405774" : 1,
      "4211399447" : 1,
      "4211405910" : 1,
      "4211405844" : 1,
      "4211408173" : 1,
      "4211397505" : 1,
      "4211374742" : 1,
      "4211406155" : 1,
      "4211399456" : 1,
      "4211387233" : 1,
      "4211407262" : 1,
      "4211388687" : 1,
      "4211398592" : 1,
      "4211398911" : 1,
      "4211407808" : 1,
      "4211371385" : 1,
      "4211397432" : 1,
      "4211388099" : 1,
      "4211385138" : 1,
      "4211406281" : 1,
      "4211407783" : 1,
      "4211385165" : 1,
      "4211406658" : 1,
      "4211407740" : 1,
      "4211406627" : 1,
      "4211407757" : 1,
      "4211405767" : 1,
      "4211385156" : 1,
      "4211407723" : 1,
      "4211405901" : 1,
      "4211369828" : 1,
      "4211407773" : 1,
      "4211388786" : 1,
      "4211407554" : 1,
      "4211406271" : 1,
      "4211407173" : 1,
      "4211407786" : 1,
      "4211407171" : 1,
      "4211406523" : 1,
      "4212498721" : 1,
      "4211406175" : 1,
      "4211397362" : 1,
      "4212498701" : 1,
      "4212498743" : 1,
      "4211407741" : 1,
      "4211406653" : 1,
      "4211407768" : 1,
      "4211405778" : 1,
      "4211405700" : 1,
      "4212498775" : 1,
      "4211397872" : 1,
      "4211399710" : 1,
      "4211397009" : 1,
      "4211406903" : 1,
      "4211406442" : 1,
      "4211397928" : 1,
      "4211405701" : 1,
      "4211397680" : 1,
      "4212498687" : 1,
      "4211405692" : 1,
      "4211397013" : 1,
      "4211397368" : 1,
      "4211406124" : 1,
      "4211407169" : 1,
      "4212848165" : 1,
      "4212498712" : 1,
      "4211374842" : 1,
      "4211374740" : 1,
      "4212498732" : 1,
      "4211397695" : 1,
      "4211406319" : 1,
      "4211397201" : 1,
      "4211385182" : 1,
      "4211406316" : 1,
      "4211405851" : 1,
      "4212498694" : 1,
      "4211406481" : 1,
      "4211393359" : 1,
      "4211405697" : 1,
      "4211405977" : 1,
      "4211405703" : 1,
      "4211366804" : 1,
      "4211406167" : 1,
      "4211405985" : 1,
      "4212498731" : 1,
      "4211406003" : 1,
      "4211406022" : 1,
      "4211394088" : 1,
      "4211388829" : 1,
      "4211399445" : 1,
      "4211407355" : 1,
      "4211406333" : 1,
      "4211388630" : 1,
      "4211406325" : 1,
      "4211406790" : 1,
      "4211406531" : 1,
      "4211405829" : 1,
      "4211406727" : 1,
      "4211407053" : 1,
      "4211406147" : 1,
      "4211406024" : 1,
      "4211405840" : 1,
      "4211380400" : 1,
      "4211385181" : 1,
      "4211374744" : 1,
      "4211397687" : 1,
      "4212498684" : 1,
      "4211405903" : 1,
      "4211408053" : 1,
      "4212498729" : 1,
      "4211407235" : 1,
      "4211406675" : 1,
      "4211399467" : 1,
      "4211405811" : 1,
      "4211406264" : 1,
      "4211397547" : 1,
      "4211406522" : 1,
      "4211385289" : 1,
      "4211396818" : 1,
      "4211373872" : 1,
      "4211406012" : 1,
      "4211383071" : 1,
      "4211407561" : 1,
      "4211406335" : 1,
      "4211407809" : 1,
      "4211406659" : 1,
      "4211399458" : 1,
      "4212498749" : 1,
      "4211407763" : 1,
      "4211397711" : 1,
      "4211406376" : 1,
      "4211406322" : 1,
      "4211397006" : 1,
      "4211406326" : 1,
      "4211399429" : 1,
      "4211407756" : 1,
      "4211399448" : 1,
      "4211371478" : 1,
      "4211406019" : 1,
      "4212498711" : 1,
      "4211406536" : 1,
      "4211405971" : 1,
      "4212498756" : 1,
      "4212498722" : 1,
      "4212498738" : 1,
      "4211405674" : 1,
      "4211407259" : 1,
      "4211388690" : 1,
      "4211407789" : 1,
      "4211406723" : 1,
      "4211384873" : 1,
      "4211397367" : 1,
      "4211406133" : 1,
      "4211399062" : 1,
      "4211406251" : 1,
      "4211397356" : 1,
      "4211407779" : 1,
      "4211407239" : 1,
      "4212498739" : 1,
      "4211405911" : 1,
      "4211374038" : 1,
      "4211399258" : 1,
      "4211405675" : 1,
      "4211407729" : 1,
      "4211406529" : 1,
      "4211385114" : 1,
      "4211375794" : 1,
      "4212498691" : 1,
      "4211405834" : 1,
      "4211398947" : 1,
      "4211407242" : 1,
      "4211405987" : 1,
      "4211405832" : 1,
      "4211407361" : 1,
      "4211406643" : 1,
      "4211397682" : 1,
      "4211399706" : 1,
      "4211399060" : 1,
      "4211406532" : 1,
      "4212498745" : 1,
      "4211406904" : 1,
      "4211385017" : 1,
      "4211389517" : 1,
      "4211406645" : 1,
      "4211407776" : 1,
      "4211406895" : 1,
      "4211399073" : 1,
      "4211393163" : 1,
      "4211378896" : 1,
      "4212498767" : 1,
      "4211405849" : 1,
      "4211406476" : 1,
      "4211407793" : 1,
      "4211407227" : 1,
      "4211396825" : 1,
      "4211389097" : 1,
      "4211407716" : 1,
      "4211407189" : 1,
      "4211371448" : 1,
      "4211405695" : 1,
      "4211406150" : 1,
      "4211399053" : 1,
      "4211397681" : 1,
      "4211380684" : 1,
      "4211399712" : 1,
      "4211406535" : 1,
      "4211406517" : 1,
      "4211407264" : 1,
      "4211407249" : 1,
      "4212498764" : 1,
      "4211406023" : 1,
      "4211397014" : 1,
      "4212498784" : 1,
      "4211385618" : 1,
      "4211405837" : 1,
      "4211407727" : 1,
      "4211371937" : 1,
      "4211385657" : 1,
      "4211400014" : 1,
      "4211397193" : 1,
      "4211407360" : 1,
      "4211406676" : 1,
      "4211405810" : 1,
      "4211387638" : 1,
      "4212498785" : 1,
      "4212498679" : 1,
      "4211388784" : 1,
      "4211406031" : 1,
      "4211407733" : 1,
      "4211398591" : 1,
      "4211406520" : 1,
      "4211388765" : 1,
      "4211406347" : 1,
      "4211397688" : 1,
      "4211388926" : 1,
      "4211407720" : 1,
      "4211385188" : 1,
      "4211406032" : 1,
      "4211405773" : 1,
      "4212498770" : 1,
      "4211406664" : 1,
      "4211397196" : 1,
      "4211406662" : 1,
      "4211374849" : 1,
      "4211405982" : 1,
      "4211397359" : 1,
      "4211406373" : 1,
      "4212498720" : 1,
      "4211385160" : 1,
      "4211385542" : 1,
      "4211407761" : 1,
      "4211399449" : 1,
      "4211397085" : 1,
      "4211380638" : 1,
      "4211374741" : 1,
      "4212498706" : 1,
      "4211407250" : 1,
      "4211399435" : 1,
      "4211371643" : 1,
      "4211406667" : 1,
      "4211406672" : 1,
      "4211405693" : 1,
      "4211406685" : 1,
      "4211405850" : 1,
      "4211406528" : 1,
      "4211385579" : 1,
      "4211388785" : 1,
      "4211371392" : 1,
      "4211406280" : 1,
      "4212498692" : 1,
      "4211405984" : 1,
      "4211406657" : 1,
      "4211407794" : 1,
      "4211397363" : 1,
      "4211403189" : 1,
      "4211406682" : 1,
      "4211407553" : 1,
      "4211405981" : 1,
      "4211405986" : 1,
      "4211407196" : 1,
      "4211407226" : 1,
      "4211406663" : 1,
      "4211407176" : 1,
      "4211397370" : 1,
      "4211397120" : 1,
      "4211377290" : 1,
      "4211408057" : 1,
      "4211406445" : 1,
      "4211407225" : 1,
      "4211399437" : 1,
      "4211378737" : 1,
      "4212498704" : 1,
      "4211388925" : 1,
      "4211394518" : 1,
      "4211406134" : 1,
      "4211387795" : 1,
      "4211406471" : 1,
      "4211406728" : 1,
      "4212498782" : 1,
      "4211407782" : 1,
      "4211387359" : 1,
      "4211399454" : 1,
      "4211405681" : 1,
      "4211399052" : 1,
      "4211405980" : 1,
      "4211373871" : 1,
      "4211405776" : 1,
      "4211407191" : 1,
      "4211397259" : 1,
      "4211406324" : 1,
      "4211406156" : 1,
      "4211406265" : 1,
      "4211405684" : 1,
      "4211407230" : 1,
      "4212498742" : 1,
      "4212498685" : 1,
      "4211407796" : 1,
      "4211406371" : 1,
      "4211377289" : 1,
      "4212498728" : 1,
      "4211405772" : 1,
      "4211406669" : 1,
      "4211406902" : 1,
      "4211399955" : 1,
      "4211399440" : 1,
      "4211405781" : 1,
      "4211407792" : 1,
      "4212498759" : 1,
      "4211406792" : 1,
      "4211406278" : 1,
      "4211407736" : 1,
      "4211406480" : 1,
      "4211401123" : 1,
      "4211405841" : 1,
      "4211407248" : 1,
      "4211406170" : 1,
      "4211399459" : 1,
      "4211399924" : 1,
      "4212498780" : 1,
      "4211399078" : 1,
      "4211371517" : 1,
      "4211405685" : 1,
      "4211405846" : 1,
      "4211407558" : 1,
      "4211399428" : 1,
      "4211406377" : 1,
      "4211406630" : 1,
      "4211373339" : 1,
      "4211387574" : 1,
      "4211405906" : 1,
      "4212498778" : 1,
      "4211397010" : 1,
      "4211408055" : 1,
      "4211399463" : 1,
      "4211406890" : 1,
      "4212498681" : 1,
      "4211407713" : 1,
      "4211394087" : 1,
      "4212498793" : 1,
      "4211397260" : 1,
      "4211407181" : 1,
      "4211399069" : 1,
      "4211397008" : 1,
      "4211388098" : 1,
      "4211405702" : 1,
      "4211397698" : 1,
      "4212498737" : 1,
      "4211397685" : 1,
      "4212498750" : 1,
      "4211407170" : 1,
      "4211406472" : 1,
      "4212498724" : 1,
      "4211407255" : 1,
      "4211397701" : 1,
      "4211407874" : 1,
      "4211399708" : 1,
      "4211405775" : 1,
      "4212498776" : 1,
      "4211405855" : 1,
      "4211397197" : 1,
      "4211406006" : 1,
      "4211407562" : 1,
      "4211374848" : 1,
      "4211385153" : 1,
      "4211406000" : 1,
      "4211385287" : 1,
      "4212498754" : 1,
      "4211406130" : 1,
      "4211407805" : 1,
      "4211374738" : 1,
      "4211406129" : 1,
      "4212498752" : 1,
      "4211405979" : 1,
      "4211406894" : 1,
      "4211405770" : 1,
      "4211406127" : 1,
      "4211406164" : 1,
      "4211406665" : 1,
      "4211407738" : 1,
      "4211397727" : 1,
      "4211407737" : 1,
      "4211406533" : 1,
      "4211388502" : 1,
      "4211406341" : 1,
      "4211408054" : 1,
      "4211405824" : 1,
      "4211405852" : 1,
      "4211406537" : 1,
      "4211397369" : 1,
      "4211399275" : 1,
      "4211407722" : 1,
      "4211385484" : 1,
      "4211406585" : 1,
      "4211373776" : 1,
      "4211385684" : 1,
      "4211406539" : 1,
      "4211397706" : 1,
      "4211397414" : 1,
      "4211407721" : 1,
      "4211407260" : 1,
      "4211406898" : 1,
      "4211406475" : 1,
      "4211399115" : 1,
      "4212498705" : 1,
      "4211400811" : 1,
      "4211406009" : 1,
      "4211385185" : 1,
      "4211407875" : 1,
      "4211405836" : 1,
      "4211388830" : 1,
      "4211399050" : 1,
      "4211405909" : 1,
      "4211406348" : 1,
      "4211406538" : 1,
      "4211406021" : 1,
      "4211397199" : 1,
      "4211406138" : 1,
      "4211407267" : 1,
      "4211407240" : 1,
      "4211405835" : 1,
      "4211397147" : 1,
      "4211389692" : 1,
      "4211397713" : 1,
      "4212498786" : 1,
      "4211407146" : 1,
      "4211406137" : 1,
      "4211405786" : 1,
      "4211408056" : 1,
      "4211405683" : 1,
      "4211406318" : 1,
      "4211385174" : 1,
      "4211399080" : 1,
      "4211406263" : 1,
      "4211406724" : 1,
      "4212498690" : 1,
      "4211371632" : 1,
      "4212848160" : 1,
      "4211405678" : 1,
      "4211405783" : 1,
      "4211413189" : 1,
      "4211374912" : 1,
      "4211399455" : 1,
      "4212498676" : 1,
      "4211405787" : 1,
      "4211407231" : 1,
      "4211406469" : 1,
      "4211405766" : 1,
      "4211388788" : 1,
      "4211406323" : 1,
      "4211399049" : 1,
      "4211406907" : 1,
      "4212498725" : 1,
      "4211371938" : 1,
      "4211407555" : 1,
      "4211387917" : 1,
      "4211399057" : 1,
      "4211405970" : 1,
      "4211407175" : 1,
      "4211398912" : 1,
      "4211387640" : 1,
      "4211407356" : 1,
      "4211385179" : 1,
      "4211397562" : 1,
      "4211406340" : 1,
      "4212498747" : 1,
      "4211399079" : 1,
      "4211399007" : 1,
      "4211405908" : 1,
      "4211399430" : 1,
      "4212498779" : 1,
      "4211405857" : 1,
      "4211406897" : 1,
      "4211407256" : 1,
      "4211406473" : 1,
      "4211389096" : 1,
      "4211407781" : 1,
      "4211407358" : 1,
      "4211378262" : 1,
      "4211407166" : 1,
      "4211393257" : 1,
      "4212498758" : 1,
      "4211405756" : 1,
      "4211407717" : 1,
      "4211385177" : 1,
      "4211399351" : 1,
      "4211406525" : 1,
      "4212498680" : 1,
      "4211371592" : 1,
      "4211406648" : 1,
      "4211395235" : 1,
      "4212498678" : 1,
      "4211407147" : 1,
      "4212498708" : 1,
      "4211406892" : 1,
      "4211387361" : 1,
      "4212498790" : 1
   },
   "hierarchical_multiplier_id" : {
      "103448630" : 1,
      "103446284" : 1,
      "103451284" : 1,
      "103444974" : 1,
      "103450004" : 1,
      "103446286" : 1,
      "103447249" : 1,
      "103450334" : 1,
      "103446323" : 1,
      "103446906" : 1,
      "103450324" : 1,
      "103447142" : 1,
      "103447859" : 1,
      "103447471" : 1,
      "103446321" : 1,
      "103447166" : 1,
      "103445955" : 1,
      "103446600" : 1,
      "103447770" : 1,
      "103449298" : 1,
      "103446322" : 1,
      "103447534" : 1,
      "103448243" : 1,
      "103446860" : 1,
      "103446276" : 1,
      "103447504" : 1,
      "103447421" : 1,
      "103447999" : 1,
      "103445420" : 1,
      "103447713" : 1,
      "103450123" : 1,
      "103447378" : 1,
      "103448690" : 1,
      "103448677" : 1,
      "103447836" : 1,
      "103447536" : 1,
      "103447073" : 1,
      "103447656" : 1,
      "103448626" : 1,
      "103446790" : 1,
      "103446436" : 1,
      "103445778" : 1,
      "103448530" : 1,
      "103444744" : 1,
      "103446603" : 1,
      "103449259" : 1,
      "103448409" : 1,
      "103447749" : 1,
      "103447638" : 1,
      "103450413" : 1,
      "103446432" : 1,
      "103447675" : 1,
      "103448789" : 1,
      "103447627" : 1,
      "103445966" : 1,
      "103445947" : 1,
      "103447763" : 1,
      "103446681" : 1,
      "103446832" : 1,
      "103447300" : 1,
      "103445154" : 1,
      "103446836" : 1,
      "103445999" : 1,
      "103448606" : 1,
      "103448602" : 1,
      "103447550" : 1,
      "103450339" : 1,
      "103446087" : 1,
      "103448781" : 1,
      "103448312" : 1,
      "103447362" : 1,
      "103445951" : 1,
      "103446083" : 1,
      "103448137" : 1,
      "103447444" : 1,
      "103447539" : 1
   }
}
