#!/usr/bin/perl

use my_inc "../..";

=head1 DESCRIPTION

    Исправляет список доменов, на которых клиент не хочет показываться (проблемные списки были обнаружены в DIRECT-56087).
    Скрипт меняет поле DontShow у проблемных кампаний и устанавливает statusBsSynced в No.

    LOG_TEE=1 ./protected/one-shot/fix_incorrect_campaigns-dontshow.pl

    Параметры:

        only-shards   -- список шардов, на которых запускать (через запятую)
        chunk-size    -- сколько кампаний обрабатывать за раз (10 000 000 по умолчанию)
        dry-run       -- запуск без внесения изменений в БД

    Скрипт работает параллельно с каждым шардом. Можно перезапускать.

=cut

use Direct::Modern;
use open ':std' => ':utf8';

use List::Util qw/min/;
use List::MoreUtils qw/uniq/;;

use ScriptHelper;

use Yandex::DBTools;
use Yandex::DBShards;
use ShardingTools qw/ppc_shards/;
use Settings;
use Direct::Validation::Domains qw/validate_disabled_domains/;

my ($ONLY_SHARDS, $CHUNK_SIZE, $DRY_RUN);
extract_script_params(
    'only-shards:s' => \$ONLY_SHARDS,
    'chunk-size:i'  => \$CHUNK_SIZE,
    'dry-run'       => \$DRY_RUN,
);

$CHUNK_SIZE ||= 10_000_000;
die "No valid chunk-size specified" if $CHUNK_SIZE < 1;

my @cond_bad_dontshow = (DontShow__rlike => '[[:space:]]+');

$log->out( 'START' );

my @shards = defined $ONLY_SHARDS ? grep { /^\d+$/ } (split /,/, $ONLY_SHARDS) : ppc_shards();
die "No valid shards specified" unless @shards;
$log->out( "Running on shards: ".join(',', @shards) );

my $script_name = get_script_name();
my $shard_results = foreach_shard_parallel(shard => \@shards, sub {
        my ($shard) = @_;

        my $log_shard = Yandex::Log->new(
            log_file_name => $script_name.".shard_${shard}.log",
            date_suf      => '%Y%m%d',
            msg_prefix    => "[shard:$shard]",
        );
        $log_shard->out( 'START' );

        my $ok = eval {
            my $max_cid = get_one_field_sql(PPC(shard => $shard), "select max(cid) from campaigns");
            return 1 unless $max_cid;

            for (my $last_cid = 1; $last_cid < $max_cid; $last_cid += $CHUNK_SIZE + 1) {
                my %cid2dontshow = %{ get_hash_sql(PPC(shard => $shard), ['SELECT cid, DontShow FROM campaigns', WHERE => { cid__between => [$last_cid, min($last_cid + $CHUNK_SIZE, $max_cid)], @cond_bad_dontshow, }, ]) };
                next unless %cid2dontshow;

                while (my ($cid, $dontshow) = each %cid2dontshow) {
                    my (@original_valid_domains, @original_other_domains);
                    foreach my $by_comma (split /\s*,\s*/, $dontshow) {
                        if ($by_comma !~ /\s/ && validate_disabled_domains([ $by_comma ], skip_ssp => 1)->is_valid) {
                            push @original_valid_domains, $by_comma;
                            next;
                        }
                        push @original_other_domains, grep {validate_disabled_domains([ $_ ], skip_ssp => 1)->is_valid} split /\s+/, $by_comma;
                    }
                    # на случай если были дубли - почистим их
                    @original_valid_domains = uniq @original_valid_domains;
                    @original_other_domains = uniq @original_other_domains;

                    my @result_domains;
                    if (scalar @original_valid_domains >= $Direct::Validation::Domains::DONT_SHOW_LIMIT) {
                        # Если так получилось, что изначально валидных доменов больше или равно лимиту, то остальные домены не рассматриваем, для них все равно нет места
                        # т.к. в рамках данного тикета мы исправляем только косяки в записи доменов, а не их кол-во, то записываем в базу сколько бы их не было
                        @result_domains = @original_valid_domains;
                    } else {
                        # Если место есть, то смотрим сколько его
                        my $available_space = $Direct::Validation::Domains::DONT_SHOW_LIMIT - scalar @original_valid_domains;
                        # и возьмем только то, на что хватает места
                        push @result_domains, uniq(@original_valid_domains, @original_other_domains[0 .. min(--$available_space, $#original_other_domains)]);
                    }
                    $cid2dontshow{$cid} = @result_domains ? join(',', @result_domains) : '';

                    $log_shard->out( "CID $cid. BEFORE: '$dontshow'" );
                    $log_shard->out( "CID $cid. AFTER: '$cid2dontshow{$cid}'" );
                }

                $log_shard->out( "Progress CID. Current / Max: $last_cid / $max_cid" );
                unless ($DRY_RUN) {
                    my $rows_affected = do_update_table(
                        PPC(shard => $shard),
                        'campaigns',
                        {
                            DontShow__dont_quote => sql_case(cid => \%cid2dontshow, default__dont_quote => '`DontShow`'),
                            statusBsSynced       => 'No',
                        },
                        where => { cid__in => [ keys %cid2dontshow ], @cond_bad_dontshow, },
                    );
                    $log_shard->out( "ROWS AFFECTED: $rows_affected" );
                } else {
                    $log_shard->out( "DRY RUN" );
                }
            }

            1;
        };

        unless ($ok) {
            $log_shard->out( "Error: $@" );
        }

        $log_shard->out( 'FINISH' );
    });
$log->out( "Per-shard results:", $shard_results );

$log->out( 'FINISH' );
