#!/usr/bin/perl
use Direct::Modern;

=head1 NAME

migrate-mobile-apps.pl

=head1 SYNOPSIS

    # параллелится по шардам и мигрирует всех
    protected/one-shot/migrate-mobile-apps.pl

    # мигрирует одного
    LOG_TEE=1 protected/one-shot/migrate-mobile-apps.pl --ClientID 7325591 --max-unique-mobile-content-ids 2

    # мигрирует всех клиентов на шарде
    LOG_TEE=1 protected/one-shot/migrate-mobile-apps.pl --shard 1 --max-unique-mobile-content-ids 2

=head1 DESCRIPTION

Мигрирует данные РМП, чтобы для всех РМП-кампаний назначить campaigns_mobile_content.mobile_app_id.
Нужные записи в mobile_apps и mobile_app_trackers при этом создаются на основе данных в mobile_content.
Кроме того, создаются записи в mobile_apps для всех записей в mobile_content, mobile_content_id которых
используются хотя бы в одной группе.

Считает, сколько максимум у клиента в одной кампании используется разных mobile_content_id в группах.
На этот максимум параметрами min-unique-mobile-content-ids и max-unique-mobile-content-ids можно
поставить ограничения.

Механика работы: ищет клиентов и учитывает ограничения по количеству разных mobile_content_id скрипт сам,
а для собственно миграции дёргает ручку mobile_app/migrateClient в java-intapi на каждого нужного клиента
по отдельности. Детальнее, как делается миграция одного клиента, можно почитать
в ru.yandex.direct.core.entity.client.service.ClientMobileContentCampaignMigrationService.

=cut

use List::Util 'max';
use Try::Tiny;

use Yandex::DBShards;
use Yandex::DBTools;

use my_inc '../..';

use JavaIntapi::MigrateMobileApps;
use ScriptHelper 'Yandex::Log' => 'messages';
use Settings;
use ShardingTools;

my $opt_shard;
my $opt_ClientID;
my $opt_min_unique_mobile_content_ids;
my $opt_max_unique_mobile_content_ids;
my $opt_usage;
extract_script_params(
    'shard=i' => \$opt_shard,
    'ClientID=i' => \$opt_ClientID,
    'min-unique-mobile-content-ids=i' => \$opt_min_unique_mobile_content_ids,
    'max-unique-mobile-content-ids=i' => \$opt_max_unique_mobile_content_ids,
    'help|usage' => \$opt_usage,
);

if ($opt_usage) {
    usage();
}

if ( $opt_shard && $opt_ClientID ) {
    die '--shard and --ClientID are mutually exclusive';
}

$log->out('start');

if ($opt_ClientID) {
    my $client_shard = get_shard( ClientID => $opt_ClientID );
    die "shard not found for ClientID=$opt_ClientID" unless $client_shard;

    process_client( $client_shard, $opt_ClientID );
} elsif ($opt_shard) {
    process_shard($opt_shard);
} else {
    process_all_shards();
}

$log->out('finish');

exit 0;

sub process_client {
    my ( $shard, $ClientID ) = @_;

    my $prefix_guard = $log->msg_prefix_guard("shard=$shard, ClientID=$ClientID");

    my $mobile_content_count_by_cid = get_all_sql( PPC( shard => $shard ), [
        'SELECT c.cid, COUNT(DISTINCT amc.mobile_content_id) AS mobile_content_count',
        'FROM adgroups_mobile_content amc',
        'INNER JOIN phrases p ON amc.pid = p.pid',
        'INNER JOIN campaigns c ON p.cid = c.cid',
        WHERE => { 'c.ClientID' => $ClientID },
        'GROUP BY c.cid',
    ] );
    $log->out( { mobile_content_count_by_cid => $mobile_content_count_by_cid } );

    my $max_unique_mobile_content_count;
    if (@$mobile_content_count_by_cid) {
        $max_unique_mobile_content_count = max map { $_->{mobile_content_count} } @$mobile_content_count_by_cid;
    } else {
        $max_unique_mobile_content_count = 0;
    }

    $log->out("max_unique_mobile_content_count=$max_unique_mobile_content_count");

    if ( defined $opt_min_unique_mobile_content_ids && $max_unique_mobile_content_count < $opt_min_unique_mobile_content_ids ) {
        $log->out("skipping: opt_min_unique_mobile_content_ids=$opt_min_unique_mobile_content_ids");
        return;
    }

    if ( defined $opt_max_unique_mobile_content_ids && $max_unique_mobile_content_count > $opt_max_unique_mobile_content_ids ) {
        $log->out("skipping: opt_max_unique_mobile_content_ids=$opt_max_unique_mobile_content_ids");
        return;
    }

    try {
        my $migration_result = JavaIntapi::MigrateMobileApps->new( client_id => $ClientID )->call;
        $log->out("migration_result=$migration_result");
    } catch {
        $log->out("MIGRATION ERROR: $_");
    };
}

sub process_shard {
    my ($shard) = @_;

    my $prefix_guard = $log->msg_prefix_guard("shard=$shard");

    $log->out('start');

    my $ClientIDs = get_one_column_sql( PPC( shard => $shard ), [
        'SELECT DISTINCT c.ClientID',
        'FROM adgroups_mobile_content amc',
        'INNER JOIN phrases p ON amc.pid = p.pid',
        'INNER JOIN campaigns c ON p.cid = c.cid',
    ] );

    $log->out( 'number of clients: ' . scalar(@$ClientIDs) );
    for my $ClientID (@$ClientIDs) {
        process_client( $shard, $ClientID );
    }

    $log->out('finish');
}

sub process_all_shards {
    foreach_shard_parallel shard => [ ppc_shards() ], sub {
        my ($shard) = @_;
        process_shard($shard);
    }
}
