#!/usr/bin/perl

use my_inc '../..';

=head2 NAME

    Скрипт для добавления Крыма в те таргетинги, в которых он входит, но не указан явно
        (например СНГ или Украина).
    Работает параллельно над всеми шардами при запуске без параметров.
    Можно прибивать и начинать работу с обработанного места, указав шард и начальный номер группы.

=head2 DESCRIPTION

    Параметры командной строки:
        --shard-id=NNN  номер шарда, с которым требуется работать. По умолчанию - все.
        --last-key=XXX  последний обработанный ключ (pk таблицы). По умолчанию - cid
        --last-val=MMM  значение последнего обработанного первичного ключа. По умолчанию - 0
        --max-val=YYYY  максимальное значение ключа, которое нужно обрабатывать.
                            Работает только в сочетании с --last-key, и только эта таблица будет обработана
        --sleep-coeff=N коэффициент "сна" после запросов на обновление БД
        --remove-only   параметр, при указании которого не требуется добавлять Крым к таргетингам,
                            где он не указан явно, но подразумевался. Предназначен для "второго прогона" миграции

=head2 COMMENT

    Возможные варианты запуска скрипта в хронологическом порядке:
        1 раз - без параметров
        2 - перезапуски при падениях - требуется указание
            --shard-id NNN --last-key XXX --last-val ZZZ
        ...
        n - второй проход (только удаление -Крыма) - с параметром 
            --remove-only
        n+1 - с параметрами 
            --remove-only --shard-id

=cut

use warnings;
use strict;
use utf8;

use List::MoreUtils qw/all/;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ListUtils;
use Yandex::Log;
use Yandex::Retry;
use Yandex::TimeCommon;
use Yandex::Validate;

use Settings;

use BS::ResyncQueue;
use GeoTools;
use LockTools;
use Property;
use ScriptHelper 'Yandex::Log' => [log_file_name => "modify_krim_in_geo.log", date_suf => '%Y%m%d', auto_rotate => 1, lock => 1, tee => $ENV{LOG_TEE}],
                 get_file_lock => undef;
use ShardingTools;

use constant RESYNC_PRIORITY => 4;
use constant SELECT_CHUNK_SIZE => 250_000;
use constant UPDATE_CHUNK_SIZE => 5_000;
use open ":std" => ":utf8";


my @SHARDS;
my $ONE_SHARD;
my $LAST_KEY;
my $LAST_VAL;
my $PARAMS_MAX_VAL;
my $SLEEP_COEFF = 3;
# если неопределено - это первая выкладка, считаем что все существующие в базе таргетинги
# по украинскому дереву, добавляем +Крым, удаляем -Крым
# если определено - вторая выкладка, считаем что таргетинги уже по АПИ-дереву
# и нужно только удалить -Крым
my $REMOVE_ONLY;
extract_script_params(
    'shard-id:i' => \$ONE_SHARD,
    'last-key:s' => \$LAST_KEY,
    'last-val:i' => \$LAST_VAL,
    'max-val:i' => \$PARAMS_MAX_VAL,
    'sleep-coeff:i' => \$SLEEP_COEFF,
    'remove-only+' => \$REMOVE_ONLY,
);

if (defined $ONE_SHARD) {
    die "invalid shard number specified: $ONE_SHARD" unless is_valid_int($ONE_SHARD, 1, $Settings::SHARDS_NUM);
    @SHARDS = ($ONE_SHARD);
} else {
    @SHARDS = ppc_shards();
}

$log->out('Preparing');
$log->out("Working on shards: @SHARDS");

my $prop = Property->new('modify_krim_in_geo');
my $prop_val = $prop->get();
my $prop_val_from_db = $prop_val;
if (!defined $prop_val
    && !$LAST_KEY && !$LAST_VAL
    && scalar(@SHARDS) == $Settings::SHARDS_NUM
    && !$REMOVE_ONLY
) {
    # сохраняем время первого запуска миграции с параметрами по умолчанию
    $prop_val = unix2mysql(time);
    $log->out("Saving migration start time in 'modify_krim_in_geo' property: $prop_val");
    $prop->set($prop_val);
}

if (defined $PARAMS_MAX_VAL) {
    die '--max-val allowed only with --last-key' unless $LAST_KEY;
    die '--max-val has invalid value. should be positive integer' unless is_valid_int($PARAMS_MAX_VAL, 0);
    die '--max-val should be greather than --last-val' unless $PARAMS_MAX_VAL > $LAST_VAL;
}

if (defined $prop_val_from_db && !$REMOVE_ONLY) {
    # защита от двойного запуска первого прохода (с добавлением) без параметров
    if (!$ONE_SHARD || !$LAST_KEY || !defined $LAST_VAL) {
        $log->die("You must specify parameters --shard-id Y --last-key WWW --last-val ZZZ to run migration again!");
    }
    my @good_keys = qw/cid pid mbid mgid/;
    if (!grep {$LAST_KEY eq $_} @good_keys) {
        die "last-key has invalid value. available values are: @good_keys";
    }
    if (!is_valid_int($LAST_VAL, 0)) {
        die "last-val has invalid value. should be positive integer";
    }
    # увеличиваем на 1, так как было указано последнее потенциально обработанное значение
    $LAST_VAL++;
} else {
    # игнорируем ввод параметров
    $LAST_KEY = 'cid';
    $LAST_VAL = 0;

    die '--max-val is not available at this moment' if defined $PARAMS_MAX_VAL;
}

my $start_time = time;
if ($REMOVE_ONLY) {
    die 'No lastchange data - run migration without --second-pass first' unless $prop_val;
    $log->out("Second migration pass, removing -Krim");
} else {
    $log->out('First migration pass, -Krim will be removed, Krim will be added');
}

if (defined $prop_val && $prop_val eq '20140917151737') {
    # первый запуск с --remove-only ничего полезного не сделал, используем значение, которое было до этоого
    # взято из логов
    $prop_val = '20140915181747';
}

$log->out('START');
my $translocal_georeg = GeoTools::get_translocal_georeg({tree => 'ua'});
local $SIG{__DIE__} = sub { $log->out('DIED!'); $log->die(@_) };
my $res = foreach_shard_parallel shard => \@SHARDS, sub {
    my $shard = shift;
    my $shard_log = Yandex::Log->new(
        log_file_name => "modify_krim_in_geo_shard_$shard.log",
        date_suf      => '%Y%m%d',
        auto_rotate   => 1,
        lock          => 1,
    );

    # лочимся
    get_file_lock(0, "modify_krim_in_geo_shard_$shard");

    my %geo_cache;

    $log->out("START processing shard $shard");
    if (!$REMOVE_ONLY) {

        # обрабатываем geo в campaigns
        if ($LAST_KEY eq 'cid') {
            _process_table(\%geo_cache, $shard, 'campaigns', $LAST_KEY, $LAST_VAL, $shard_log, $PARAMS_MAX_VAL);
            ($LAST_KEY, $LAST_VAL) = ('pid', 0) unless defined $PARAMS_MAX_VAL;
        }

        # обрабатываем geo во phrases
        if ($LAST_KEY eq 'pid') {
            _process_table(\%geo_cache, $shard, 'phrases', $LAST_KEY, $LAST_VAL, $shard_log, $PARAMS_MAX_VAL);
            ($LAST_KEY, $LAST_VAL) = ('mbid', 0) unless defined $PARAMS_MAX_VAL;
        }

        # обрабатываем geo в mediaplan_banners
        if ($LAST_KEY eq 'mbid') {
            _process_table(\%geo_cache, $shard, 'mediaplan_banners', $LAST_KEY, $LAST_VAL, $shard_log, $PARAMS_MAX_VAL);
            ($LAST_KEY, $LAST_VAL) = ('mgid', 0) unless defined $PARAMS_MAX_VAL;
        }

        # обрабатываем geo в media_groups
        if ($LAST_KEY eq 'mgid') {
            _process_table(\%geo_cache, $shard, 'media_groups', $LAST_KEY, $LAST_VAL, $shard_log, $PARAMS_MAX_VAL);
        }

        $log->out("info for shard $shard: [ finished ]");
    
    } else {
        # Второй проход - другая логика
        # Обрабатываем те группы, которые изменились с момента прошлого запуска
        $log->out("Processing phrases in shard $shard");
        $shard_log->out("Working for phrases with LastChange > $prop_val");

        my $fetched_rows_cnt;
        my $max_pid = 0;
        do {
            my %geo_for_update;

            $shard_log->out("Processing phrases: LastChange > $prop_val OR LastChange = $prop_val AND pid > $max_pid");
            $fetched_rows_cnt = 0;
            my $sth = exec_sql(PPC(shard => $shard),
                'SELECT pid, cid, geo, LastChange FROM phrases WHERE LastChange > ? OR (LastChange = ? AND pid > ?) ORDER BY LastChange, pid ASC LIMIT ?',
                $prop_val, $prop_val, $max_pid, SELECT_CHUNK_SIZE
            );
            while (my ($pid, $cid, $geo, $LastChange) = $sth->fetchrow_array()) {
                ++$fetched_rows_cnt;
                ($prop_val, $max_pid) = ($LastChange, $pid);
                if (!defined $geo || $geo eq '' || $geo eq '0') {
                    # неопределенное или пустое гео = весь мир - ничего делать не требуется
                    next;
                }
                my $new_geo = $geo_cache{$geo} //= _check_modify($geo, pid => $pid, $shard_log);
                if ($new_geo ne 'skip') {
                    $geo_for_update{$geo}->{new} //= $new_geo;
                    push @{ $geo_for_update{$geo}->{data} }, {pid => $pid, cid => $cid, priority => RESYNC_PRIORITY};
                }
            }
            $sth->finish();
            _update_data(\%geo_for_update, $shard, 'phrases', 'pid', $shard_log);

        } until ($fetched_rows_cnt < SELECT_CHUNK_SIZE);
        $shard_log->out("Processing phrases finished in this shard");
    }

    $log->out("FINISH processing shard $shard");
};

if ($REMOVE_ONLY){
    $prop_val = unix2mysql($start_time);
    $log->out("Saving migration start time in 'modify_krim_in_geo' property: $prop_val");
    $prop->set($prop_val);
}
if (all {all {$_} @$_} values %$res) {
    $log->out('FINISH');
    exit 0;
} else {
    $log->die('There were errors in shards! check shard logs for details');
    exit 1;
}


=head2 _update_data

    Обновляет данные в базе. Принимает следующие параметры
        ссылку на хеш с данными {'старое гео' => { data => {pid => , cid =>, priority => }, new => 'новое гео' }
        номер шарда
        название таблицы
        название первичного ключа
        объект Yandex::Log для логирования

=cut
sub _update_data($$$$$){
    my ($geo_for_update, $shard, $table_name, $pk_name, $logger) = @_;

    while (my ($old_geo, $data) = each %$geo_for_update) {
        for my $chunk (chunks($data->{data}, UPDATE_CHUNK_SIZE)) {
            my @ids = map { $_->{$pk_name} } @$chunk;
            $logger->out({new_geo => $data->{new}, old_geo => $old_geo, $pk_name => \@ids });

            my ($affected, $resync) = (0, 0);
            relaxed times => $SLEEP_COEFF, sub {
                $affected = do_update_table(PPC(shard => $shard), $table_name,
                                            {geo => $data->{new}, ($table_name eq 'media_groups' ? (statusBsSynced => 'No') : ())},
                                            where => {geo => $old_geo, $pk_name => \@ids}
                );
                if ($table_name eq 'phrases') {
                    $resync = bs_resync($chunk);
                }
            };
            my $expected = scalar @ids;
            $logger->out("Updated $table_name - affected $affected rows, expected: $expected. Added $resync items to bs_resync_queue");
        }
    }
}

=head2 _check_modify

    Проверяет, нужно ли изменить что-то в таргетинге всвязи с Крымом.
    Если не нужно, возвращает ноль.
    Если нужно - возвращает исходное гео с добавленым или удаленным (для минус-регионов) Крымом

=cut
sub _check_modify {
    my ($geo, $type, $id, $flog) = @_;
    my $new_geo;

    # разбираем таргетинг
    my (%target_regions, %minus_regions, $targeted_in_krim);
    for my $r (split(/[^-+0-9]+/, $geo)) {
        next if $r eq '';
        next unless exists $translocal_georeg->{abs($r)};
        if ($r < 0) {
            $minus_regions{-$r} = undef;
        } else {
            $target_regions{+$r} = undef;
        }
    }
    for my $r ($geo_regions::KRIM, @{ $translocal_georeg->{ $geo_regions::KRIM }->{parents} }) {
        # если один из предков был отминусован - дальше так проверять нельзя.
        last if exists $minus_regions{$r};
        if (exists $target_regions{$r}) {
            $targeted_in_krim = 1;
            last;
        }
    }
    if (exists $target_regions{0} || exists $target_regions{ $geo_regions::KRIM }) {
        # если в таргетинге указан "весь мир" или "Крым", то ничего делать не нужно
    } elsif ($targeted_in_krim && !exists $target_regions{ $geo_regions::KRIM }) {
        # если Крым входит в таргетинг, но явно не указан (СНГ или Украина)
        # Добавляем Крым явно
        $new_geo = join(',', $geo, $geo_regions::KRIM) unless $REMOVE_ONLY;
    } elsif (exists $minus_regions{ $geo_regions::KRIM }) {
        # Если Крым отминусован - его нужно удалить
        $new_geo = ($geo =~ s/\s*-\s*$geo_regions::KRIM\b//r);
    } elsif (my @childs = grep {exists $minus_regions{$_} } @{ $translocal_georeg->{ $geo_regions::KRIM }->{childs} } ) {
        # возможен странный кейс, когда отминусованы дочерние регионы Крыма, но сам Крым в тарегтинге указан не был (например "СНГ кроме Ялта")
        # в базе таких значений нет, но обрабатывать их нужно
        $log->out("Strange geo: '$geo' with minus-regions: @childs");
        $new_geo = $geo;
        for my $c (@childs) {
            $new_geo =~ s/\s*-\s*$c\b//;
        }
    } else {
        # Ничего менять не нужно
    }

    if ($REMOVE_ONLY && !$new_geo && exists $minus_regions{ $geo_regions::KRIM } && !exists $target_regions{0}) {
        # Если Крым отминусован - его нужно удалить
        $new_geo = ($geo =~ s/\s*-\s*$geo_regions::KRIM\b//r);
    }

    my %errors;
    my $v = validate_geo(($new_geo // $geo), \%errors, {tree => "api"});
    if ($v) {
        $flog->out({$type => $id, geo => $geo, validation_error => \%errors});
        # хотим валидировать все гео (и те что уже поменяли), но заменять хотим только "не тронутые"
        $new_geo = $geo if !defined $new_geo;
    }
    if (defined $new_geo) {
        return refine_geoid($new_geo, undef, {tree => "api"});
    } else {
        return 'skip';
    }
}

=head2 _process_table

    Обработать гео в таблице

=cut
sub _process_table($$$$$$) {
    my ($geo_cache, $shard, $table_name, $pk_name, $min_pk_value, $shard_log, $max_pk_value) = @_;

    # обрабатываем geo в $table_name
    my $MAX_PK = get_one_field_sql(PPC(shard => $shard), "SELECT MAX($pk_name) FROM $table_name");
    $log->out("Processing $table_name in shard $shard");
    my $default_first_end_value = $min_pk_value + SELECT_CHUNK_SIZE - 1;
    my $first_chunk_end_value = $max_pk_value && $max_pk_value <= $default_first_end_value ? $max_pk_value
                                                                                           : $default_first_end_value;
    my $chunk_max = $max_pk_value ? $max_pk_value : $MAX_PK;
    $shard_log->out("Working for $pk_name: $min_pk_value - $chunk_max");
    for (my ($chunk_start, $chunk_end) = ($min_pk_value, $first_chunk_end_value);
         $chunk_start <= $chunk_max;
         ($chunk_start, $chunk_end) = ($chunk_end + 1, $chunk_end + SELECT_CHUNK_SIZE)
    ) {
        $log->out("info for shard $shard: [ --shard-id $shard --last-key $pk_name --last-val $chunk_end ]");
        $shard_log->out("processing chunk $pk_name: $chunk_start - $chunk_end");
        my %geo_for_update;

        my $sth = exec_sql(PPC(shard => $shard), "SELECT $pk_name, geo, cid FROM $table_name WHERE $pk_name BETWEEN ? AND ?", $chunk_start, $chunk_end);
        while (my ($pk_value, $geo, $cid) = $sth->fetchrow_array()) {
            if (!defined $geo || $geo eq '' || $geo eq '0') {
                # неопределенное или пустое гео = весь мир, добавлять Крым не требуется
                next;
            }
            my $new_geo = $geo_cache->{$geo} //= _check_modify($geo, $pk_name => $pk_value, $shard_log);
            if ($new_geo ne 'skip') {
                $geo_for_update{$geo}->{new} //= $new_geo;
                push @{ $geo_for_update{$geo}->{data} }, {$pk_name => $pk_value, cid => $cid, priority => RESYNC_PRIORITY};
            }
        }
        $sth->finish();
        _update_data(\%geo_for_update, $shard, $table_name, $pk_name, $shard_log);
    }
}
