#!/usr/bin/perl

=head1 DESCRIPTION

Переливка баннерных картинок в новую аватарницу

=cut


use Direct::Modern;
use my_inc '../..';

use ScriptHelper sharded => 1;

use Settings;
use Yandex::DBTools;
use Direct::Avatars;
use Yandex::HashUtils;
use Yandex::ListUtils qw/chunks/;
use JSON;
use BannerImages;
use BS::ResyncQueue qw/bs_resync/;

use Parallel::ForkManager;

my $MAX_PROCESSES = 5;
my $FORK_CHUNK_SIZE = 1000;
my $PUT_CHUNK_SIZE = 100;

local $Yandex::HTTP::DISABLE_STDERR_SPAM = 1;

$log->out('START');

extract_script_params(
    'chunk-size=i' => \$FORK_CHUNK_SIZE,
    'p|max-processes=i' => \$MAX_PROCESSES,
    'hash=s' => \my @hashes,
    'hash-from=s' => \my $hash_from,
    'hash-to=s' => \my $hash_to,
);


my $avatars = Direct::Avatars->new(namespace => 'direct');
my $pm = Parallel::ForkManager->new($MAX_PROCESSES);


my $start_hash = $hash_from || '';
while (my $chunk = _get_chunk($SHARD, $start_hash)) {
    if (my $reason = smart_check_stop_file()) {
        $log->out("$reason! Let's finish.");
        last;
    }

    $start_hash = $chunk->[-1]->{image_hash};
    my $pid = $pm->start and next;

    local $log->{msg_prefix} = "[$$]";
    $log->out(sprintf "child started with %d images", scalar @$chunk);
    _process_chunk($SHARD, $chunk);
    $log->out("child finished");
    $pm->finish;
}

$pm->wait_all_children;

$log->out('FINISH');


sub _process_chunk {
    my ($shard, $big_chunk) = @_;

    for my $chunk (chunks $big_chunk, $PUT_CHUNK_SIZE) {
        my %image_info;
        my %request;
        for my $image_info (@$chunk) {
            my $image_hash = $image_info->{image_hash};
            $image_info{$image_hash} = $image_info;

            my $src_url = $avatars->get_avatars_link($image_info);
            $request{$image_hash} = {url => $src_url};
        }

        $log->out(sprintf 'Start uploading set of %d images', scalar @$chunk);
        my $multi_result = $avatars->put_multi(\%request);

        RESULT:
        for my $image_hash (sort keys %$multi_result) {
            my $put_result = $multi_result->{$image_hash};
            my $image_info = $image_info{$image_hash};
            my $src_url = $request{$image_hash}->{url};
            if (!$put_result->{'group-id'}) {
                my $description = $put_result->{description} // '<no description>';
                $log->out("ERROR: failed to upload image $src_url: $description");
                next RESULT;
            }

            my $update_data = {
                mds_group_id => $put_result->{"group-id"},
                avatars_host => $put_result->{avatars_host},
            };

            my $orig = hash_cut $image_info, qw/width height/;
            my $old_format = hash_merge {orig => $orig}, decode_json($image_info->{formats});
            my $has_format_errors;
            for my $format (keys %$old_format) {
                my $old_size = $old_format->{$format};
                my $new_size = $put_result->{sizes}->{$format};
                if (!$new_size) {
                    $log->out("WARN: image $image_hash doesn't have /$format");
                    $has_format_errors = 1;
                }

                my $old_format_str = "$old_size->{width}x$old_size->{height}";
                my $new_format_str = "$new_size->{width}x$new_size->{height}";
                if ($old_format_str ne $new_format_str) {
                    $log->out("WARN: image $image_hash format /$format differs: $old_format_str -> $new_format_str");
                    $has_format_errors = 1;
                }
            }

            if ($has_format_errors) {
                my %new_sizes =
                    map {($_ => hash_cut($put_result->{sizes}->{$_}, qw/height width/))}
                    grep {exists $put_result->{sizes}->{$_}}
                    keys %$old_format;
                my $is_ok = eval {BannerImages::validate_avatars_sizes($image_info->{image_type} => \%new_sizes); 1};
                if (!$is_ok) {
                    my ($error_msg) = $@ =~ m#^(.*?)\sat\s#xms;
                    $log->out("ERROR: image $image_hash has problems: $error_msg / $src_url");
                    next RESULT;
                }
                $update_data->{formats} = encode_json \%new_sizes;
                $log->out("image $image_hash: updating formats $update_data->{formats}");
            }

            do_update_table(PPC(shard => $shard), banner_images_formats => $update_data,
                where => {image_hash => $image_hash},
            );

            $log->out("image $image_hash moved to mds_group_id=$update_data->{mds_group_id}");

            if (my $banners = $image_info->{banners}) {
                bs_resync($banners);
                $log->out(sprintf '%d banners resynced', scalar @$banners);
            }
        }
    }

    return;
}


sub _get_chunk {
    my ($shard, $shart_hash) = @_;

    my $chunk = get_all_sql(PPC(shard => $shard), [
            'SELECT bif.*
            FROM banner_images_formats bif
            ',
            WHERE => {
                image_hash__gt => $shart_hash,
                mds_group_id__is_null => 1,
                (@hashes ? (image_hash => @hashes) : ()),
                ($hash_to ? (image_hash__le => $hash_to) : ()),
            },
            'ORDER BY image_hash ASC',
            LIMIT => $FORK_CHUNK_SIZE,
        ]);
    return if !@$chunk;

    my $banners = get_all_sql(PPC(shard => $shard), [
            'SELECT image_hash, bid, pid, cid
            FROM banner_images bi
            JOIN banners b USING(bid)',
            WHERE => {
                image_hash => [map {$_->{image_hash}} @$chunk],
                statusArch => 'No',
            },
        ]);
    my %banners_by_hash;
    for my $banner (@$banners) {
        push @{$banners_by_hash{$banner->{image_hash}}}, hash_cut $banner, qw/bid pid cid/;
    }
    for my $image (@$chunk) {
        $image->{banners} = $banners_by_hash{$image->{image_hash}};
    }

    return $chunk;
}


