#!/usr/bin/perl

use my_inc '../..';

use warnings;
use strict;
use utf8;

use List::MoreUtils qw/uniq all/;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::ListUtils;
use Yandex::Retry;
use Yandex::Validate;

use Settings;
use ScriptHelper 'Yandex::Log' => [date_suf => "%Y%m%d", auto_rotate => 1, tee => $ENV{LOG_TEE}, use_syslog => 0];
use BS::ResyncQueue;
use ShardingTools;
use Sitelinks;

use constant SELECT_CHUNK_SIZE => 10_000;
use constant UPDATE_CHUNK_SIZE => 1_000;
use constant RESYNC_PRIORITY => -1;
use constant FLUSH_PROFILING_AFTER_ITERATION_CNT => 100;

$Yandex::DBShards::CACHE_MAX_SIZE =  4 * SELECT_CHUNK_SIZE;

my $SLEEP_COEFF = 1;
my $START_ID = 0;
my $END_ID;
my $DRY_RUN = 0;
extract_script_params(
    'sleep-coeff:i' => \$SLEEP_COEFF,
    'start-id:i' => \$START_ID,
    'end-id:i' => \$END_ID,
    'dry-run!' => \$DRY_RUN,
);

unless (is_valid_int($SLEEP_COEFF, 0)) {
    die "--sleep-coeff has invalid value. should be positive integer";
}
unless (is_valid_int($START_ID, 0)) {
    die "--start-id has invalid value. should be positive integer";
}
if (defined $END_ID && !is_valid_int($END_ID, 0)) {
    die "--end-id has invalid value. should be positive integer";
}

$log->out('START');
$log->out("Working from sitelinks_set_id: $START_ID");
my $MAX_SS_ID;
if (defined $END_ID) {
    $MAX_SS_ID = $END_ID;
} else {
    $MAX_SS_ID = get_one_field_sql(PPCDICT, 'SELECT MAX(sitelinks_set_id) FROM shard_inc_sitelinks_set_id');
}
$log->out("Working to sitelinks_set_id: $MAX_SS_ID");

my $iteration_num = 0;
for (my ($chunk_start, $chunk_end) = ($START_ID, $START_ID + SELECT_CHUNK_SIZE - 1);
     $chunk_start <= $MAX_SS_ID;
     ($chunk_start, $chunk_end) = ($chunk_end + 1, $chunk_end + SELECT_CHUNK_SIZE)
) {
relaxed times => $SLEEP_COEFF, sub {
    $log->out("processing sitelinks_set_id chunk: $chunk_start - $chunk_end");

    # хеш для хранения sitelinks_set_id с которыми все совсем плохо (например дублируются между шардами)
    my $global_skip = {};
    # сведения о сетах в метабазе. ключ - sitelinks_set_id, значение - хеш с ClientID, shard
    my $ss_metabase;
    # сведения о сетах в мастер-базе. ключ - sitelinks_set_id, значение - хеш с ClientID, shard
    my $ss_ppc;
    # сведения о связках сет-ссылки. таблица sitelinks_set_to_link, sitelinks_set_id => {shard =>, sl_id => [...]} 
    my $sstl_ppc;
    # сведения о баннерах, использующих сеты. ключ - sitelinks_set_id, значение хеш { шард => массив баннеров }
    my $banners;
    # хеш bid->ClientID
    my $bid2clientid = {};
    # хеш bid->cid (нужен для подготовки данных для bs_resync)
    my $bid2cid = {};

    ### Получаем текущее состояние ###
    # привязка сета к клиенту в метабазе
    my $tmp_set_id_2_shard_metabase = get_shard_multi(sitelinks_set_id => [$chunk_start .. $chunk_end]);
    while (my ($ss_id, $shard) = each %$tmp_set_id_2_shard_metabase) {
        next unless defined $shard;
        if ($shard) {
            $ss_metabase->{$ss_id}->{shard} = $shard;
        } else {
            # клиент в процессе решардинга
            $global_skip->{$ss_id} = 'resharding in progress';
        }
    }

    # привязка сета к шарду в метабазе
    my $tmp_sed_id_2_client_id_metabase = get_shard_multi(sitelinks_set_id => [$chunk_start .. $chunk_end], 'ClientID');
    while (my ($ss_id, $ClientID) = each %$tmp_sed_id_2_client_id_metabase) {
        next unless defined $ClientID;
        $ss_metabase->{$ss_id}->{ClientID} = $ClientID;   
    }

    for my $shard (ppc_shards()) {
        # привязки сета к шарду и клиенту в базе
        my $tmp_set_id_client_id_db = get_all_sql(PPC(shard => $shard),
                                'SELECT sitelinks_set_id, ClientID FROM sitelinks_sets WHERE sitelinks_set_id BETWEEN ? AND ?',
                                $chunk_start, $chunk_end
        );
        for my $row (@$tmp_set_id_client_id_db) {
            if (exists $ss_ppc->{ $row->{sitelinks_set_id} }) {
                # сет уже был найден в каком-либо шарде
                $global_skip->{ $row->{sitelinks_set_id} } = 'duplicated ss';
            } else {
                $ss_ppc->{ $row->{sitelinks_set_id} } = { ClientID => $row->{ClientID}, shard => $shard };
            }
        }

        # наличие связей "сет-линки" в шарде
        my $tmp_set_id_2_links_db = get_all_sql(PPC(shard => $shard),
                                 'SELECT sitelinks_set_id AS ss_id
                                       , GROUP_CONCAT(sl_id ORDER BY order_num ASC) AS sl_ids
                                    FROM sitelinks_set_to_link
                                   WHERE sitelinks_set_id BETWEEN ? AND ?
                                GROUP BY sitelinks_set_id',
                                $chunk_start, $chunk_end
        );
        for my $row (@$tmp_set_id_2_links_db) {
            if (exists $sstl_ppc->{ $row->{ss_id} }) {
                # сет уже был найден в каком-либо шарде
                $global_skip->{ $row->{ss_id} } = 'duplicated sstl';
            } else {
                $sstl_ppc->{ $row->{ss_id} } = { shard => $shard, sl_id => [split(',',  $row->{sl_ids} )]};
            }
        }

        # использование сетов в баннерах
        my $tmp_set_id_2_banners_db = get_all_sql(PPC(shard => $shard),
                                'SELECT b.sitelinks_set_id
                                      , b.bid
                                      , b.cid
                                      , u.ClientID
                                   FROM banners b
                                        LEFT JOIN campaigns c ON c.cid = b.cid
                                        LEFT JOIN users u ON u.uid = c.uid
                                  WHERE sitelinks_set_id BETWEEN ? AND ?',
                                $chunk_start, $chunk_end
        );
        for my $row (@$tmp_set_id_2_banners_db) {
            push @{ $banners->{ $row->{sitelinks_set_id} }->{$shard} }, $row->{bid};
            $bid2clientid->{ $row->{bid} } = $row->{ClientID};
            $bid2cid->{ $row->{bid} } = $row->{cid};
        }
    }

    # хеш шард => массив sitelinks_set_id для удаления из таблицы sitelinks_set_to_link
    my %sstl_to_delete_by_shard;
    # хеш для удаления из таблицы sitelinks_sets
    my %ss_to_delete_by_shard;
    # массив sitelinks_set_id для отложенного удаления из метабазы
    my @mb_to_delete;
    # данные для переотправки в БК
    my @resync_data;
    ### Разбираем данные и что-то чиним ###
    SS_ID: for my $ss_id ($chunk_start .. $chunk_end) {
        if (exists $global_skip->{$ss_id}) {
            # сет или связи сет-линки дублируются между шардами
            $log->out({
                action => sprintf('Skip set (warn) - %s', $global_skip->{$ss_id}),
                sitelinks_set_id => $ss_id
            });
            next SS_ID;

        } elsif (!defined $ss_metabase->{$ss_id}
                 && !defined $ss_ppc->{$ss_id}
                 && !defined $sstl_ppc->{$ss_id}
                 && !exists $banners->{$ss_id}
        ) {
            # сет не существует, пропускаем.
            $log->out({
                action => "Skip set - doesn't exists",
                sitelinks_set_id => $ss_id
            });
            next SS_ID;

        } elsif (!defined $ss_metabase->{$ss_id}
                 && !defined $ss_ppc->{$ss_id}
                 && defined $sstl_ppc->{$ss_id}
                 && !exists $banners->{$ss_id}
        ) {
            # самого сета не существует, но есть связи сет-линки - удалим их (их уже нельзя переиспользовать)
            $log->out({
                action => 'set will be deleted from sitelinks_set_to_link (only sstl defined)',
                sitelinks_set_id => $ss_id,
                sl_id => $sstl_ppc->{$ss_id}->{sl_id},
            });
            push @{ $sstl_to_delete_by_shard{ $sstl_ppc->{$ss_id}->{shard} } }, $ss_id;
            next SS_ID;

        } elsif (!defined $ss_metabase->{$ss_id}
                 && defined $ss_ppc->{$ss_id}
                 && !defined $sstl_ppc->{$ss_id}
                 && !exists $banners->{$ss_id}
        ) {
            # сет отсутствует в метабазе и нет связей с ссылками
            # возможно клиент сохранит сет, совпадающий с данным по хешу, но восстановить из хеша sl_id ссылок все равно не получится 
            $log->out({
                action => 'broken set will be deleted from sitelinks_sets (only ss defined)',
                sitelinks_set_id => $ss_id,
            });
            push @{ $ss_to_delete_by_shard{ $ss_ppc->{$ss_id}->{shard} } }, $ss_id;
            next SS_ID;

        } elsif (!defined $ss_metabase->{$ss_id}
                 && defined $ss_ppc->{$ss_id}
                 && defined $sstl_ppc->{$ss_id}
                 && !exists $banners->{$ss_id}
        ) {
            # возможно такой сет остался от прошлого прохода, тогда у него будет ClientID=0
            if ($ss_ppc->{$ss_id}->{ClientID} == 0) {
                $log->out({
                    action => 'set will be deleted from sitelinks_sets (ss and sstl defined, ClientID=0)',
                    sitelinks_set_id => $ss_id,
                });
                push @{ $ss_to_delete_by_shard{ $ss_ppc->{$ss_id}->{shard} } }, $ss_id;
                $log->out({
                    action => 'set will be deleted from sitelinks_set_to_link (ss and sstl defined, ClientID=0)',
                    sitelinks_set_id => $ss_id,
                    sl_id => $sstl_ppc->{$ss_id}->{sl_id},
                });
                push @{ $sstl_to_delete_by_shard{ $sstl_ppc->{$ss_id}->{shard} } }, $ss_id;
            } else {
                # затираем у сета ClientID, чтобы его нельзя было больше использовать
                # кажется таких совсем мало (порядок - десятки), поэтому прибиваем сразу, без пачек.
                $log->out({
                    action => 'wipe ClientID in sitelinks_sets (ss and sstl defined)',
                    sitelinks_set_id => $ss_id,
                    ClientID => $ss_ppc->{$ss_id}->{ClientID},
                });
                unless ($DRY_RUN) {
                    do_sql(PPC(shard => $ss_ppc->{$ss_id}->{shard}),
                           'UPDATE sitelinks_sets ss
                                   LEFT JOIN banners b ON b.sitelinks_set_id = ss.sitelinks_set_id
                               SET ss.ClientID = 0
                             WHERE ss.sitelinks_set_id = ?
                                   AND b.sitelinks_set_id IS NULL',
                           $ss_id
                    );
                }
            }
            next SS_ID;

        } elsif (!defined $ss_metabase->{$ss_id}
                 && defined $ss_ppc->{$ss_id}
                 && defined $sstl_ppc->{$ss_id}
                 && exists $banners->{$ss_id}
                 && $ss_ppc->{$ss_id}->{shard} == $sstl_ppc->{$ss_id}->{shard}
                 && all(sub { $_ == $ss_ppc->{$ss_id}->{shard} }, keys %{ $banners->{$ss_id} })
                 && all(sub { ($bid2clientid->{$_}//0) == $ss_ppc->{$ss_id}->{ClientID} }, map { @$_ } values %{ $banners->{$ss_id} })
        ) {
            # противоположный предыдущему случай (с ограничением на консистентность)
            # точно актуально для тестовых сред, где бекапы сняты в разное время
            unless (get_shard(ClientID => $ss_ppc->{$ss_id}->{ClientID})) {
                $log->out({
                    action => 'saving client to metabase',
                    ClientID => $ss_ppc->{$ss_id}->{ClientID},
                    shard => $ss_ppc->{$ss_id}->{shard},
                });
                unless ($DRY_RUN) {
                    save_shard(ClientID => $ss_ppc->{$ss_id}->{ClientID}, shard => $ss_ppc->{$ss_id}->{shard});
                }
            }
            $log->out({
                action => 'saving set to metabase',
                sitelinks_set_id => $ss_id,
                ClientID => $ss_ppc->{$ss_id}->{ClientID},
            });
            unless ($DRY_RUN) {
                save_shard(sitelinks_set_id => $ss_id, ClientID => $ss_ppc->{$ss_id}->{ClientID});
            }
            next SS_ID;

        } elsif (defined $ss_metabase->{$ss_id}
                 && defined $ss_ppc->{$ss_id}
                 && defined $sstl_ppc->{$ss_id}
        ) {
            # сет существует и используется (или нет (= )

            # флажок, означающий, что оригинальный сет нужно сохранить.
            my $dont_wipe_original_set;

            # шард из метабазы совпадает с фактическим шардом сета и связки сет-линки
            my $set_shard_ok = $ss_metabase->{$ss_id}->{shard} == $ss_ppc->{$ss_id}->{shard}
                                && $ss_ppc->{$ss_id}->{shard} == $sstl_ppc->{$ss_id}->{shard};
            # совпадает ClientID в самом сете и в метабазе
            my $set_client_ok = $ss_metabase->{$ss_id}->{ClientID} == $ss_ppc->{$ss_id}->{ClientID};

            if (exists $banners->{$ss_id}
                && $set_shard_ok
                && $set_client_ok
                && all(sub { $_ == $ss_ppc->{$ss_id}->{shard} }, keys %{ $banners->{$ss_id} })
                && all(sub { ($bid2clientid->{$_}//0) == $ss_ppc->{$ss_id}->{ClientID} }, map { @$_ } values %{ $banners->{$ss_id} })
            ) {
                # с сетом все совсем хорошо - он консистентен и используется
                $log->out({
                    action => 'Skip set - everything is ok',
                    sitelinks_set_id => $ss_id,
                });
                next SS_ID;
            } elsif (!exists $banners->{$ss_id}
                     && $set_shard_ok
                     && $set_client_ok
            ) {
                # с сетом все хорошо, но он не используется - оставляем как есть
                $log->out({
                    action => 'Skip set - ok, but unused',
                    sitelinks_set_id => $ss_id,
                });
                next SS_ID;
            } elsif (exists $banners->{$ss_id}) {
                # собираем сет
                my $set = get_all_sql(PPC(shard => $sstl_ppc->{$ss_id}->{shard}), [
                    'SELECT sl_id, title, href, hash
                       FROM sitelinks_set_to_link stl
                            JOIN sitelinks_links sl USING(sl_id)',
                      WHERE => { 'stl.sitelinks_set_id' => $ss_id },
                  'ORDER BY order_num ASC'
                ]);
                if (!@$set) {
                    $log->out({
                        action => "Skip set - can't get set data from db (warn)",
                        sitelinks_set_id => $ss_id,
                    });
                    next SS_ID;
                }

                # нужно пересохранить сет на все баннеры
                my %clientid2new_set_id;
                while (my ($banner_shard, $bids) = each %{ $banners->{$ss_id} }) {
                    my %new_set_id2bids;
                    BID: for my $bid (@$bids) {
                        my $ClientID = $bid2clientid->{$bid};
                        if (!$ClientID) {
                            $log->out("Can't resave set $ss_id on banner $bid - no ClientID!");
                            next BID;
                        } elsif ($set_client_ok
                                 && $set_shard_ok
                                 && $banner_shard == $ss_ppc->{$ss_id}->{shard}
                                 && $ClientID == $ss_ppc->{$ss_id}->{ClientID}
                        ) {
                            # для этого баннера сет консистентен - оставляем как есть
                            $log->out("Skip resaving set $ss_id on banner $bid");
                            # и отменяем "затирание" для этого сета
                            $dont_wipe_original_set //= 1;
                            next BID;
                        } else {
                            # пересохраним сет клиенту (если еще не)
                            # определяем sitelinks_set_id на замену (в dry-run пропускаем весь блок - тут всё пишущее)
                            if (!defined $clientid2new_set_id{ $ClientID } && !$DRY_RUN) {
                                # частичный копипаст из Sitelinks::save_sitelinks_set
                                my @sitelinks_ids;
                                if ($banner_shard == $sstl_ppc->{$ss_id}->{shard}) {
                                    # линки лежат в том же шарде, что и баннер - можно не пересохранять
                                    @sitelinks_ids = @{ $sstl_ppc->{$ss_id}->{sl_id} };
                                } else {
                                    @sitelinks_ids = map { Sitelinks::_save_sitelink($_, $ClientID) } @$set;
                                }
                                my $new_set_id;
                                if (my $possible_set_id = Sitelinks::_get_sitelinks_set_id(\@sitelinks_ids, $ClientID)) {
                                    # дополнительно валидируем, что новый сет - хороший и не поломаный
                                    my $possible_set_mb_shard = get_shard(sitelinks_set_id => $possible_set_id);
                                    my $possible_set_mb_clientid = get_shard(sitelinks_set_id => $possible_set_id, 'ClientID');
                                    if ($possible_set_mb_shard
                                        && $possible_set_mb_clientid
                                        && $possible_set_mb_shard == $banner_shard
                                        && $possible_set_mb_clientid == $ClientID
                                    ) {
                                        $new_set_id = $possible_set_id;
                                    }
                                }
                                if (!defined $new_set_id) {
                                    # не нашлось или не прошел доп. проверку - сохраняем
                                    $new_set_id = get_new_id('sitelinks_set_id', ClientID => $ClientID);
                                    do_insert_into_table(PPC(ClientID => $ClientID), 'sitelinks_sets', {
                                                            sitelinks_set_id => $new_set_id,
                                                            ClientID => $ClientID,
                                                            links_hash => Sitelinks::_calc_sitelinks_set_links_hash(\@sitelinks_ids)
                                    });
                                    my $order_num = 0;
                                    do_mass_insert_sql(PPC(ClientID => $ClientID)
                                                       , 'INSERT IGNORE INTO sitelinks_set_to_link (sitelinks_set_id, sl_id, order_num) VALUES %s'
                                                       , [map { [$new_set_id, $_, $order_num++] } @sitelinks_ids]
                                    );
                                }
                                $clientid2new_set_id{$ClientID} = $new_set_id;
                            } elsif ($DRY_RUN) {
                                # чтобы пройти полный цикл логирования - подставим фейковый id
                                $clientid2new_set_id{$ClientID} //= $MAX_SS_ID + $ClientID;
                            } else {
                                # сет новому клинту уже сохранили, его id известен, ничего не делаем
                            }
                            push @{ $new_set_id2bids{ $clientid2new_set_id{$ClientID} } }, $bid;
                        }
                    } # конец for-цикла разбора баннеров в одном шарде
                    while (my ($new_set_id, $bids_for_update) = each %new_set_id2bids) {
                        for my $chunk (chunks($bids_for_update, UPDATE_CHUNK_SIZE)) {
                            my @resync_data_chunk;
                            for my $bid (@$chunk) {
                                push @resync_data_chunk, {cid => $bid2cid->{$bid}, bid => $bid, priority => RESYNC_PRIORITY};
                            }
                            $log->out({
                                action => 'replace sitelinks_set_id on banners',
                                old_id => $ss_id,
                                new_id => $new_set_id,
                                bids => $chunk,
                                shard => $banner_shard,
                                resync_data => \@resync_data_chunk,
                            });
                            unless ($DRY_RUN) {
                                do_update_table(PPC(shard => $banner_shard),
                                                'banners',
                                                {sitelinks_set_id => $new_set_id},
                                                where => {bid => $chunk, sitelinks_set_id => $ss_id}
                                );
                            }
                            push @resync_data, @resync_data_chunk;
                        }
                    }
                } # конец while-цикла разбора всех баннеров с таким sitelinks_set_id
            }

            # если с сетом все ок - next был выше. сюда попадаем - если не было баннеров, или пересохраняли сеты            
            if ($dont_wipe_original_set) {
                # если мы оставили этот сет каким-то баннерам - ничего не трогаем
                next SS_ID;
            } else {
                # затираем у старого сета ClientID, чтобы его нельзя было больше использовать
                $log->out({
                    action => 'wipe ClientID in sitelinks_sets',
                    sitelinks_set_id => $ss_id,
                    ClientID => $ss_ppc->{$ss_id}->{ClientID},
                });
                my $affected = 1; # для корректной работы в dry-run режиме
                unless ($DRY_RUN) {
                    $affected = do_sql(PPC(shard => $ss_ppc->{$ss_id}->{shard}),
                                      'UPDATE sitelinks_sets ss
                                              LEFT JOIN banners b ON b.sitelinks_set_id = ss.sitelinks_set_id
                                          SET ss.ClientID = 0
                                        WHERE ss.sitelinks_set_id = ?
                                              AND b.sitelinks_set_id IS NULL',
                                      $ss_id
                    );
                }
                if (0 + $affected) {
                    # получилось - нет баннеров с таким сетом. ставим на отложенное удаление сет и сет-линки
                    $log->out({
                        action => 'set will be deleted from sitelinks_sets, sitelinks_set_to_link, metabase (old wiped set, after resaving)',
                        sitelinks_set_id => $ss_id,
                        sl_id => $sstl_ppc->{$ss_id}->{sl_id},
                        mb_ClientID => $ss_metabase->{$ss_id}->{ClientID},
                        ss_ClientID => $ss_ppc->{$ss_id}->{ClientID},
                    });
                    push @mb_to_delete, $ss_id;
                    push @{ $ss_to_delete_by_shard{ $ss_ppc->{$ss_id}->{shard} } }, $ss_id;
                    push @{ $sstl_to_delete_by_shard{ $sstl_ppc->{$ss_id}->{shard} } }, $ss_id;
                } else {
                    $log->out({
                        action => "skip deleting wiped (ClientID = 0) set - new banners found or set deleted (warn)",
                        sitelinks_set_id => $ss_id,
                    });
                }
            }
            next SS_ID;

        } elsif (defined $ss_metabase->{$ss_id}
                 && defined $ss_ppc->{$ss_id}
                 && !defined $sstl_ppc->{$ss_id}
                 && !exists $banners->{$ss_id}
        ) {
            # какой-то странный случай, но один такой попался
            # поскольку самих связей сет-линки нет - сносим.
            $log->out({
                action => 'set will be deleted from sitelinks_sets, metabase (only ss and metabase defined)',
                sitelinks_set_id => $ss_id,
                mb_ClientID => $ss_metabase->{$ss_id}->{ClientID},
                ss_ClientID => $ss_ppc->{$ss_id}->{ClientID},
            });
            push @mb_to_delete, $ss_id;
            push @{ $ss_to_delete_by_shard{ $ss_ppc->{$ss_id}->{shard} } }, $ss_id;
            next SS_ID;

        } elsif (!defined $ss_metabase->{$ss_id}
                 && defined $ss_ppc->{$ss_id}
                 && defined $sstl_ppc->{$ss_id}
                 && exists $banners->{$ss_id}
                 && $ss_ppc->{$ss_id}->{ClientID} == 0
                 && all(sub { ! $bid2clientid->{$_} }, map { @$_ } values %{ $banners->{$ss_id} })
        ) {
            # такое может получится после предыдущих проходов для баннеров, у которых не определился ClientID
            # все сносим
            $log->out({
                action => 'set will be deleted from sitelinks_sets, sitelinks_set_to_link (ss, sstl, banners defined, ClientID=0)',
                sitelinks_set_id => $ss_id,
                sl_id => $sstl_ppc->{$ss_id}->{sl_id},
            });
            push @{ $ss_to_delete_by_shard{ $ss_ppc->{$ss_id}->{shard} } }, $ss_id;
            push @{ $sstl_to_delete_by_shard{ $sstl_ppc->{$ss_id}->{shard} } }, $ss_id;

            while (my ($banner_shard, $bids) = each %{ $banners->{$ss_id} }) {
                for my $chunk (chunks($bids, UPDATE_CHUNK_SIZE)) {
                    $log->out({
                        action => 'remove set from banners (ss, sstl, banners defined, ClientID=0)',
                        sitelinks_set_id => $ss_id,
                        bid => $chunk,
                        shard => $banner_shard,
                    });
                    unless ($DRY_RUN) {
                        do_update_table(PPC(shard => $banner_shard),
                                        'banners',
                                        {sitelinks_set_id => undef, statusSitelinksModerate => 'New', statusBsSynced => 'No'},
                                        where => {sitelinks_set_id => $ss_id, bid => $chunk},
                        );
                    }
                }
            }
            next SS_ID;

        } elsif (defined $ss_metabase->{$ss_id}
                 && !defined $ss_ppc->{$ss_id}
                 && !defined $sstl_ppc->{$ss_id}
                 && !exists $banners->{$ss_id}
        ) {
            # такое может быть, если упали сразу после взятия id из метабазы (до записи в базу)
            $log->out({
                action => 'set will be deleted from metabase (only metabase defined)',
                sitelinks_set_id => $ss_id,
                ClientID => $ss_metabase->{$ss_id}->{ClientID},
            });
            push @mb_to_delete, $ss_id;

        } else {
            $log->out({
                unhandled_data => {
                    ss_id => $ss_id,
                    ss_metabase => $ss_metabase->{$ss_id},
                    ss_ppc => $ss_ppc->{$ss_id},
                    sstl_ppc => $sstl_ppc->{$ss_id},
                    banners => $banners->{$ss_id},
                    bid2clientid => hash_cut($bid2clientid, map { @$_ } values %{ $banners->{$ss_id} }),
                },
            });
        }
    } # конец цикла по sitelinks_set_id внутри текущего чанка

    # удаляем сеты, которые были только в sitelinks_set_to_link
    while (my ($shard, $ss_ids) = each %sstl_to_delete_by_shard) {
        for my $chunk (chunks($ss_ids, UPDATE_CHUNK_SIZE)) {
            $log->out({
                action => 'bulk deleting sets from sitelinks_set_to_link',
                sitelinks_set_id => $chunk,
                shard => $shard,
            });
            unless ($DRY_RUN) {
                do_delete_from_table(PPC(shard => $shard), 'sitelinks_set_to_link', where => {sitelinks_set_id => $chunk});
            }
        }
    }

    # удаляем сеты, которые были только в sitelinks_sets
    while (my ($shard, $ss_ids) = each %ss_to_delete_by_shard) {
        for my $chunk (chunks($ss_ids, UPDATE_CHUNK_SIZE)) {
            $log->out({
                action => 'bulk deleting sets from sitelinks_sets',
                sitelinks_set_id => $chunk,
                shard => $shard,
            });
            unless ($DRY_RUN) {
                do_delete_from_table(PPC(shard => $shard), 'sitelinks_sets', where => {sitelinks_set_id => $chunk});
            }
        }
    }
    # проверяем, то не появилось (в момент проверки выше по коду - не было) новых баннеров, использующих данный сет
    my $bid2ss_id = get_hash_sql(PPC(shard => 'all'), [
                                 'SELECT bid, sitelinks_set_id FROM banners',
                                 WHERE => {sitelinks_set_id => [map { @$_ } values %ss_to_delete_by_shard]}
    ]);
    for my $bids_chunk (chunks([keys(%$bid2ss_id)], UPDATE_CHUNK_SIZE)) {
        my $ss_ids_chunk = [@{$bid2ss_id}{@$bids_chunk}];
        $log->out({
            action => 'remove sets from banners (only ss defined)',
            sitelinks_set_id => $ss_ids_chunk,
            bid => $bids_chunk,
        });
        unless ($DRY_RUN) {
            do_update_table(PPC(bid => $bids_chunk),
                            'banners',
                            {sitelinks_set_id => undef, statusSitelinksModerate => 'New', statusBsSynced => 'No'},
                            where => {sitelinks_set_id => $ss_ids_chunk, bid => SHARD_IDS},
            );
        }
    }

    # удаляем из метабазы (внутри своя нарезка на чанки)
    if (@mb_to_delete) {
        $log->out({
            action => 'remove sets from metabase',
            sitelinks_set_id => \@mb_to_delete,
        });
        unless ($DRY_RUN) {
            delete_shard(sitelinks_set_id => \@mb_to_delete);
        }
    }

    # насыпаем данных в переотправку
    if (@resync_data) {
        $log->out(sprintf("Adding %d itmes to bs_resync_queue", scalar(@resync_data)));
        unless ($DRY_RUN) {
            bs_resync(\@resync_data);
        }
    }

    Yandex::DBShards::clear_cache();
}; # end of "relaxed" sub

    # сбрасываем профайлинг после N итераций
    if (++$iteration_num >= FLUSH_PROFILING_AFTER_ITERATION_CNT) {
        $iteration_num = 0;
        Yandex::Trace::restart(\$ScriptHelper::trace);
    }
} # end of main loop

$log->out('FINISH');

exit 0;
