#!/usr/bin/perl

use my_inc '../..';


=head1 NAME

relocate-zab-bu-geo.pl - исправление геотаргетинга

=head1 DESCRIPTION

    После обновления геодерева Бурятия и Забайкальский край переехали из Сибири на Дальний Восток.
    Исправляем геотаргетинг после смены положения региона в дереве таким образом, чтобы у рекламодателя не изменился охват.

=cut

use Direct::Modern;

use Settings;
use ScriptHelper;
use Yandex::DBTools;
use Yandex::DBShards;
use ShardingTools;
use Yandex::Retry;
use Yandex::HashUtils qw/hash_cut/;
use BS::ResyncQueue qw/bs_resync/;
use GeoTools qw/refine_geoid/;

my $SLEEP_COEF = 1;
my $CHUNK_SIZE = 500;

extract_script_params(
    'sleep-coef=f' => \$SLEEP_COEF,
    'shard-id=s' => \my @SHARDS,
    'cid=i' => \my @CIDS,
    'dry|dry-run!' => \my $DRY,
);


my $BS_RESYNC_PRIORITY_HIGH = 94; # +1 к приоритету PRIORITY_NDSDISCOUNT_GRAPH_CHANGED
my $BS_RESYNC_PRIORITY_LOW  = 46;
my %shards = map { $_ => 1 } (@SHARDS ? @SHARDS : ppc_shards());

my $geo_id = {
    ZAB => 21949, # Забайкальский край
     BU => 11330, # Республика Бурятия
};
my $old_parent = 59; # Сибирь
my $new_parent = 73; # Дальний Восток

my $geo_condition = [
    (@CIDS ? (cid => \@CIDS) : ()),
    geo__rlike => "(^|,)-?($old_parent|$new_parent)(,|\$)",
];



$log->out('START');

foreach_shard_parallel_verbose(
    $log,
    sub {
        my ($shard) = @_;
        return unless $shards{$shard};
        my $log_guard = $log->msg_prefix_guard("[shard $shard]");

        iterate_table($shard, campaigns => 'cid');
        iterate_table($shard, phrases => 'pid', resync => 1);
        iterate_table($shard, mediaplan_banners => 'mbid');
    }
);

$log->out('FINISH');



sub updated_geo_targeting {
    my ($geo, $client_id) = @_;

    my %new_geo = map { $_ => 1 } split /\s*,\s*/ => $geo;

    if (   exists $new_geo{$old_parent} && exists $new_geo{$new_parent}
        || exists $new_geo{-$old_parent} && exists $new_geo{-$new_parent})
    {
        # оставляем гео без изменений если присутствуют оба федеральных округа
        return $geo;
    } elsif (exists $new_geo{$old_parent} || exists $new_geo{-$new_parent}) {
        # добавляем Забайкальский край и Бурятию, если явно отминусованы - удаляем
        if (exists $new_geo{-$geo_id->{ZAB}}) {
            delete $new_geo{-$geo_id->{ZAB}};
        } else {
            $new_geo{$geo_id->{ZAB}} = 1
        }
        if (exists $new_geo{-$geo_id->{BU}}) {
            delete $new_geo{-$geo_id->{BU}};
        } else {
            $new_geo{$geo_id->{BU}} = 1
        }
    } elsif (exists $new_geo{-$old_parent} || exists $new_geo{$new_parent}) {
        # вычитаем Забайкальский край и Бурятию, если есть явный плюс - удаляем
        if (exists $new_geo{$geo_id->{ZAB}}) {
            delete $new_geo{$geo_id->{ZAB}};
        } else {
            $new_geo{-$geo_id->{ZAB}} = 1
        }
        if (exists $new_geo{$geo_id->{BU}}) {
            delete $new_geo{$geo_id->{BU}};
        } else {
            $new_geo{-$geo_id->{BU}} = 1
        }
    } else {
        $log->warn("Invalid geo selected: $geo");
        return $geo;
    }

    return refine_geoid([keys %new_geo], undef, {ClientID => $client_id})
}


sub iterate_table {
    my ($shard, $table, $key, %opt) = @_;

    my $min = 0;
    my $count = 0;
    while (1) {
        my $relax = relaxed_guard(times => $SLEEP_COEF);

        $log->out("Quering up to $CHUNK_SIZE records from $table starting from $key=$min");
        my $batch = get_all_sql(PPC(shard => $shard), [
            "select $key, cid, geo from $table",
            where => [
                "${key}__ge" => $min,
                @$geo_condition,
            ],
            'order by' => $key,
            limit => $CHUNK_SIZE,
        ]);
        my $cid2data = get_hashes_hash_sql(PPC(shard => $shard), ["select cid, ClientID, archived, OrderID, statusShow from campaigns", where => {cid => [ map { $_->{cid} } @$batch]}]);

        $log->out(sprintf "Got %d records", scalar @$batch);
        last if !@$batch;
        $min = $batch->[-1]->{$key} + 1;

        my @changed;
        for my $record (@$batch) {
            my $new_geo = updated_geo_targeting($record->{geo}, $cid2data->{$record->{cid}}->{ClientID});
            next if $new_geo eq $record->{geo};
            $log->out(sprintf "change $key=%d %s --> %s", $record->{$key}, $record->{geo}, $new_geo);
            $record->{old} = $record->{geo};
            $record->{geo} = $new_geo;
            if ($cid2data->{$record->{cid}}->{statusShow} eq 'Yes') {
                $record->{priority} = $BS_RESYNC_PRIORITY_HIGH;
            } else {
                $record->{priority} = $BS_RESYNC_PRIORITY_LOW;
            }
            push @changed, $record;
        }

        if (@changed) {
            if ($DRY) {
                $log->out(sprintf "DRY: skipped updating %d records in $table", scalar @changed);
            }
            else {
                my $affected = do_update_table(PPC(shard => $shard), $table,
                    { geo__dont_quote => sql_case($key, {map { $_->{$key} => $_->{geo} } @changed}) },
                    where => {
                        $key => [ map { $_->{$key} } @changed ],
                        geo__dont_quote => sql_case($key, {map { $_->{$key} => $_->{old} } @changed}),
                });
                $count += @changed;
                $log->out(sprintf "Updated %d records in $table", $affected);

                if ($opt{resync}) {
                    my @resync;
                    for my $rec (@changed) {
                        if (   $cid2data->{$rec->{cid}}->{archived} eq 'No'
                            && $cid2data->{$rec->{cid}}->{OrderID} > 0)
                        {
                            push @resync, hash_cut $rec, qw/cid pid bid priority/;
                        }
                    }
                    bs_resync(\@resync);
                }
            }
        }
    }
    $log->out("Changed total $count records from $table");
}


