#!/usr/bin/perl

=head1 SYNOPSIS

    LOG_TEE=1 ./protected/one-shot/resend_country_currency_to_balance.pl --file=broken_clients.txt
    LOG_TEE=1 ./protected/one-shot/resend_country_currency_to_balance.pl --client-id=123123

=head1 DESCRIPTION

    Перепосылает в Баланс страну, валюту и город клиентов, указанных в файле.
    Если клиент агентский, то страна не выставляется, а город берется у агентства

    После отправки в Баланс запускается синхронизация НДС

=cut

use Direct::Modern;

use File::Slurp;

use my_inc '../..';

use Client;
use Client::NDSDiscountSchedule;
use Primitives;
use RBAC2::Extended;
use ScriptHelper;
use Settings;

my $OPERATOR_UID = 1;
my $rbac = eval { RBAC2::Extended->get_singleton($OPERATOR_UID) }
    or $log->die("Can't connect to RBAC: $@");

extract_script_params(
    'file=s' => \my $client_ids_file,
    'client-id=s' => \my $only_client_id,
);

my @client_ids;
if ($client_ids_file) {
    @client_ids = read_file($client_ids_file, chomp => 1);
} elsif ($only_client_id) {
    @client_ids = ($only_client_id);
} else {
    die "please specify either --file or --client-id";
}

$log->out('START');
if ($client_ids_file) {
    $log->out("working on clients from file $client_ids_file");
} else {
    $log->out("working only on client $only_client_id");
}

for my $client_id (@client_ids) {
    my $client_data = get_client_data($client_id, [qw/ClientID country_region_id subregion_id work_currency/]);
    if (!%$client_data) {
        $log->out("client $client_id no found!");
        next;
    }

    my $agency_id = Primitives::get_client_first_agency($client_id);
    if ($agency_id) {
        $log->out("client $client_id is a subclient");
    } else {
        $log->out("client $client_id is a client");
    }

    my %balance_upd;
    if (!$agency_id) {
        $balance_upd{REGION_ID} = $client_data->{country_region_id};
        $balance_upd{SUBREGION_ID} = $client_data->{subregion_id};
    } else {
        my $agency_data = get_client_data($agency_id, [qw/subregion_id/]);
        if (!%$agency_data) {
            $log->out("no subregion_id for agency $agency_id");
            next;
        }
        $balance_upd{SUBREGION_ID} = $agency_data->{subregion_id};
    }
    $balance_upd{CURRENCY} = $client_data->{work_currency};
    $balance_upd{MIGRATE_TO_CURRENCY} = Client::get_client_migrate_to_currency($client_id);

    $log->out({UPDATE => $client_id, data => \%balance_upd});
    if (my $is_error = update_client_id($OPERATOR_UID, $client_id, \%balance_upd)) {
        $log->out("update ClientID $client_id FAILED");
    } else {
        $log->out("ClientID $client_id updated in balance");
    }

    $log->out("resyncing NDS schedule for client $client_id");
    Client::NDSDiscountSchedule::sync_nds_schedule_for_clients([$client_id], rbac => $rbac, log => $log, dont_die => 1, log_data_from_balance => 1);
}

$log->out('FINISH');
