#!/usr/bin/perl

=head1 DESCRIPTION

   https://st.yandex-team.ru/DIRECT-105006
   Восстанавливаем модели атрибуции для конверсионных стратегий со strategy_name='autobudget'
   Данные берем из yt таблицы в которой есть исходные данные о кампании
       attribution_model_before - модель атрибуции до миграции
       strategy_name - стратегия до миграции

=head1 USAGE

    Запуск в холостом режиме, без записи в базу
    LOG_TEE=1 ./protected/one-shot/restore_attribution_model_from_yt_and_add_to_resync.pl --dry-run --priority 85 --attribution-type 2 --table <путь к таблице>

    Запуск с записью в базу
    LOG_TEE=1 ./protected/one-shot/restore_attribution_model_from_yt_and_add_to_resync.pl --priority 85 --attribution-type 2 --table <путь к таблице>
=cut

use Direct::Modern;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::Validate;
use Yandex::YT::Table;

use my_inc '../..';

use BS::ResyncQueue;
use ScriptHelper;

use Settings;
use Tools ();

$Yandex::YT::Streaming::DEFAULT_FORMAT = 'json';

my $ATTRIBUTION_MODEL_VALUES = {
    1 => "last_click",
    2 => "last_significant_click",
    3 => "first_click",
};

my @FIELDS_TO_READ = qw/cid type attribution_model_before strategy_name/;

my $table_path;
my $priority = 0;
my $attribution_type;
my $dryrun;
my $chunk_size = 5000;
extract_script_params(
    'table=s'            => \$table_path,
    'priority=i'         => \$priority,
    'attribution-type=i' => \$attribution_type,
    'dry-run'            => \$dryrun,
    'chunk-size=i'       => \$chunk_size
);

unless ($table_path) {
    die "--table parameter is mandatory";
}

unless (is_valid_int($priority, -128,127)) {
    die "--priority must be in [-128,127] range";
}

my $direct_attribution_model = $ATTRIBUTION_MODEL_VALUES->{$attribution_type};
unless ($direct_attribution_model) {
    die "There are no attribution_model for attribution_type $attribution_type";
}
my $direct_attribution_model_quoted = sql_quote($direct_attribution_model);
$log->out("START");
$log->out("use updating campaigns to attribution_model = $direct_attribution_model");


sub update_attribuiton_model_and_add_to_resync {
    my $cids = shift;
    my $cids_to_update = get_one_column_sql(PPC(cid => $cids ), [ "select cid from campaigns",
        "where" => {
            cid  => SHARD_IDS,
            strategy_name => 'autobudget',
            attribution_model => 'last_yandex_direct_click'} ]);
    if (@$cids_to_update < @$cids) {
        my %cids_to_update_hash = map {$_ => 1} @$cids_to_update;
        my @skipping_cids = grep { !$cids_to_update_hash{$_} } @$cids;
        $log->out({"skipping campaigns with changed strategy or attribution_model" => \@skipping_cids});
    }

    $log->out({"update to attribution_model=$direct_attribution_model. cids: " => $cids_to_update});
    unless ($dryrun) {
        do_sql(PPC(cid => $cids_to_update ), [ "update campaigns set attribution_model = $direct_attribution_model_quoted",
            "where" => {
                cid  => SHARD_IDS,
                strategy_name => 'autobudget',
                attribution_model => 'last_yandex_direct_click'} ]);

        $log->out("add data chunk to bs_resync_queue");
        my $cnt = BS::ResyncQueue::bs_resync_camps(\@$cids_to_update, priority => $priority);
        $log->out("added $cnt rows to resync_queue");
    }
}

sub check_update_createria {
    my $row = shift;

    if ($row->{strategy_name} eq "autobudget"
        && $row->{attribution_model_before} eq $direct_attribution_model ) {
        return 1;
    }
    return 0;
}

$log->out("setup YT env");
Tools::set_yt_environment();

$log->out("Import campaigns from $table_path with bs attribution_model $direct_attribution_model");
my $table = Yandex::YT::Table->new($table_path);
unless ($table->exists()) {
    $log->die("table $table_path doesn't exists");
}

$log->out("start import data from yt");
my @cids_to_update_attribution_type;

my $reader = $table->reader(\@FIELDS_TO_READ);
while(my $row = $reader->next()) {
    next unless check_update_createria($row);
    $log->out($row);

    push @cids_to_update_attribution_type, $row->{cid};
    if (@cids_to_update_attribution_type >= $chunk_size) {
        update_attribuiton_model_and_add_to_resync(\@cids_to_update_attribution_type);
        @cids_to_update_attribution_type=();
    }

}
update_attribuiton_model_and_add_to_resync(\@cids_to_update_attribution_type);

$log->out("FINISH");
