#!/usr/bin/perl

use my_inc "../..";


=head1 NAME

    send_wallets_to_service

=head1 SYNOPSIS

    send_wallets_to_service.pl --chunk-size=100 --sleep=10 --wallet-cids-file=wallet_cids_file.dat
        Переводить за раз по 100 кампаний, делая паузы в 10 секунд. Параметр wallet-cids-file опциален и позволяет ограничить затрагиваемые кошельки (при этом есть ограничение на максимальный размер файла)

=head1 DESCRIPTION

    Перевод на сервисирование самоходных кампаний-кошельков, при наличии в них сервисируемых кампаний.

=cut

use strict;
use warnings;
use utf8;

use lib::abs '..';

use Settings;
use Yandex::DBTools;
use ScriptHelper;
use Yandex::ListUtils qw/chunks/;
use ShardingTools;
use List::MoreUtils qw/uniq/;

use Campaign;
use RBAC2::Extended;
use RBACDirect;
use Path::Tiny;
use Yandex::Validate qw/is_valid_id/;
use Campaign::Types qw/get_camp_kind_types/;

my $WALLETS_FILE_LIMIT = 1000;
my ($CHUNK_SIZE, $SLEEP, $WALLETS_FILENAME);

extract_script_params(
    "chunk-size=i"  => \$CHUNK_SIZE,
    "sleep=i"       => \$SLEEP,
    'wallet-cids-file=s' => \$WALLETS_FILENAME,
);

usage() && exit unless $CHUNK_SIZE && $SLEEP;

$log->out('START');

my @wallet_cids = ();
if ($WALLETS_FILENAME) {
    @wallet_cids = path($WALLETS_FILENAME)->lines({ chomp => 1}) or $log->die("Can't open data file $WALLETS_FILENAME: $@");
    if (scalar(@wallet_cids) > $WALLETS_FILE_LIMIT) {
        $log->die("$WALLETS_FILENAME is too big: got ".scalar(@wallet_cids).", lines max $WALLETS_FILE_LIMIT");
    }
    @wallet_cids = grep { is_valid_id($_) } @wallet_cids;
    $log->out("restricting wallet cids to ".join(",",  @wallet_cids));
}

my $rbac = eval { RBAC2::Extended->get_singleton(1) }
    or $log->die("Can't connect to RBAC: $@");

my $quit = 0;

$SIG{INT} = sub {
    $log->out('SIGINT RECEIVED');
    $quit = 1;
};

$SIG{TERM} = sub {
    $log->out('SIGTERM RECEIVED');
    $quit = 1;
};

my $total_wallets = 0;
my %updated_wallets;
for my $shard (ppc_shards()) {
    # Выберем несервисируемые кошельки с сервисируемыми кампаниями внутри
    # Сортировка по количеству кампаний у менеджера внутри данного кошелька
    # Т.к. могут быть случаи, что внутри одного кошелька могут быть несколько кампаний на разных менеджерах - выберем того менеджера, у кого больше кампаний
    my %where = ('c.archived' => 'No', 'c.ManagerUID__gt' => '0', 'c.type' => get_camp_kind_types('under_wallet'), 'wc.ManagerUID__is_null' => 1);
    if (@wallet_cids) {
        $where{'wc.cid'} = \@wallet_cids;
    }
    my $where_cond = sql_condition(\%where);

    my $broken_wallets = get_all_sql(PPC(shard => $shard), ["
        SELECT w.*,
            (SELECT COUNT(xc.cid) FROM campaigns xc WHERE xc.uid = w.uid AND xc.ManagerUID = w.ManagerUID AND xc.archived = 'No' AND xc.wallet_cid = w.wallet_cid) serv_count
        FROM
        (
            SELECT DISTINCT c.wallet_cid, c.uid, c.ManagerUID
            FROM campaigns c
            JOIN campaigns wc ON (wc.cid = c.wallet_cid AND wc.uid = c.uid)
            WHERE $where_cond
        ) w
        ORDER BY serv_count DESC;
       ",
    ]);

    my @wallet_ids = uniq map { $_->{wallet_cid} } @$broken_wallets;
    $total_wallets += scalar(@wallet_ids);

    # список id всех биллинговых агрегатов под определенным ОС
    my $wallet_cid2bill_agg_cids = {};
    my $bill_agg_rows = get_all_sql(PPC(shard => $shard), ["
        SELECT wallet_cid, cid
        FROM campaigns",
        WHERE => [
            'type' => 'billing_aggregate',
            'wallet_cid' => \@wallet_ids,
        ],
    ]);
    push @{$wallet_cid2bill_agg_cids->{$_->{wallet_cid}}}, $_->{cid} for @$bill_agg_rows;

    foreach my $chunk (chunks ($broken_wallets, $CHUNK_SIZE)) {
        for my $row (@$chunk) {
            my $uid = $row->{uid};
            my $manager_uid = $row->{ManagerUID};
            my $wallet_cid = $row->{wallet_cid};

            next if $updated_wallets{$wallet_cid};
            $log->out(qq/Set servicing for wallet $wallet_cid (Manager: $manager_uid)/);

            eval {
                do_in_transaction {
                    if (!rbac_is_scampaign($rbac, $wallet_cid)) {
                        # Кошелек ожидает сервисирования ?
                        if (rbac_get_camp_wait_servicing($rbac, $wallet_cid)) {
                            my $rbac_error = rbac_accept_servicing($rbac, $wallet_cid, $uid, $manager_uid);
                            die "rbac_accept_servicing(\$rbac, $wallet_cid, $uid, $manager_uid) == $rbac_error" if $rbac_error;
                        } else {
                            my $rbac_error = rbac_move_nscamp_to_scamp($rbac, $wallet_cid, $manager_uid, $uid);
                            die "rbac_move_nscamp_to_scamp(\$rbac, $wallet_cid, $manager_uid, $uid) == $rbac_error" if $rbac_error;
                        }
                    }

                    do_update_table(PPC(shard => $shard), 'campaigns', {
                        ManagerUID => $manager_uid,
                        statusBsSynced => 'No'
                    }, where => {cid => $wallet_cid});

                    Campaign::campaign_manager_changed($rbac, $manager_uid, $wallet_cid, $manager_uid);

                    my $bill_agg_cids = $wallet_cid2bill_agg_cids->{$wallet_cid};
                    for my $ba_cid (@$bill_agg_cids) {
                        if (!rbac_is_scampaign($rbac, $ba_cid)) {
                            # Биллинговый агрегат ожидает сервисирования ?
                            if (rbac_get_camp_wait_servicing($rbac, $ba_cid)) {
                                my $rbac_error = rbac_accept_servicing($rbac, $ba_cid, $uid, $manager_uid);
                                die "rbac_accept_servicing(\$rbac, $ba_cid, $uid, $manager_uid) == $rbac_error" if $rbac_error;
                            } else {
                                my $rbac_error = rbac_move_nscamp_to_scamp($rbac, $ba_cid, $manager_uid, $uid);
                                die "rbac_move_nscamp_to_scamp(\$rbac, $ba_cid, $manager_uid, $uid) == $rbac_error" if $rbac_error;
                            }
                        }

                        do_update_table(PPC(shard => $shard), 'campaigns', {
                            ManagerUID => $manager_uid,
                            statusBsSynced => 'No'
                        }, where => {cid => $ba_cid});

                        Campaign::campaign_manager_changed($rbac, $manager_uid, $ba_cid, $manager_uid);
                    }

                    $updated_wallets{$wallet_cid}++;
                };

                1;
            } or do {
                $log->out("Error: $@");
            };

            last if $quit;
        }

        last if $quit;
        sleep $SLEEP;
    }

    last if $quit;
}

$log->out("Found: $total_wallets wallets \tUpdated: ".scalar(keys %updated_wallets));
$log->out('FINISH');
