#!/usr/bin/perl
#
#

use my_inc "../..";

use 5.010;
use strict;
use warnings;
no warnings 'uninitialized';
use utf8;
use open ':std' => ':utf8';

use Data::Dumper;


use Settings;
use Yandex::DBShards;
use Yandex::DBTools;
use ShardingTools;
use CommonMaps qw/check_address_map/;
use BS::ResyncQueue;
use ScriptHelper;

our $log;

my @ADDR_FIELDS = qw/country city street house build apart/;
my %MAX_IDS;

my %STAT = (
    for_update     => {descr => 'Визитки для перекодирования'},
    exists_address => {descr => 'Визиток с адресом'},
    null_address   => {descr => 'Визиток без адреса'},
    redefined      => {descr => 'Удалось геокодировать'},
    notdefined     => {descr => 'Не удалось геокодировать'},
    updated        => {descr => 'Всего обновили визиток'},
    need_but_not   => {descr => 'Адрес изменился, пока скрипт работает'},
    banners        => {descr => 'Обновлено баннеров'},
    correct        => {descr => 'Не нуждается в корректировке'},
);
sub set_stat($;$) {
    if (@_ == 1) {
        $STAT{$_[0]}{value}++;
    }
    else {
        $STAT{$_[0]}{value} += $_[1];
    }
}

sub print_stat {
    $log->out(
        '-' x 42,
        "Статистика:", 
        (map { sprintf("%15s: %d", $_->{descr}, $_->{value}) } values %STAT),
        '-' x 42,
    );
}


my $COUNT;
my $LIMIT = 5_000;
extract_script_params(
    'limit|l=i' => \$LIMIT,
    'count|c=i' => \$COUNT,
    'id|i=i'    => \my $VCARD_ID,
);





$log->out('start');


foreach_shard shard => [ ppc_shards() ], sub {
    my ($shard, $key) = @_; 

    $MAX_IDS{$shard} = $VCARD_ID if $VCARD_ID; # для отладки

    $log->out("process shard $shard");

    my $count = 0;
    while () {
        last if $COUNT && $count >= $COUNT; # для отладки

        my $data = get_vcards($shard, $LIMIT) or last;

        $count += @$data;

        my $vcards = { map {$_->{vcard_id} => $_} @$data }; 

        define_addresses($shard, $vcards) or next;
        update_vcards($shard, $vcards)    or next;
        update_banners($shard, $vcards);

        $log->out("shard $shard, обработано $count визиток");
        print_stat();
    }

    $log->out("shard $shard processed");

};

print_stat();
$log->out('finish');



sub get_vcards {
    my ($shard, $limit) = (shift, shift);

    $log->out('Выбираем визитки');

    my $max_vcard_id = $MAX_IDS{$shard} ||= 0;
    my $data = get_all_sql( PPC(shard => $shard), 
        [
            q[SELECT vcard_id, cid,
                    country, city, street, house, build, apart,
                    concat_ws('||', country, city, street, house, build, apart) as addr_hash,
                    u.ClientID,
                    v.address_id as address_id,
                    concat_ws(',', m.x, m.y)               AS manual_point,
                    concat_ws(',', m.x1, m.y1, m.x2, m.y2) AS manual_bounds
            FROM vcards v JOIN users u ON v.uid = u.uid
            LEFT JOIN addresses a ON v.address_id = a.aid
            LEFT JOIN maps      m ON a.map_id = m.mid
            ],
            where => { # address_id__is_null  => 1, 
                       city__is_not_null    => 1,
                       country__is_not_null => 1, 
                       street__is_not_null  => 1,
                       vcard_id__gt__int    => $max_vcard_id,
                     },
            'ORDER BY vcard_id',
            "LIMIT  $limit"
        ],
    );

    return unless @$data;

    $MAX_IDS{$shard} = $data->[-1]{vcard_id};

    set_stat for_update => @$data;
    $log->out("визиток для обработки: " . @$data); 

    return $data;
}


sub addr {
    my $v = shift;
    join ', ', map {$v->{$_} ? "$_: $v->{$_}" : ()} @ADDR_FIELDS;
}

sub define_addresses {
    my ($shard, $vcards) = (shift, shift);

    $log->out('Определение адресов, ' . keys %$vcards);

    no warnings 'uninitialized';

    foreach my $vcard ( values %$vcards ) {
        my $client_id = $vcard->{ClientID};

        set_stat $vcard->{address_id} ? 'exists_address' : 'null_address';

        my $map = CommonMaps::check_address_map(
            $vcard, 
            { ClientID => $client_id }
        );

        unless ($map && $map->{aid}) {
            delete $vcards->{$vcard->{vcard_id}};
            $log->out("адрес не определен: $vcard->{vcard_id} " . addr($vcard)); 
            set_stat 'notdefined';
            next;
        }

        if ($vcard->{address_id} && $vcard->{address_id} == $map->{aid}) {
            set_stat 'correct';
            delete $vcards->{$vcard->{vcard_id}};
            next;
        }

#        $log->out("for update: vcard $vcard->{vcard_id}, " . addr($vcard));

        $vcard->{orig_address_id} = $vcard->{address_id};
        $vcard->{address_id} = $map->{aid};
    }


    set_stat redefined => scalar keys %$vcards;

    return scalar keys %$vcards;
}

sub update_vcards {
    my ($shard, $vcards) = (shift, shift);

    $log->out('Обновление визиток, ' . keys %$vcards);

    my $addr_fields = join ', ', @ADDR_FIELDS;

    foreach my $vcard ( values %$vcards ) {
        my $aid = $vcard->{orig_address_id}; 
        my $res = do_sql( PPC(shard => $shard),
            [
                'UPDATE vcards SET address_id = ?, LastChange = LastChange',
                WHERE => { 
                    vcard_id   => $vcard->{vcard_id}, 
                    defined $aid ? (address_id => $aid) : (address_id__is_null => 1),
                },
                " AND concat_ws('||', $addr_fields) = ?",
            ],
            $vcard->{address_id},
            $vcard->{addr_hash},
        );

        unless ($res > 0) {
            $log->out("Визитка не обновлена:\n", Dumper $vcard);
            delete $vcards->{$vcard->{vcard_id}};
            set_stat 'need_but_not';
        }
    }

    set_stat updated => keys %$vcards;

    return scalar keys %$vcards;
}




sub update_banners {
    my ($shard, $vcards) = (shift, shift);

    $log->out('обновление баннеров');

    my @vcard_ids = keys %$vcards;

    unless (@vcard_ids) {
        $log->out('update_banners: нет визиток');
        return;
    }

    my $banners = get_all_sql( PPC(shard => $shard), [ 
        "SELECT bid, cid FROM banners",
        where => {
            vcard_id => \@vcard_ids,
        }
    ]);

    foreach ( @{$banners || []} ) {
        $_->{pid}      = 0;
        $_->{priority} = BS::ResyncQueue::PRIORITY_MAINTENANCE_FIX_VCARD_GEO_ID;
    }


    my $res = BS::ResyncQueue::bs_resync $banners;

    $log->out("обновлено $res баннеров");
    
    set_stat banners => $res;
}


