#!/usr/bin/perl

use my_inc "../..";

=head2 DESCRIPTION

    Принцип работы - выбираем всех клиентов и ту страну, что у них сейчас в Директе (0 если ее нет), затем проходимся в цикле по всем клиентам.
        Если страна уже есть - ничего не делаем (только сверяемся с файлом - напишем в лог, если есть расхождение)
        Если страны у нас нет, но она есть в файле от баланса (файлу - верим, такая страна уже проставлена в балансе, "дедукцию" не применяем)
            - обновляем клиента у себя в базе
            - обновляем в балансе - теоритически вызов на обновление сделан не будет, т.к. новые данные совпадут с текущими,
                если нет - накроем в балансе страну значением из их же списка.
        Если у нас страны нет и файле ничего не нашлось - спрашиваем у баланса страны/валюты доступные для платежей клиенту,
            - в ответе упоминается только одна страна - считаем ее определенной, и обновляем у себя и в балансе данные
            - ответ пустой или страна не уникальна - ничего не делаем, пропускаем

    Обновление страны у себя в базе и в балансе производим внутри одной транзации (если не получится обновить в балансе - у себя откатим)


    Деплой обрабатывает клиентов отсортированных по шарду и ClientID (1: 3,4; 2: 1, 2).
    Если деплой упадет (переключение базы) или его нужно будет остановить, - нужно будет делать последующие запуски по всем шардам не с самого начала, а c ключами
        --start-client-id=NNNN --only-shard-id=M
    В качестве начального ClientID нужно будет взять последний обработанный (по логам) ClientID по каждому шарду или 0, если до шарда не дошли и
    запустить 4 (по числу шардов) экземпляра скрипта отдельно по каждому шарду

=cut


use warnings;
use strict;
use utf8;

use List::MoreUtils qw/uniq/;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::Validate;
use Yandex::Balance;
use Yandex::Retry;

use BalanceWrapper;
use Settings;
use ScriptHelper;
use geo_regions;
use Client;


# Вызовов к балансу ожидается много, складываем их логи в отдельное место, для удобства.
local %Yandex::Balance::BALANCE_CALLS_LOG_SETTINGS = (
    use_syslog => 0,
    log_file_name => "20140429_setup_clients_country_balance_calls.log",
    date_suf => "%Y%m%d",
);

my ($START_CLIENT_ID, $ONLY_SHARD_ID);
extract_script_params(
    'start-client-id=i' => \$START_CLIENT_ID,
    'only-shard-id=i'   => \$ONLY_SHARD_ID,
);

$log->out('START');

if (defined $START_CLIENT_ID) {
    $log->die("Invalid start ClientID value specified: $START_CLIENT_ID") unless is_valid_id($START_CLIENT_ID);
} else {
    $log->out('Start ClientID value is not specified. Using 0 as start value.');
    $START_CLIENT_ID = 0;   
}

if (defined $ONLY_SHARD_ID) {
    $log->die("Invalid only-shard-id value specified: $ONLY_SHARD_ID") unless is_valid_int($ONLY_SHARD_ID, 1, $Settings::SHARDS_NUM);
} else {
    $log->out("Only shard id value is not specified. Using 'all' as value.");
    $ONLY_SHARD_ID = 'all';
}
$log->out("Processing clients with ClientID greather than $START_CLIENT_ID.");
$log->out("Processing $ONLY_SHARD_ID shard(s).");


# закешируем себе хеш со списком стран, для валидации данных от баланса (из файлика)
my %countries;
foreach my $region (@geo_regions::COUNTRY_REGIONS) {
    $countries{ $region->{region_id} } = undef;
}


my %client_regions;
$log->out('Loading clients regions from file');

# /home/ppalex/deploy/client_region_one_file.tsv
open (my $clients_file, '<', '/tmp/client_region_one_file.tsv') or $log->die("Can't open file with regions: $!");
while (my $line = <$clients_file>) {
    chomp ($line);
    if ($line =~ m/^(\d+)\t(\d+)$/) {
        my ($ClientID, $region_id) = ($1, $2);
        # проверим ClientID
        unless (is_valid_id($ClientID)) {
            $log->out("Invalid ClientID in file: $ClientID");
        }
        # проверим страну
        if (exists $countries{$region_id}) {
            $client_regions{$ClientID} = $region_id;
        } else {
            $log->out("Invalid country $region_id specified for ClientID $ClientID");
        }
    } else {
        $log->out("Unrecognized format of line: $line");
    }
}
close ($clients_file);
$log->out('Loaded ' . (scalar keys %client_regions) . ' clients from file');


$log->out('Fetching clients from db');
my $db_clients = get_one_column_sql(PPC(shard => $ONLY_SHARD_ID), [
    'SELECT DISTINCT ClientID
       FROM users',
      WHERE => {ClientID__ge => $START_CLIENT_ID},
  'ORDER BY ClientID ASC', # можно не указывать
]) || [];
$log->out('Got ' . (scalar @$db_clients) . ' clients to process');


$log->out("Start processing clients");
# foreach_shard_parallel(ClientID => $db_clients, with_undef_shard => 1, chunk_size => 20_000, sub {
# плюшки с транзакционнностью никак не дружат с форканьем :(
foreach_shard(ClientID => $db_clients, with_undef_shard => 1, chunk_size => 20_000, sub {
    my ($shard, $chunk) = @_;

    if (!$shard) {
        # считаем, что метабаза и база консистентны.
        # в списке есть часть ClientID не из нашей базы.
        # в продакшене на момент написания скрипта база и метабаза расходятся только для 2866130
        foreach my $ClientID (@$chunk) {
            $log->out("Skipped ClientID: $ClientID - UNKNOWN SHARD!");
        }
    } else {
        $log->msg_prefix("shard $shard:");


        my $clients_data = get_hash_sql(PPC(shard => $shard), ['
               SELECT u.ClientID
                    , IFNULL(cl.country_region_id, 0) AS country_region_id
                 FROM users u
                      LEFT JOIN clients cl USING(ClientID)
             ', WHERE => {ClientID => $chunk},
            'GROUP BY ClientID',
        ]);
        
        foreach my $client_id (@$chunk) {
            # если за время работы деплоя клиента смерджили с кем-то или перенесли
            unless (exists $clients_data->{$client_id}) {
                $log->out("Skipped ClientID: $client_id - NO DATA FROM DB");
                next;
            }
            my $region_id = $clients_data->{$client_id};

            if ($region_id) {
                # Какая-то страна уже есть - все ок. формально сверимся со списком
                if (exists $client_regions{$client_id} && $client_regions{$client_id} != $region_id) {
                    # клиент есть в проверочном списке и страна не совпадает (такого быть не должно!)
                    $log->out("Skipped ClientID: $client_id - WARNING: DIFFERENT COUNTRY IN FILE ($client_regions{$client_id}) AND DB ($region_id)");
                    next;
                } else {
                    # или в проверочном списке такого клиента нет или совсем все ок - совпало
                    $log->out("Skipped ClientID: $client_id - already has country_region_id $region_id");
                    next;
                }
            } else {
                # Страны нет :(
                my $NEW_COUNTRY_ID;
                my $got_from;

                # применяем "наш дедуктивный метод" (c)
                $log->out("Fetching data for ClientID: $client_id from balance");
                my $fcc;
                # и еще одно заevalированное место кода
                my $res = eval {
                    $fcc = retry(tries => 2, pauses => [10], sub {
                        return BalanceWrapper::get_firm_country_currency($client_id, currency_filter => 1);
                    });
                    return 1;
                };
                if (!$res) {
                    # не получилось
                    $log->out("Skipped ClientID: $client_id - WARNING: ERROR FETCHING FIRM COUNTRY CURRENCY DATA: " . ($@ =~ s![\r\n]+! / !gr) );
                } elsif ($fcc && ref $fcc eq 'ARRAY') {
                    # looks good
                    if (@$fcc) {
                        my @possible_countries = uniq map { $_->{region_id} } @$fcc;
                        if (scalar(@possible_countries) == 1) {
                            # GOT IT!
                            $NEW_COUNTRY_ID = $possible_countries[0];
                            $got_from = 'balance';
                        } else {
                            $log->out("Skipped ClientID: $client_id - possible country is not unique");
                        }
                    } else {
                        $log->out("Skipped ClientID: $client_id - got empty country currency data");
                    }
                } else {
                    # impossible
                    $log->out("Skipped ClientID: $client_id - WARNING: INVALID FIRM COUNTRY CURRENCY DATA");
                }

                # общий next для всех скипов и веток выше
                next unless $NEW_COUNTRY_ID;

                # сравниваем с файликом от Баланса
                if (exists $client_regions{$client_id}) {
                    my $file_region_id = $client_regions{$client_id};
                    if ($NEW_COUNTRY_ID == $file_region_id) {
                        $got_from .= '+file';
                    } else {
                        $log->out("Skipped ClientID: $client_id WARNING: REGION $NEW_COUNTRY_ID FROM BALANCE DOESN'T MATCH REGION $file_region_id FROM BALANCE-GENERATED FILE");
                        next;
                    }
                }

                if (!exists $countries{$NEW_COUNTRY_ID}) {
                    # very strange...
                    $log->out("Skipped ClientID: $client_id - WARNING: REGION $NEW_COUNTRY_ID IS NOT A COUNTRY!");
                    next;
                }
                $log->out("Processing ClientID: $client_id - got country $NEW_COUNTRY_ID from $got_from");

                # уровень вложенности eval'ов зашкаливает...
                my $transaction_result = eval {
                    do_in_transaction {
                        # in direct
                        create_update_client({
                            client_data => {
                                ClientID            => $client_id,
                                country_region_id   => $NEW_COUNTRY_ID,
                                user_is_not_created => 1, # for updating last_change in users
                            },
                        });
                        # in balance    #UID
                        update_client_id(1, $client_id, {REGION_ID => $NEW_COUNTRY_ID}) == 0 or $log->die("error updating ClientID $client_id in balance");
                    };
                    return 1;
                };
                if (!$transaction_result) {
                    $log->out("Processing ClientID: $client_id REGION_ID: $NEW_COUNTRY_ID - FAILED: " . ($@ =~ s![\r\n]+! / !gr));
                } else {
                    $log->out("Processing ClientID: $client_id REGION_ID: $NEW_COUNTRY_ID - SUCCESS");
                }

            } #/end of else $region_id

        } #/end of foreach (@$chunk)

    }

});


$log->out('FINISH');
