#!/usr/bin/env perl

use my_inc "../..";

=head1 NAME

    shard_client_domains

=head1 SYNOPSIS

    shard_client_domains.pl
        Разложить по шардам в базу PPC таблицу MONITOR.client_domains

=head1 DESCRIPTION

    Разложить по шардам в базу PPC таблицу MONITOR.client_domains

    Параметры:
        --chunk-size N  => обрабатывать по N записей за раз (по умолчанию 100,000)
        --recreate      => пересоздать таблицы с нуля

=cut

use Direct::Modern;
use open ':std' => ':utf8';

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::DBSchema qw/create_table_by_schema/;
use Yandex::ListUtils qw/chunks/;
use List::MoreUtils qw/uniq all/;
use ScriptHelper;
use Yandex::MirrorsTools::Hostings ();
use Primitives qw/get_domain2domain_id/;

my ($CHUNK_SIZE, $RECREATE);
extract_script_params(
    "chunk-size=i" => \$CHUNK_SIZE,
    "recreate"     => \$RECREATE,
);

$CHUNK_SIZE ||= 100_000;
$log->out('START');

if ($RECREATE) {
    $log->out("Dropping old tables");

    do_sql(PPCDICT, q{DROP TABLE IF EXISTS inc_client_domains_record_id});
    do_sql(PPCDICT, q{DROP TABLE IF EXISTS inc_client_domains_sync_id});
    do_sql(PPC(shard => 'all'), q{DROP TABLE IF EXISTS client_domains});

    do_sql(PPCDICT, q{DROP TABLE IF EXISTS inc_client_domains_stripped_record_id});
    do_sql(PPC(shard => 'all'), q{DROP TABLE IF EXISTS client_domains_stripped});
}

# Предварительное создание таблиц
create_table_by_schema(PPCDICT, 'inc_client_domains_record_id', if_not_exists => 1);
create_table_by_schema(PPCDICT, 'inc_client_domains_sync_id', if_not_exists => 1);
create_table_by_schema(PPC(shard => 'all'), 'client_domains', if_not_exists => 1);

create_table_by_schema(PPCDICT, 'inc_client_domains_stripped_record_id', if_not_exists => 1);
create_table_by_schema(PPC(shard => 'all'), 'client_domains_stripped', if_not_exists => 1);

# Установим sync_id
my $last_sync_id = get_one_field_sql(MONITOR, q{SELECT max(sync_id) FROM client_urls_crm_queue});
do_sql(PPCDICT, q{ALTER TABLE inc_client_domains_sync_id AUTO_INCREMENT = } . ($last_sync_id + 1));

# Закешируем существующие данные из client_domains_stripped
my $CL_D_STRIPPED = get_hashes_hash_sql(PPC(shard => 'all'), q{
    SELECT CONCAT_WS(':', ClientID, domain_id) ukey, record_id, logtime FROM client_domains_stripped
});

# Копирование данных
my ($total, $min_rec_id, $max_rec_id) = get_one_line_array_sql(MONITOR, q{
    SELECT count(record_id), min(record_id), max(record_id) FROM client_domains
});
my $rows_growth = 0;
for (my $start_rec_id = $min_rec_id; $start_rec_id <= $max_rec_id; $start_rec_id += $CHUNK_SIZE) {
    my $rows = get_all_sql(MONITOR, q{
        SELECT cd.record_id, cd.ClientID, cd.camp_type, cd.domain, cd.removed, cd.logtime, q.sync_id
        FROM client_domains cd LEFT JOIN client_urls_crm_queue q ON (q.record_id = cd.record_id)
        WHERE cd.record_id >= ? AND cd.record_id < ?
        ORDER BY cd.record_id
    }, $start_rec_id, $start_rec_id + $CHUNK_SIZE);
    $rows_growth += scalar @$rows;

    $log->out("Processing ${rows_growth}/${total} records");

    # Разделим полученные данные по ClientID
    my $rows_by_clientid = {};
    push @{$rows_by_clientid->{$_->{ClientID}}}, $_ for @$rows;

    for my $chunk (sharded_chunks ClientID => [keys %$rows_by_clientid], with_undef_shard => 1) {
        my ($shard, $client_ids) = ($chunk->{shard}, $chunk->{ClientID});

        # Строки с привязкой к номеру шарда
        my @rows_sharded = map { @$_ } @$rows_by_clientid{@$client_ids};
        next unless @rows_sharded;

        if (!$shard) {
            $log->out("Found clients in zero-shard: ".join(',', @$client_ids));
            $log->out("Rows: ");
            $log->out(\@rows_sharded);
            next;
        }

        # Вычислим stripped домены
        my %domain2stripped = map { $_->{domain} => Yandex::MirrorsTools::Hostings::strip_domain($_->{domain}) } @rows_sharded;

        # Запросим id всех доменов
        my $domain2domain_id = get_domain2domain_id([uniq(%domain2stripped)]);

        # Вставим данные в таблицу client_domains
        my @fields = grep { $_ ne 'domain' } keys %{$rows_sharded[0]};
        do_mass_insert_sql(PPC(shard => $shard),
            'INSERT IGNORE INTO client_domains ('.join(', ', @fields).', domain_id) VALUES %s',
            [map { [@$_{@fields}, $domain2domain_id->{$_->{domain}}] } @rows_sharded]
        );
        $log->out("[shard:$shard] Insert in client_domains done");

        # Вставим данные в таблицу client_domains_stripped
        # Т.к. скрипт может перезапускаться, часть данных уже может существовать,
        # соответственно запрашивать каждый раз новые идентификаторы нельзя.

        # Соберем записи для вставки: не существовавшие ранее, или с временем logtime больше чем существующее
        my @domains_stripped_to_insert;
        for my $row (grep { $_->{camp_type} eq 'text' } @rows_sharded) {
            my $domain_id = $domain2domain_id->{$domain2stripped{$row->{domain}}};
            if (!$domain_id) {
                $log->die("No domain_id found for ".$row->{domain}." -> ".$domain2stripped{$row->{domain}});
            }
            my $ukey = join ':', $row->{ClientID}, $domain_id;
            my $record_id = exists $CL_D_STRIPPED->{$ukey} ? $CL_D_STRIPPED->{$ukey}->{record_id} : get_new_id('client_domains_stripped_record_id');

            unless (exists $CL_D_STRIPPED->{$ukey} && $CL_D_STRIPPED->{$ukey}->{logtime} ge $row->{logtime}) {
                push @domains_stripped_to_insert, [$record_id, $row->{ClientID}, $domain_id, $row->{logtime}];
                $CL_D_STRIPPED->{$ukey} = {record_id => $record_id, logtime => $row->{logtime}};
            }
        }

        do_mass_insert_sql(PPC(shard => $shard),q{
            INSERT INTO client_domains_stripped (record_id, ClientID, domain_id, logtime) VALUES %s
                ON DUPLICATE KEY UPDATE logtime = GREATEST(logtime, VALUES(logtime))
            }, \@domains_stripped_to_insert,
        );
        $log->out("[shard:$shard] Insert in client_domains_stripped done");
    }
}

# Обновим имена переменных из PPCDICT.ppc_properties
for my $prop (@{get_all_sql(PPCDICT, q{SELECT name FROM ppc_properties WHERE name LIKE 'PrepareDataForBalance%'})}) {
    my $new_name = $prop->{name} =~ s/^PrepareDataForBalance/PrepareClientDomains/r;
    do_sql(PPCDICT, "UPDATE ppc_properties SET name = ? WHERE name = ?", $new_name, $prop->{name});
}
do_sql(PPCDICT, q{UPDATE ppc_properties SET name = 'ExportDomainsToBalance_last_syncid' WHERE name = 'domains_for_balance_last_syncid'});

$log->out('FINISH');
