#!/usr/bin/perl

use strict;
use warnings;
use utf8;
use open ':std' => ':utf8';

use my_inc "../..";

use Yandex::DBTools;
use Yandex::DBShards;
use Settings;
use ScriptHelper;
use DBStat;
use List::Util qw/sum0/;
use Date::Calc qw/Add_Delta_YM Add_Delta_Days/;
use User;

my ($year, $out_file);
extract_script_params(
    "y|year=s" => \$year,
    "out-file=s" => \$out_file,
);

die "Usage: $0 --year=2014 --out-file=request.data" unless $year && $out_file;

my $requests = get_optimazed_requests($year);
my (%orders, %stats_before, %stats_after, %manager_uids);

foreach my $request (@$requests) {
    next if exists $orders{$request->{OrderID}};
    $orders{$request->{OrderID}} = $request;
    
    my $break_point;
    if ($request->{ManagerUID}) {
        $manager_uids{$request->{ManagerUID}} = 1;
        $break_point = $request->{serviced_date};
    } else {
        $break_point = $request->{ready_time};
    }
    $request->{break_point} = $break_point; 
    my ($y, $m, $d) = split '-', $break_point;
    my $from = join "-", Add_Delta_YM($y, $m, $d, 0, -6);
    push @{$stats_before{"${from}:${break_point}"}}, $request->{OrderID};
    
    my $to = join "-", Add_Delta_YM(Add_Delta_Days($y, $m, $d, 1), 0, 6);
    $break_point = join '-', Add_Delta_Days($y, $m, $d, 1);
    push @{$stats_after{"${break_point}:${to}"}}, $request->{OrderID};
}

while (my ($period, $order_ids) = each %stats_before) {
    my ($from, $to) = split ':', $period;
    my $orders_spent = get_orders_spent($order_ids, $from, $to);
    while (my ($order_id, $cost) = each %$orders_spent) {
        $orders{$order_id}->{cost_before} = $cost;
    } 
} 
while (my ($period, $order_ids) = each %stats_after) {
    my ($from, $to) = split ':', $period;
    my $orders_spent = get_orders_spent($order_ids, $from, $to);
    while (my ($order_id, $cost) = each %$orders_spent) {
        $orders{$order_id}->{cost_after} = $cost;
    } 
}

my $managers = get_users_data([keys %manager_uids], ['login']);

open FILE, '>', $out_file or die $!;
print FILE join(';', 'кампания №', 'тип помощи', 'логин менеджера',
    'средние ежемесячные траты до ПП, у.е.', 'средние ежемесячные траты после ПП, у.е.') . "\n";
my %req_types = (
    FirstAid => 'Первая помощь',
    SecondAid => 'Вторая помощь',
    Moderate => 'Первая помощь с модерации'
);    
foreach my $order (sort { $b->{cost_after} <=> $a->{cost_after} } values %orders) {
    my @line = ($order->{cid}, $req_types{$order->{req_type}});
    push @line, $order->{ManagerUID} ? $managers->{$order->{ManagerUID}}->{login} : '';
    push @line, sprintf("%0.2f", $order->{cost_before}), sprintf("%0.2f", $order->{cost_after});  
    print FILE join(";", @line) . "\n";     
}
close FILE;

sub get_optimazed_requests {
    
    my ($year) = @_;
    
    my $requests = get_all_sql(PPC(shard => 'all'), q[
        SELECT
            c.cid, c.ManagerUID, c.OrderID, IFNULL(c.currency, "YND_FIXED") AS currency,
            DATE(IFNULL(r.accept_time, r.ready_time)) AS serviced_date,
            DATE(r.ready_time) AS ready_time,
            r.req_type
        FROM 
            optimizing_campaign_requests r
            JOIN campaigns c USING(cid)
        WHERE
            YEAR(r.ready_time) = ?
            AND c.OrderID > 0
        ORDER BY serviced_date
    ], $year);
    
    return $requests;
}

sub get_orders_spent {
    
    my ($order_ids, $from, $to) = @_;
    
    my $currencies = get_hash_sql(PPC(OrderID => $order_ids), [ '
                                            SELECT OrderID, IFNULL(currency, "YND_FIXED")
                                              FROM campaigns', 
                                             WHERE => {OrderID => SHARD_IDS}
                                         ]);
    # разбиваем по флагу with_nds
    my %by_nds;
    for my $oid (@$order_ids) {
        my $with_nds = $currencies->{$oid} eq 'YND_FIXED' ? 1 : 0;
        push @{$by_nds{$with_nds}}, $oid;
    }

    my %ret;
    for my $with_nds (keys %by_nds) {
        my @oids = @{$by_nds{$with_nds}};
        my $stat = eval {
            my %opts = (with_discount => 1, with_nds => $with_nds, single_currency => 1, currency => 'YND_FIXED');
            DBStat->get_orders_stat_date_fetch_bs([map {+{OrderID => $_}} @oids], $from, $to, %opts);
        };
        if (!defined $stat) {
            warn "error getting statistics: $@", 'get_order_spent_today error';
        } else {
            for my $oid (@oids) {
                # 6 months
                $ret{$oid} = sum0(map {$_->{Cost}} values %{$stat->{$oid}}) / 6;
            }
        }
    }
    return \%ret;
}
