#!/usr/bin/perl

use my_inc '../..';

=head1 NAME

    stop_and_delete_disabled_phrases.pl

=SYNOPSYS

    scp ppcdev2.yandex.ru:/home/ppalex/deploy/all_disabled_phrases.tsv /tmp
    /var/www/ppc.yandex.ru/protected/one-shot/stop_and_delete_disabled_phrases.pl --sleep-coeff=1 --childs=5


=head2 DESCRIPTION

    Скрипт для остановки и удаления фраз, отключенных (на поиск и везде) за низкий CTR.
    Работает на основе TSV-файла с OrderID BannerID PhraseID dis_search dis_flat, лежащего по пути
        /tmp/all_disabled_phrases.tsv

    Фразы, отключенные только на поиске - выключает, кроме кампаний со стратегией "независимое размещение: показы на поиске отключены"
    Фразы, отключенные полностью - удаляет (включая архивные, параметры фраз, ручные цены).
    Перед отключением или удалением данные логируются.

    В результате своей работы скрипт создает два лога:
        * phrases_to_remove_from_bs.tsv.YYYYMMDD_HHMMSS - файл со списком OrderID BannerID PhrasedID,
                                                            который нужно передать в БК для удалениея фраз по нему
        * data_for_resync.tsv.YYYYMMDD_HHMMSS           - файл для ленивой переотправки тех баннеров, в которых были отключены фразы

    
    Принимаемые параметры:
        --sleep-coeff=N     - коэффициента сна (во сколько раз дольше спать после обработки одного заказа)
                                по-умолчанию 1, может быть указан 0.
        --dry-run           - только выборка и логирование данных, без изменений в базе.
        --childs=MM         - максимальное количество параллельно работающих потоков в одном шарде
                                разные шарды обрабатываются параллельно (при 4 шардах и 5 потоках - в пике будет +20 форков)
                                по-умолчанию 5, допустимые значения - от 1 до 20

=cut

use warnings;
use strict;
use utf8;

use List::MoreUtils qw/uniq/;
use Parallel::ForkManager;
use POSIX qw/strftime/;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ListUtils;
use Yandex::Retry;
use Yandex::Validate;

use Settings;
use EnvTools;
use Primitives ();
use ScriptHelper;

use constant RESYNC_PRIORITY        => -1;
use constant ORDERS_PER_CHUNK       => 10_000;

my ($SLEEP_COEFF, $DRY_RUN, $MAX_CHILDS_PER_SHARD);
extract_script_params(
    'sleep-coeff=i' => \$SLEEP_COEFF,
    'dry-run!'      => \$DRY_RUN,
    'childs=i'      => \$MAX_CHILDS_PER_SHARD,
);
$SLEEP_COEFF //= 1;
$MAX_CHILDS_PER_SHARD //= 5;

unless (is_valid_int($MAX_CHILDS_PER_SHARD, 1, 20)) {
    die 'invalid "childs" value - should be between 1 and 20';
}

$log->out('START');
my $date_suf = strftime("%Y%m%d_%H%M%S", localtime(time));

my $filename = is_beta() ? '/home/ppalex/dis_phrases/all_disabled_phrases.tsv' : '/tmp/all_disabled_phrases.tsv';
open (my $phrases_list_file, '<', $filename) or $log->die("Can't open file with disabled phrases: $!");
$log->out('Opened file with all disabled phrases');
open (my $log_for_bs, '>', "$Yandex::Log::LOG_ROOT/phrases_to_remove_from_bs.tsv.$date_suf") or $log->die("Can't open file for bs log: $!");
$log->out("Opened file phrases_to_remove_from_bs.tsv.$date_suf for bs data");
open (my $resync_file, '>', "$Yandex::Log::LOG_ROOT/data_for_resync.tsv.$date_suf") or $log->die("Can't open file for resync: $!");
$log->out("Opened file data_for_resync.tsv.$date_suf for resync data");

my %orders;
=pod Структура хеша %orders

    OrderID1 => {
        all => [
            [BannerID1, PhraseID1],
            [BannerID1, PhraseID2],
            [BannerID2, PhraseID3],
        ],
        search => [
            [BannerID2, PhraseID4],
            [BannerID3, PhraseID5],
        ],
    },
    OrderID2 => {
        ...
    },

=cut

$log->out('Loading data from file');
while (my $line = <$phrases_list_file>) {
    chomp($line);
    if (my ($OrderID, $BannerID, $PhraseID, $dis_search, $dis_flat) = $line =~ m/^(\d+)\t(\d+)\t(\d+)\t(\d+)\t(\d+)$/) {
        next unless ($dis_search && !$dis_flat || $dis_search && $dis_flat);

        $orders{$OrderID} //= {all => [], search => []};
        # отключены везде
        push @{ $orders{$OrderID}->{all} }, [$BannerID, $PhraseID] if $dis_search && $dis_flat;
        # отключены только на поиске
        push @{ $orders{$OrderID}->{search} }, [$BannerID, $PhraseID] if $dis_search && !$dis_flat;
    } else {
        $log->out("Skipping bad line in phrases file: $line");
    }
}
close ($phrases_list_file) or $log->warn("Can't close file with disabled phrases: $!");
my $total_orders_count = scalar keys %orders;
$log->out("Loaded $total_orders_count orders");

my %stats = (
    suspended_bids                  => 0,
    suspended_bids_arc              => 0,
    deleted_bids                    => 0,
    deleted_bids_arc                => 0,
    deleted_bids_href_params        => 0,
    deleted_bids_manual_prices      => 0,
);

# пытаемя логировать ошибки из форков.
local $SIG{__DIE__} = sub { $log->out('DIED!'); $log->die(@_) };

foreach_shard_parallel OrderID => [keys %orders], with_undef_shard => 1, sub {
    my ($shard, $orders_in_one_shard) = @_;

    if (!$shard && @$orders_in_one_shard) {
        Yandex::Trace::restart(\$ScriptHelper::trace, tags => 'shard_undef');
        $log->out("Skipping OrderID: $_ - no shard!") for @$orders_in_one_shard;

    } else {
        my $pm_in_shard = Parallel::ForkManager->new($MAX_CHILDS_PER_SHARD);

        my @chunks = chunks($orders_in_one_shard, ORDERS_PER_CHUNK);
        my ($cur, $total) = (0, scalar(@chunks));
        for my $orders_chunk (@chunks) {
            ++$cur;
            $pm_in_shard->start and next;

            $0 .= " - chunk $cur/$total";
            Yandex::Trace::restart(\$ScriptHelper::trace, tags => "shard_$shard,chunk_$cur");
            $log->msg_prefix("[$shard: $cur/$total]");
            
            # собственно работа с чанком заказов в известном шарде
            my $orderid2cid = Primitives::get_orderid2cid(OrderID => $orders_chunk);

            my ($i, $orders_chunk_size) = (0, scalar(@$orders_chunk));
            foreach my $OrderID (@$orders_chunk) {
                ++$i;
                my $data = $orders{$OrderID};
                relaxed times => $SLEEP_COEFF, sub {
                    my $cid = $orderid2cid->{$OrderID};
                    unless ($cid) {
                        $log->out("No cid for OrderID: $OrderID, skipping");
                        next;
                    }

                    $log->out("Processing campaign $cid (OrderID: $OrderID) [$i/$orders_chunk_size]");

                    my @pids_to_resync; # кому будем сбрасывать statusBsSynced 

                    # отключенные только на поиске
                    if (@{ $data->{search} }) {
                        # проверяем стратегию на кампании. считаем, что время между выборкой и обновлением - минимально
                        my $need_stop_phrases = get_one_field_sql(PPC(shard => $shard), [
                            'SELECT NOT(strategy = "different_places" AND platform = "context") AS need_stop_phrases
                            FROM campaigns
                            JOIN camp_options USING(cid)',
                            WHERE => {cid => $cid}
                        ]);

                        if ($need_stop_phrases) {
                            # получаем текущее состояние bids (только для включенных фраз)
                            my $old_data_bids = get_all_sql(PPC(shard => $shard), ['
                                 SELECT bi.id
                                      , bi.is_suspended
                                      , bi.PhraseID
                                      , b.bid
                                      , b.BannerID
                                      , p.pid
                                   FROM phrases p
                                        JOIN banners b ON b.pid = p.pid
                                        JOIN bids bi ON bi.pid = p.pid
                               ', WHERE => _get_condition_for_select_phrases_to_suspend('bi', $cid, $data->{search}, 'only_enabled')
                            ]);
                            $log->out({old_data => $old_data_bids, cid => $cid, OrderID => $OrderID, type => 'is_suspended_bids'});

                            # и bids_arc (аналогично - только включенные фразы)
                            my $old_data_bids_arc = get_all_sql(PPC(shard => $shard), ['
                                 SELECT bia.id
                                      , bia.is_suspended
                                      , bia.PhraseID
                                      , b.bid
                                      , b.BannerID
                                      , p.pid
                                   FROM phrases p
                                        JOIN banners b ON b.pid = p.pid
                                        JOIN bids_arc bia ON bia.cid = p.cid AND bia.pid = p.pid
                               ', WHERE => _get_condition_for_select_phrases_to_suspend('bia', $cid, $data->{search}, 'only_enabled')
                            ]);
                            $log->out({old_data => $old_data_bids_arc, cid => $cid, OrderID => $OrderID, type => 'is_suspended_bids_arc'});

                            unless ($DRY_RUN) {
                            # теперь отключаем
                            # в bids (PRIMARY KEY: id)
                                my @ids_to_suspend = map { $_->{id} } (@$old_data_bids, @$old_data_bids_arc);
                                my $res_bids = do_in_transaction {
                                    do_update_table(PPC(shard => $shard),
                                        bids => {is_suspended => 1},
                                        where => {id => \@ids_to_suspend},
                                    );
                                };

                                $stats{suspended_bids} += $res_bids;
                                $log->out("Disabling phrases on campaign $cid (OrderID: $OrderID) in bids: affected $res_bids rows");

                                # в bids_arc (PRIMARY KEY: cid-pid-id), отключаем по cid и парам pid<->id
                                my $res_bids_arc = do_update_table(PPC(shard => $shard), 'bids_arc', {is_suspended => 1}, where => {
                                    cid => $cid,
                                    _OR => [
                                            map {( _AND => { pid => $_->{pid}, id => $_->{id} } )} (@$old_data_bids, @$old_data_bids_arc)
                                        ],
                                });
                                $stats{suspended_bids_arc} += $res_bids_arc;
                                $log->out("Disabling phrases on campaign $cid (OrderID: $OrderID) in bids_arc: affected $res_bids_arc rows");
                            }

                            # нужно занести все в лог, чтобы передать для "ручного" отключения в БК, так как штатным транспортом может и не отправиться.
                            for my $r (@{ $data->{search} }) {
                                print {$log_for_bs} join("\t", $OrderID, $r->[0], $r->[1]), "\n";
                            }
                            # и про запас подготовить себе файл для ленивой переотправки
                            for my $r ( xuniq { join('_', $_ ->{bid}, $_->{pid}) } (@$old_data_bids, @$old_data_bids_arc) ) {
                                print {$resync_file} join("\t", $cid, $r->{bid}, $r->{pid}, RESYNC_PRIORITY), "\n";
                            }

                            push @pids_to_resync, map { $_->{pid} } (@$old_data_bids, @$old_data_bids_arc);

                        } else {
                            $log->out("Skip suspending phrases in campaign $cid (OrderID $OrderID) due to strategy");
                        }
                    } # конец работы с фразами, отключенными только на поиске

                    # отключенные везде - удаляем(!)
                    if (@{ $data->{all} }) {
                        # делаем полную выборку, чтобы иметь возможность откатиться
                        # bids
                        my $old_data_bids = get_all_sql(PPC(shard => $shard), ['
                             SELECT bi.*
                                  , b.bid AS b_bid
                                  , b.BannerID
                               FROM phrases p
                                    JOIN banners b ON b.pid = p.pid
                                    JOIN bids bi ON bi.pid = b.pid
                           ', WHERE => _get_condition_for_select_phrases_to_suspend('bi', $cid, $data->{all}, undef)
                        ]);
                        $log->out({old_data => $old_data_bids, cid => $cid, OrderID => $OrderID, type => 'bids'});

                        # bids_arc
                        my $old_data_bids_arc = get_all_sql(PPC(shard => $shard), ['
                             SELECT bia.*
                                  , b.bid AS b_bid
                                  , b.BannerID
                               FROM phrases p
                                    JOIN banners b ON b.pid = p.pid
                                    JOIN bids_arc bia ON bia.cid = p.cid AND bia.pid = b.pid
                           ', WHERE => _get_condition_for_select_phrases_to_suspend('bia', $cid, $data->{all}, undef)
                        ]);
                        $log->out({old_data => $old_data_bids_arc, cid => $cid, OrderID => $OrderID, type => 'bids_arc'});


                        # начинаем удалять..
                        # сами фразы удаляем в последнюю очередь, чтобы при падениях иметь возможность
                        # перезапустить и доудалить остальные данные (они удаляются по выборке из bids*)
                        my $ids = [ map { $_->{id} } (@$old_data_bids, @$old_data_bids_arc) ];

                        # bids_href_params (PRIMARY KEY: id)
                        my $old_data_bids_href_params = get_all_sql(PPC(shard => $shard), [
                            'SELECT * FROM bids_href_params ', WHERE => { cid => $cid, id => $ids }
                        ]);
                        $log->out({old_data => $old_data_bids_href_params, cid => $cid, OrderID => $OrderID, type => 'bids_href_params'});
                        if (@$old_data_bids_href_params && !$DRY_RUN) {
                            my $res_bids_href_params = do_delete_from_table(PPC(shard => $shard), 'bids_href_params', where => { cid => $cid, id => $ids });
                            $stats{deleted_bids_href_params} += $res_bids_href_params;
                            $log->out("Deleted $res_bids_href_params rows from bids_href_params on campaign $cid (OrderID: $OrderID)");
                        }

                        # bids_manual_prices (PRIMARY KEY: cid-id)
                        my $old_data_bids_manual_prices = get_all_sql(PPC(shard => $shard), [
                            'SELECT * FROM bids_manual_prices', WHERE => { cid => $cid, id  => $ids }
                        ]);
                        $log->out({old_data => $old_data_bids_manual_prices, cid => $cid, OrderID => $OrderID, type => 'bids_manual_prices'});
                        if (@$old_data_bids_manual_prices && !$DRY_RUN) {
                            my $res_bids_manual_prices = do_delete_from_table(PPC(shard => $shard), 'bids_manual_prices', where => {
                                cid => $cid,
                                id  => $ids,
                            });
                            $stats{deleted_bids_manual_prices} += $res_bids_manual_prices;
                            $log->out("Deleted $res_bids_manual_prices rows from bids_manual_prices on campaign $cid (OrderID: $OrderID)");
                        }

                        # bids_phraseid_history (PRIMARY KEY: cid-id)
                        # bsClearIdHistory.pl почистит сам

                        # bids_phraseid_associate (PRIMARY KEY: cid-pid-PhraseID-logtime)
                        # оставляем, нужна для матчинга статистики в старые ID'шники

                        # bs_auction_stat (PRIMARY KEY: pid-PhraseID)
                        # тоже удалится из bsClearIdHistory.pl

                        # bs_auction_stat_context (PRIMARY KEY: pid-PhraseID)
                        # тоже удалится из bsClearIdHistory.pl

                        # bs_auction_stat_off (PRIMARY KEY: pid-PhraseID)
                        # удалим всю таблицу целиком

                        # собственно фразы
                        unless ($DRY_RUN) {
                            # # bids (PRIMARY KEY: id)
                            my $res_bids = do_in_transaction {
                                do_delete_from_table(PPC(shard => $shard), 'bids_base', where => {bid_id => $ids});
                                do_delete_from_table(PPC(shard => $shard), 'bids', where => {id => $ids});
                            };
                            $stats{deleted_bids} += $res_bids;
                            $log->out("Deleted $res_bids rows from bids on campaign $cid (OrderID: $OrderID)");

                            # bids_arc (PRIMARY KEY: cid-pid-id)
                            my $res_bids_arc = do_delete_from_table(PPC(shard => $shard), 'bids_arc', where => {
                                cid => $cid,
                                _OR => [
                                        map {( _AND => { pid => $_->{pid}, id => $_->{id} } )} (@$old_data_bids, @$old_data_bids_arc)
                                    ],
                            });
                            $stats{deleted_bids_arc} += $res_bids_arc;
                            $log->out("Deleted $res_bids_arc rows from bids_arc on campaign $cid (OrderID: $OrderID)");
                        }

                        push @pids_to_resync, map { $_->{pid} } (@$old_data_bids, @$old_data_bids_arc);

                    } # конец работы с полностью отключенными фразами

                    if (@pids_to_resync) {
                        @pids_to_resync = uniq @pids_to_resync;
                        # в banners сбрасываем - пытаясь подстраховаться от "потерянных" ресурсов баннера в БК
                        $log->out({pids_to_bs_resync => \@pids_to_resync, cid => $cid, OrderID => $OrderID});
                        unless ($DRY_RUN) {
                            do_update_table(PPC(shard => $shard), 'banners', {statusBsSynced => 'No'}, where => { pid => \@pids_to_resync });
                            do_update_table(PPC(shard => $shard), 'phrases', {statusBsSynced => 'No'}, where => { pid => \@pids_to_resync });
                        }
                    }
                }; # end of relaxed sub
            } # end of while

            $log->out(\%stats);
            $log->out('chunk done');
            $pm_in_shard->finish;
        }

        # дожидаемся всех детей-форков для текущего шарда
        $pm_in_shard->wait_all_children;
        $log->out("shard $shard done");
    }
};

$log->out('FINISH');

exit 0;


###################################################
sub _get_condition_for_select_phrases_to_suspend {
    my ($table_alias, $cid, $rows, $only_enabled) = @_;
    return {
        'p.cid'      => $cid,
        _OR => [
            map { (_AND => { 'b.BannerID' => $_->[0], "$table_alias.PhraseID" => $_->[1] }) } @$rows
        ],
        ($only_enabled ? (is_suspended => 0) : ()),
    };
}
