#!/usr/bin/env perl

=head1 NAME

unservice_disabled_wallets - снять отключенные кошельки с сервисирования менеджером по списку wallet_cid

=head1 SYNOPSIS

/var/www/ppc.yandex.ru/protected/one-shot/unservice_disabled_wallets.pl --data-file=/var/www/ppc.yandex.ru/deploy/20190430_unservice_wallets.data

=head1 DESCRIPTION

    Вывести кошельки из-под менеджера при выполнении нескольких условий:
    1. под кошельком нет кампаний
    2. у клиента есть кампании кроме кошельков
    3. у клиента нет сервисируемых кампаний кроме кошельков

Параметры:

    --help — показать справку из описания,
    --data-file=s — обязательный путь до файла со списком wallet_cid с произвольным разделителем (например, LF),
    --dry-run — (необязательный) не делать изменения в БД.

=cut

use Direct::Modern;

use Yandex::DBShards;
use Yandex::DBTools;

use List::MoreUtils qw /uniq/;
use Path::Tiny;

use my_inc '../..';

use Campaign qw/campaign_manager_changed/;
use RBACDirect;
use ScriptHelper;
use Settings;


$log->out('START');

my ($DRY_RUN, $PATH);
extract_script_params(
    'dry-run' => \$DRY_RUN,
    'data-file=s' => \$PATH,
);

$log->die("No data-file option wqas given") unless $PATH;
my $wallets_data = path($PATH)->slurp;

=head3 @wallets

список кошельков для обработки можно получить так:
SELECT wc.cid FROM
(SELECT wc.cid, wc.clientid clientid FROM campaigns wc
    LEFT JOIN campaigns c ON wc.clientid = c.clientid AND wc.cid = c.wallet_cid
        WHERE wc.type = "wallet" AND wc.manageruid <> 0 GROUP BY wc.cid, wc.clientid
        HAVING count(c.cid) = 0) wc
LEFT JOIN campaigns c ON wc.clientid = c.clientid AND c.manageruid <> 0 AND wc.cid <> c.cid GROUP BY wc.cid
    HAVING count(c.cid) = 0

=cut

my @wallets = split /\D+/, $wallets_data;

# запросы одноразовые, не усложняю запросом всего сразу
my $wallet2clientid = get_hash_sql(PPC(cid => \@wallets), ["SELECT cid, clientid FROM campaigns", where => {type => 'wallet', cid => SHARD_IDS}]);
my $wallet2manageruid = get_hash_sql(PPC(cid => \@wallets), ["SELECT cid, manageruid FROM campaigns", where => {type => 'wallet', cid => SHARD_IDS, manageruid__is_not_null => 1}]);
my @clientids = uniq values %$wallet2clientid;

# таких быть не должно, если только клиент не создал сервисируемые кампании недавно
my $under_wallet = get_hash_sql(PPC(cid => \@wallets), ["SELECT wallet_cid, COUNT(*) FROM campaigns", where => {wallet_cid => SHARD_IDS}, "GROUP BY wallet_cid"]);

# нас не должны интересовать клиенты, у которых только кошельки; считаем другие кампании
my $under_client = get_hash_sql(PPC(ClientID => \@clientids), ["SELECT clientid, COUNT(*) FROM campaigns", where => {clientid => SHARD_IDS, type__ne => 'wallet'}, "GROUP BY clientid"]);

# при этом у клиента не должно быть сервисируемых кампаний (кроме кошельков)
my $serviced_under_client = get_hash_sql(PPC(ClientID => \@clientids), ["SELECT clientid, COUNT(*) FROM campaigns", where => {clientid => SHARD_IDS, type__ne => 'wallet',
    manageruid__is_not_null => 1}, "GROUP BY clientid"]);

my %totals = (skipped => 0, failed => 0, unserviced => 0);

for my $cid (@wallets) {
    my $g = $log->msg_prefix_guard("[Wallet cid $cid]");

    if (!$wallet2clientid->{$cid}) {
        $totals{skipped}++;
        $log->out("SKIPPING: Wallet not found");
        next;
    }
    my $clientid = $wallet2clientid->{$cid};

    # аналогично rbac_is_scampaign
    if (!$wallet2manageruid->{$cid}) {
        $totals{skipped}++;
        $log->out("SKIPPING: Wallet is not serviced");
        next;
    }

    if ($under_wallet->{$cid}) {
        $totals{skipped}++;
        $log->out("SKIPPING: Wallet is enabled");
        next;
    }

    if (!$under_client->{$clientid}) {
        $totals{skipped}++;
        $log->out("SKIPPING: Client $clientid has no nonwallet campaigns");
        next;
    }

    if ($serviced_under_client->{$clientid}) {
        $totals{skipped}++;
        $log->out("SKIPPING: Client $clientid has serviced nonwallet campaigns");
        next;
    }

    if (!$DRY_RUN) {
        eval {
            Campaign::unservice_orders_in_balance([$cid]);
        };
        if ($@) {
            $totals{failed}++;
            $log->out("ERROR: Unservice wallet in Balance error " . $@);
            next;
        }
    }
    $log->out("Successfully unserviced in Balance");

    if (!$DRY_RUN) {
        my $rbac_error = rbac_move_scamp_to_nscamp(undef, $cid);
        if ($rbac_error) {
            $totals{failed}++;
            $log->out("ERROR: rbac_move_scamp_to_nscamp error $rbac_error");
            next;
        }
    }
    $log->out("Successfully unserviced in Direct");

    if (!$DRY_RUN) {
        campaign_manager_changed(undef, 0, $cid, 0);
        do_sql(PPC(cid => $cid), "UPDATE campaigns SET statusBsSynced = 'No' WHERE cid = ?", $cid);
    }

    $totals{unserviced}++;
    $log->out("Done");
}

$log->out("Unserviced: " . $totals{unserviced} . ", failed: " . $totals{failed} . ", skipped: " . $totals{skipped});

$log->out('FINISH');
