#!/usr/bin/env perl

use my_inc "../..";

=head1 NAME

    unservice_empty_wallets

=head1 SYNOPSIS

    unservice_empty_wallets.pl --chunk-size=100 --sleep=10
        Снимать за раз по 100 кампаний, делая паузы в 10 секунд.

=head1 DESCRIPTION

    Снятие с сервисирования кампании-кошельки, внутри которых отсутствуют не архивные сервисируемые кампании

=cut

use strict;
use warnings;
use utf8;

use Settings;
use Yandex::DBTools;
use ScriptHelper;
use Yandex::ListUtils qw/chunks/;
use ShardingTools;
use List::MoreUtils qw/uniq/;

use Campaign;
use RBAC2::Extended;
use RBACDirect;

my ($CHUNK_SIZE, $SLEEP);

extract_script_params(
    "chunk-size=i"  => \$CHUNK_SIZE,
    "sleep=i"       => \$SLEEP,
);

usage() && exit unless $CHUNK_SIZE && $SLEEP;

$log->out('START');

my $rbac;
eval { $rbac = RBAC2::Extended->get_singleton(1); } or do { $log->die("Can't connect to RBAC: $@"); };

my $quit = 0;

$SIG{INT} = sub {
    $log->out('SIGINT RECEIVED');
    $quit = 1;
};

$SIG{TERM} = sub {
    $log->out('SIGTERM RECEIVED');
    $quit = 1;
};

my $total_wallets = 0;
my %updated_wallets;
for my $shard (ppc_shards()) {
    # Выберем сервисируемые кошельки с несервисируемыми кампаниями внутри
    my $broken_wallets = get_all_sql(PPC(shard => $shard), [q/
        SELECT wc.cid wallet_cid, wc.ManagerUID, count(IF(c.ManagerUID > 0, 1, NULL)) serv_count, count(IF(c.ManagerUID > 0, NULL, 1)) unserv_count
        FROM campaigns c
        JOIN campaigns wc ON (wc.cid = c.wallet_cid)
        WHERE
            c.archived = 'No' AND
            c.type = 'text' AND
            wc.ManagerUID > 0
        GROUP BY wc.cid
        HAVING serv_count = 0 AND unserv_count > 0
       /,
    ]);

    $total_wallets += scalar(uniq map { $_->{wallet_cid} } @$broken_wallets);

    foreach my $chunk (chunks($broken_wallets, $CHUNK_SIZE)) {
        for my $row (@$chunk) {
            my $manager_uid = $row->{ManagerUID};
            my $wallet_cid = $row->{wallet_cid};

            next if $updated_wallets{$wallet_cid};
            $log->out(qq/Unservice wallet $wallet_cid (Manager: $manager_uid)/);

            eval {
                if (rbac_is_scampaign($rbac, $wallet_cid)) {
                    my $rbac_error = rbac_move_scamp_to_nscamp($rbac, $wallet_cid);
                    die "rbac_move_scamp_to_nscamp(\$rbac, $wallet_cid) == $rbac_error" if $rbac_error;
                } else {
                    $log->out(qq/Wallet $wallet_cid doesnt serviced by any manager in RBAC/);
                }

                do_update_table(PPC(shard => $shard), 'campaigns', {statusBsSynced => 'No'}, where => {cid => $wallet_cid});
                Campaign::campaign_manager_changed($rbac, $manager_uid, $wallet_cid, 0);

                $updated_wallets{$wallet_cid}++;

                1;
            } or do {
                $log->out("Error: $@");
            };

            last if $quit;
        }

        last if $quit;
        sleep $SLEEP;
    }

    last if $quit;
}

$log->out("Found: $total_wallets wallets \tUpdated: ".scalar(keys %updated_wallets));
$log->out('FINISH');
