#!/usr/bin/perl

=head1 DESCRIPTION

   https://st.yandex-team.ru/DIRECT-103450
   Миграция выставляет attribution_model = 'last_yandex_direct_click':
     - для кампаний (text, dynamic, mobile_content performance) c неконверсионными стратегиями
     - для всех медийных кампаний. Типы: cpm_banner cpm_deals cpm_yndx_frontpage cpm_price
     - для всех кликовых с типами: content_promotion mcbanner mcb geo
     - для внутренней рекламы: internal_free internal_distrib
     - для всех кошельков. Ни на что не влияет. Новые будут созадваться уже с LYDC

=head1 USAGE

    Запуск в холостом режиме, без записи в базу
    LOG_TEE=1 ./protected/one-shot/update_attribution_model_for_cpc_strategies.pl --dry-run [--shard 1 ]

    Запуск с записью в базу
    LOG_TEE=1 ./protected/one-shot/update_attribution_model_for_cpc_strategies.pl [-- shard 1]
=cut

use Direct::Modern;

use Try::Tiny;

use Yandex::DBShards;
use Yandex::DBTools;

use my_inc '../..', for => 'protected';
use BS::ResyncQueue;
use ScriptHelper 'Yandex::Log' => [ date_suf => '%Y%m%d', auto_rotate => 1, tee => $ENV{LOG_TEE}, lock => 1 ];
use Settings;
use ShardingTools;

my $DRY_RUN;
my $SHARD;
my $CAMPAIGNS_CHUNK_SIZE = 10000;

my $ONE_ITERATION = undef;

extract_script_params(
    'dry-run'          => \$DRY_RUN,
    'one-iteration'    => \$ONE_ITERATION,
    'shard:i'          => \$SHARD,
);

$log->out("start");
my @shards_to_process = $SHARD ? ($SHARD) : ppc_shards();

my $update_conditions = [
    _OR => [
        type__in => [ qw/cpm_banner cpm_deals cpm_yndx_frontpage cpm_price content_promotion wallet mcbanner mcb geo internal_free internal_distrib/ ],
        _AND     => {
            type__in                  => [ qw/text dynamic mobile_content performance/ ],
            strategy_name__not_in     => [ qw/autobudget_avg_cpa autobudget_avg_cpi autobudget_roi autobudget_avg_cpa_per_filter autobudget_avg_cpa_per_camp/ ],
        },
        _AND     => {
            type__in      => [ qw/text dynamic mobile_content/ ],
            strategy_name => 'autobudget_week_bundle',
            _TEXT         => 'json_extract(strategy_data, "$.goal_id") is null',
        }
    ]
];

foreach_shard ( shard => [@shards_to_process], sub {
    my ($shard) = @_;
    my $msg_prefix_guard = $log->msg_prefix_guard("[shard $shard]");

    my ($min_cid, $max_cid) = get_one_line_array_sql(PPC(shard => $shard), "select min(cid), max(cid) from campaigns");

    while($min_cid < $max_cid) {

        $log->out("fetching from $min_cid to $min_cid + $CAMPAIGNS_CHUNK_SIZE ");
        my $campaigns_batch = get_all_sql(PPC(shard => $shard), ["
            SELECT
             cid,
             type,
             strategy_name,
             strategy_data,
             attribution_model
            FROM
              campaigns",
            WHERE => {
                cid__between => [$min_cid, $min_cid+$CAMPAIGNS_CHUNK_SIZE-1],
                @$update_conditions
            },
        ]);

        $log->out("fetched rows: " . scalar @$campaigns_batch) ;
        $log->out({campaigns_batch => $campaigns_batch});

        $min_cid += $CAMPAIGNS_CHUNK_SIZE;
        next unless @$campaigns_batch;

        my @cids = map { $_->{cid}} @$campaigns_batch;
        $log->out("update cids: @cids");

        unless($DRY_RUN) {
            do_sql(PPC(shard => $shard), [ "update campaigns
                set
                   attribution_model = 'last_yandex_direct_click'",
                "where" => {
                    cid => \@cids,
                    @$update_conditions
                }]);

            $log->out( { campaigns_to_resync => \@cids } );
            my $count = bs_resync_camps(\@cids, priority => 85);
            $log->out("$count rows added to resync queue");
        }
    }
    $log->out("finished");
});
$log->out("all shards finished");

