#!/usr/bin/perl

use my_inc "../..";

=head1 DESCRIPTION

    Синхронизирует с конструктором все турболендинги, у которых не проставлены конверсионные цели.
    Если у турболендинга нет конверсионных целей - будет вытягивать его при каждом запуске.

    Нужен для проставления признака конверсионности целям тех турболендингов,
    которые не обновлялись после выкладки https://st.yandex-team.ru/DIRECT-90833

    LOG_TEE=1 ./protected/one-shot/update_turbolanding_goals.pl 

    Параметры (для тестирования):
        shard-id -- запустить только на определенном шарде
        client-id -- запустить только для определенного клиента

    Программа работает параллельно с каждым шардом. Можно перезапускать.

=cut

use Direct::Modern;
use open ':std' => ':utf8';

use Settings;
use ScriptHelper 'Yandex::Log' => 'messages';

use Yandex::DBTools;
use Yandex::DBShards;
use ShardingTools qw/ppc_shards/;
use Yandex::ListUtils qw/chunks xminus/;
use Yandex::Retry qw/relaxed/;

use BannerStorage;
use JavaIntapi::SaveTurbolandings;
use RBACElementary;

use Try::Tiny qw/try catch/;

#На текущий момент максимальное количество клиентов с турболендингами на одном шарде ~20 000
my $CHUNK_SIZE = 100;
my $SLEEP_COEFF = 1;
my ($ONLY_SHARD_ID, $CLIENT_ID);
extract_script_params(
    'shard-id:i' => \$ONLY_SHARD_ID,
    'client-id:i'  => \$CLIENT_ID,
    'sleep-coeff=i' => \$SLEEP_COEFF,
);
$log->out('START');

my $script_name = get_script_name();
my $shard_results = foreach_shard_parallel(shard => [defined $ONLY_SHARD_ID ? $ONLY_SHARD_ID : ppc_shards()], sub {
    my ($shard) = @_;

    $log->msg_prefix("shard=$shard");
    $log->out('START');
    my $total = 0;
    try {
        my  $tl_data= get_all_sql(PPC(shard => $shard), [
            'SELECT DISTINCT t.ClientID, t.tl_id
                FROM turbolandings t',
                WHERE => {
                    ($CLIENT_ID ? (ClientID => $CLIENT_ID) : ()),
                    _TEXT => 'NOT EXISTS (
                        SELECT 1 FROM turbolanding_metrika_goals g WHERE g.tl_id = t.tl_id AND is_conversion_goal > 0)'
                },
                'ORDER BY' => 't.ClientID'
        ]);

        $log->out((scalar @$tl_data)." turbolandings found");

        foreach my $chunk (chunks $tl_data, $CHUNK_SIZE) {
            my $tl_ids_by_client_id;
           foreach my $row (@$chunk) {
                my $client_id = $row->{ClientID};
                my $tl_id = $row->{tl_id};
                $log->out("Preparing to fetch client: $client_id, turbolanding: $tl_id");
                $tl_ids_by_client_id->{$client_id} //= [];
                push @{$tl_ids_by_client_id->{$client_id}}, $row->{tl_id};
            }
            my $chief_uid_by_client_id = rbac_get_chief_reps_of_clients([keys %$tl_ids_by_client_id]);
            foreach my $client_id (keys %$tl_ids_by_client_id){
                my $chief_uid = $chief_uid_by_client_id->{$client_id};
                my $tl_ids = $tl_ids_by_client_id->{$client_id};
                unless ($chief_uid) {
                    $log->out("Client $client_id hasn't chief uid, turbolandings skipped: [".(join ', ', @$tl_ids)."]");
                    next;
                }

                relaxed times => $SLEEP_COEFF, sub {
                    _resync_turbolandings($client_id, $chief_uid, $tl_ids, \$total)
                };

            }
            

            sleep 1;
        }

        $log->out("$total clients synced");
        $log->out('FINISH');
    }
    catch {
        $log->out( join "\n", @_);
        die;
    };
});

$log->out('FINISH');

sub _resync_turbolandings {
    my ($client_id, $chief_uid, $tl_ids, $ref_total_clients_processed) = @_;
    try {
        my $bs_turbolandings_res = BannerStorage::banner_storage_turbolandings_call(
            GET => 'landings',
            {
                client_id => $client_id,
                ids => join(',', grep {$_ > 0} @$tl_ids),
                limit => 1 + @$tl_ids,
            },
            url => BannerStorage::turbolandings_api_url(),
        );
        die $bs_turbolandings_res->{error} if ($bs_turbolandings_res->{error});

        my $bs_turbolandings = $bs_turbolandings_res->{content}->{items} // [];
        my $lost_tl_ids = xminus $tl_ids, [map {$_->{id}} @$bs_turbolandings];
        if (@$lost_tl_ids){
            $log->out("Client: $client_id, unknown turbolandings: [".(join ', ', @$lost_tl_ids)."]");
        }

        if(@$bs_turbolandings){

            my $ok = JavaIntapi::SaveTurbolandings->new(
                    client_id => $client_id,
                    operator_uid => $chief_uid,
                    turbolandings => $bs_turbolandings,
            )->call();

            die 'intapi error' unless $ok;
            
            $$ref_total_clients_processed++;
            $log->out("Client: $client_id, turbolandings resynced successfully: [".(join ', ', @{xminus($tl_ids, $lost_tl_ids)})."]");
        }
    }
    catch {
        $log->out('Got error: '.shift);
        $log->out("Client: $client_id, turbolandings resync failed: [".(join ', ', @$tl_ids)."]");
    };
}
