#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: */59 */4 * * *
    ulimit: -v 4000000
    <switchman>
        group: scripts-other
        <leases>
            mem: 14000
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    time: 46 */4 * * *
    ulimit: -v 4000000
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl:          13h
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 DESCRIPTION

    Считает и пишет в agency_total_sums итоговые суммы ("остатки") по агентствам.
    Надо, чтобы показывать менеджерам в списке всех их клиентов/агентств,
    а считать на лету с мультивалютностью получается слишком долго. 

    Если будет необходимость -- можно будет кешировать и суммы для клиентов.

    Параметры

    --help, -h
        показать справку

    --agency-uids, -u <uid>[,<uid>]
        для отладки/тестирования: пересчитать только агентства с указанными uid'ами
        можно указывать несколько раз, можно указывать uid'ы через запятую:
        -u 97068736,706870 -u 6873609

    --agency-client-ids, -c <ClientID>[,<ClientID>]
        для отладки/тестирования: пересчитать только агентства с указанными ClientID
        можно указывать несколько раз, можно указывать ClientID через запятую:
        -c 97068736,706870 -c 6873609

    Если указан и ключ -u, и ключ -c -- считаются все перечисленные агентства.

    Если не указан и -u, ни -с -- пересчитываются остатки по всем агентствам.


    $Id$

=cut

use warnings;
use strict;

use Yandex::TimeCommon;
use Yandex::DBTools;

use my_inc "..";

use ScriptHelper 'Yandex::Log' => 'messages';

use Settings;
use RBAC2::Extended;
use RBACDirect;
use Property;
use PrimitivesIds;

use Agency;
use Currency::Rate;
use Campaign::Types qw(get_camp_kind_types);

run() unless caller();


sub run
{
    $log->out("start");

    my (@agency_uids_to_process, @agency_client_ids_to_process);
    extract_script_params(
        'a|agency-uids=s' => \@agency_uids_to_process,
        'c|agency-client-ids=s' => \@agency_client_ids_to_process,
    );
    @agency_uids_to_process = split(/,/,join(',',@agency_uids_to_process));
    @agency_client_ids_to_process = split(/,/,join(',',@agency_client_ids_to_process));

    my $now = unix2mysql(ts_round(time, "hour")); 
    my $laststart_prop = new Property('AGENCY_TOTAL_SUMS_LASTSTART'); 

    my $rbac = RBAC2::Extended->get_singleton(1);

    my $ag_client_ids = [];

    # ClientID из параметров
    push @$ag_client_ids, @agency_client_ids_to_process if @agency_client_ids_to_process;

    # ClientID из uid'ов -- заданных в параметрах или всех uid'ов агентств
    my $ag_uids = @agency_uids_to_process || @agency_client_ids_to_process ? \@agency_uids_to_process : rbac_get_all_agencies_chief_reps($rbac);
    my $ag_uids_client_ids = get_clientids(uid => $ag_uids);
    push @$ag_client_ids, @$ag_uids_client_ids;

    # interface соответствует agency_total_sums.camp_type, смотри её описание в схеме
    my %interface2kind = (
        'text' => 'web_edit_base',
        'mcb' => 'media',
    );

    $log->out("got ".(scalar @$ag_client_ids)." agencies");
    for my $ClientID (@$ag_client_ids){
        my $msg_prefix_guard = $log->msg_prefix_guard("[ClientID $ClientID]");
        $log->out("Collecting currency for agency");
        my $currency = get_agency_currency($rbac, $ClientID);
        for my $interface (keys %interface2kind) {
            my $kind = $interface2kind{$interface};
            $log->out("Collecting sum for campaigns of kind $kind and interface $interface");
            my $type = get_camp_kind_types($kind);
            my $sums = agency_total_sums(
                $ClientID, 
                rbac => $rbac,
                type => $type,
                currency => $currency,
            );
            my $to_insert = {
                ClientID => $ClientID,
                camp_type => $interface,
            };
            $to_insert->{currency}     = $sums->{currency};
            $to_insert->{sum_currency} = $sums->{sum};
            $to_insert->{sum_spent_currency} = $sums->{sum_spent};

            for my $f (qw/sum sum_spent/){
                $to_insert->{$f."_ynd"} = convert_currency($sums->{$f}, $sums->{currency}, 'YND_FIXED', with_nds => 1);
            }

            do_insert_into_table(PPC(ClientID => $ClientID), 'agency_total_sums', $to_insert, on_duplicate_key_update => 1, key => ['ClientID', 'camp_type']);
            $log->out({agency_client_id => $ClientID, %$to_insert});
        }
    }

    # если пересчитывали все агентства -- запоминаем время
    $laststart_prop->set($now) if !@agency_uids_to_process;

    juggler_ok();
    $log->out("finish");
    exit;
}
