#!/usr/bin/perl

use my_inc "..";

=head1 NAME

    ppcArchiveOldBanners.pl

=head1 DESCRIPTION

    Архивирует баннеры, остановленные больше 30 дней назад и в которых за 30 дней ничего не менялось.

=head1 USAGE

    LOG_TEE=1 ./protected/ppcArchiveOldBanners.pl --shard-id N
    LOG_TEE=1 ./protected/ppcArchiveOldBanners.pl --shard-id N --cid

=cut

=head1 METADATA

<crontab>
    time: 48 2,3,4,5,6 * * *
    sharded: 1
    package: scripts-switchman
    <switchman>
        group: scripts-other
        <leases>
            mem: 40
        </leases>
    </switchman>
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded:        1
    ttl:            2d4h
    tag: direct_group_internal_systems
</juggler>

=cut

use Direct::Modern;

use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';

use Settings;
use Property;
use Campaign::Types;

use Yandex::DBTools;
use Yandex::TimeCommon qw(today unix2mysql);
use Yandex::ListUtils qw(chunks);
use Yandex::Advmon;

use List::Util qw(max);

use constant {
    DAYS_STOPPED => 30,
    DAYS_WITHOUT_EDITING => 30,
    ONE_TIME_BANNER_ARC_COUNT_LIMIT => 500_000,
    UPDATE_CHUNK_SIZE => 1_000,
    SLEEP_BETWEEN_UPDATES => 60,
};

local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};

my @only_cids;
extract_script_params(
    'cid=i@' => \@only_cids,
);
@only_cids = map { split /\s*,\s*/ } @only_cids;

$log->out('START');

my ($today, $last_success_date_prop);
if (@only_cids) {
    $log->out('Woring only on cids:', \@only_cids);
} else {
    $last_success_date_prop = new Property("ppcArchiveOldBanners_last_success_date_shard_$SHARD");
    my $last_success_date = $last_success_date_prop->get();
    $today = today();
    if ($last_success_date && $last_success_date eq $today) {
        $log->out("Script has already succeded today, exitting");
        exit(0);
    }
}

my $cur_ts = time();

my %cond_to_arc = (
    'b.statusArch' => 'No',
    'b.statusShow' => 'No',
    'b.LastChange__lt' => unix2mysql($cur_ts - max(DAYS_STOPPED,DAYS_WITHOUT_EDITING)*24*60*60),
    'p.LastChange__lt' => unix2mysql($cur_ts - DAYS_WITHOUT_EDITING*24*60*60),
);
if (@only_cids) {
    $cond_to_arc{'b.cid'} = \@only_cids;
}

$log->out('Fetching banners to archive');
my $bids_to_arc = get_one_column_sql(PPC(shard => $SHARD), ['
    SELECT b.bid
    FROM banners b
    INNER JOIN phrases p ON b.pid = p.pid
    INNER JOIN campaigns c ON p.cid = c.cid
    INNER JOIN users u ON c.uid = u.uid
',  WHERE => {
        %cond_to_arc,
        'c.type' => get_camp_kind_types('auto_archive_banners'),
        'c.archived' => 'No',
        'c.statusEmpty' => 'No',
        'c.AgencyID__not_in' => \@Settings::UNARCHIVABLE_AGENCYIDS,
        'u.login__not_in' => \@Settings::UNARCHIVABLE_TEST_USERS,
    }, '
    LIMIT ?
'], ONE_TIME_BANNER_ARC_COUNT_LIMIT);
my $bids_to_arc_cnt = scalar @$bids_to_arc;
$log->out("Got $bids_to_arc_cnt banners to archive:", $bids_to_arc);
monitor_values({flow => { ppcArchiveOldBanners => { banners_to_archive_count => $bids_to_arc_cnt } } });

my $total_affected_rows = 0;
for my $bids_chunk (chunks($bids_to_arc, UPDATE_CHUNK_SIZE)) {
    $log->out('Archiving banners with bids:', $bids_chunk);
    my $affected_rows_cnt = do_update_table(PPC(shard => $SHARD), 'banners b INNER JOIN phrases p ON b.pid = p.pid', {'b.statusArch' => 'Yes', 'b.statusBsSynced' => 'No', 'b.LastChange__dont_quote' => 'NOW()'}, where => {%cond_to_arc, 'b.bid' => $bids_chunk});
    $log->out("Updated $affected_rows_cnt banners");
    $total_affected_rows += $affected_rows_cnt;

    $log->out('Sleeping for ' . SLEEP_BETWEEN_UPDATES . ' seconds');
    sleep SLEEP_BETWEEN_UPDATES;
}
$log->out("Done archiving. Have archived $total_affected_rows banners.");
monitor_values({flow => { ppcArchiveOldBanners => { archived_banners_count => $total_affected_rows } } });

if (!@only_cids) {
    $log->out("Writing $today to property");
    $last_success_date_prop->set($today);
}

$log->out('Writing to juggler queue');
juggler_ok();

$log->out('FINISH');

