#!/usr/bin/perl

use my_inc "..";



=head1 METADATA

<crontab>
    params_postfix: 2>&1 | head -c 100000
    time: */16 11-18 * * *
    ulimit: -v 16000000
    <switchman>
        group: scripts-other
        <leases>
            mem: 16000
        </leases>
    </switchman>
    package: scripts-switchman
    sharded: 1
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl:          20h
    sharded:      1
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 DESCRIPTION

    $Id$
    Подсчёт прогноза бюджета

=head1 RUNNING

    LOG_TEE=1 ./protected/ppcAutobudgetForecast.pl --shard-id N --cids cid1,cid2,cid3 ...

=head2 CAVEATS

    Явно течем по памяти (и потом падаем с невозможность выделить память)
    Боремся с этим двумя способами:
    - разбиваем все заказы на относительно большие пачки и обсчитываем их внутри форка,
        чтобы процесс не потреблял большого суммарного количества памяти
    - внутри форка каждые FLUSH_CACHES_EVERY_N_CAMPAIGNS сбрасываем кеши Yandex::ExecuteJS

=cut

use warnings;
use strict;

use List::Util qw/max min/;
use List::MoreUtils qw/any/;
use Parallel::ForkManager;

use Yandex::DBTools;
use Yandex::ListUtils;
use Yandex::ProcInfo;
use Yandex::TimeCommon;

use Campaign;
use Currencies;
use DirectCache;
use Forecast::Autobudget;
use Forecast;
use Notification;
use PlacePrice;
use RBAC2::Extended;
use ScriptHelper 'get_file_lock' => ['dont_die'], sharded => 1, 'Yandex::Log' => 'messages';
use Settings;
use Holidays qw/is_weekend is_holiday_everywhere regions_with_holidays regions_with_weekend_workday/;
use User;
use geo_regions;

use constant MAX_PROCS => 1;
use constant MAX_CAMPS_IN_CHUNK => 5_000;
use constant MAGIC_EXIT_CODE => 42;
use constant FLUSH_CACHES_EVERY_N_CAMPAIGNS => 20;

my $ONLY_CIDS = '';
extract_script_params(
    "cids=s" => \$ONLY_CIDS,
);

$Settings::BS_RANK_PARALLEL_LEVEL = int(15 / MAX_PROCS);

# Через сколько дней прогноз устаревает
my $EXPIRE = 1;

my @cids = grep {/^\d+$/} split /\s*,\s*/, $ONLY_CIDS;
if ( @cids ) {
    $ENV{DEBUG_FORECAST} = 1;
}

# проверяем, рабочий ли сегодня день хотя бы в одной из стран присутствия
# раньше незапуск по выходным был в кронтабе, но это не дружит с juggler-проверками
my $today = today();
if ( is_holiday_everywhere($today) ) {
    $log->out("Holiday!");
    juggler_ok(description => 'Holiday today - skip working');
    exit;
}

$log->out(" Let's start..."); 

my $where = @cids 
            ? "c.cid in (".join(',',@cids).")"
            : "( c.autobudgetForecastDate is null OR c.autobudgetForecastDate < SUBDATE(now(), $EXPIRE) )";
# Если сегодня выходной но хотя бы в одном регионе есть рабочий день - берем кампании только из таких регионов
# Если сегодня рабочий день, но хотя бы в одном регионе есть выходной - не берем кампании из таких регионов
if ( is_weekend($today) ) {
    my $regions_with_workdays = regions_with_weekend_workday($today);
    if (@$regions_with_workdays) {
        my $timezones_to_use = get_one_column_sql(PPCDICT, ["SELECT timezone_id FROM geo_timezones",
                                                             WHERE => {country_id__in => $regions_with_workdays}]);
        push @$timezones_to_use, 0 if (any {$_ == $geo_regions::RUS} @$regions_with_workdays);
        $where = "$where AND ".sql_condition({"c.timezone_id__in" => $timezones_to_use});
    }
} else {
    my $regions_with_holidays = regions_with_holidays($today);
    if (@$regions_with_holidays) {
        my $timezones_to_exclude = get_one_column_sql(PPCDICT, ["SELECT timezone_id FROM geo_timezones",
                                                                WHERE => {country_id__in => $regions_with_holidays}]);
        push @$timezones_to_exclude, 0 if (any {$_ == $geo_regions::RUS} @$regions_with_holidays);
        $where = "$where AND ".sql_condition({"c.timezone_id__not_in" => $timezones_to_exclude});
    }
}
my $data = get_all_sql( PPC(shard => $SHARD), 
                          "SELECT c.cid
                                , c.autobudget
                                , c.statusAutobudgetForecast
                                , af.autobudgetForecastClicks
                                , c.name
                                , c.uid
                                , c.ManagerUID
                                , c.AgencyUID
                                , co.strategy
                                , c.platform
                                , IFNULL(s.type, '') as strategy_name, s.strategy_data
                                , IFNULL(c.currency, 'YND_FIXED') AS currency
                                , c.type
                              FROM campaigns c
                                   LEFT JOIN camp_options co ON co.cid = c.cid
                                   LEFT JOIN strategies s on c.strategy_id = s.strategy_id
                                   LEFT JOIN autobudget_forecast af on c.cid = af.cid
                             WHERE c.autobudget = 'Yes'
                               AND c.type = 'text'
                               AND c.statusEmpty = 'No'
                               AND c.statusModerate != 'New'
                               AND c.archived = 'No'
                               AND c.sum - c.sum_spent > 0.01
                               AND c.statusShow = 'Yes'
                               AND $where
                               " ); 

$log->out("Got " . scalar (@$data) . " camps to calc forecast");
my $pm = new Parallel::ForkManager(MAX_PROCS);
$pm->run_on_finish(sub {
    my ($pid, $exit) = @_;
    if (defined $exit && $exit != MAGIC_EXIT_CODE) {
        $log->die("child $pid exited abnormally (exit code $exit)");
    }
});

for my $data_chunk (chunks($data, MAX_CAMPS_IN_CHUNK)) {
    $pm->start and next;

    my $prefix_guard = $log->msg_prefix_guard("[$$]");
    $log->out("start processing another campaigns chunk");

    undef $data;    # free memory
    my $direct_cache = new DirectCache(groups => ['forecast_db_queries']);
    my $camps_calced_cnt = 0;

    my $rbac = RBAC2::Extended->get_singleton(1);
    for my $row (@$data_chunk) {
        if ($camps_calced_cnt++ == FLUSH_CACHES_EVERY_N_CAMPAIGNS) {
            Yandex::ExecuteJS::flush_context_cache();

            $camps_calced_cnt = 0;
        }

        Campaign::_deserialize_camp_fields($row);
        $row->{strategy} = Campaign::campaign_strategy($row);
        my $forecast = $row->{strategy}->{search}->{name} =~ /autobudget/
            ? _autobudget_for_search($row)
            : _autobudget_for_net($row);

        next unless $forecast;

        # пока записываем в обе таблицы, потом оторвем поля и запись в campaigns
        do_sql(PPC(shard => $SHARD), "UPDATE campaigns SET autobudgetForecastDate = now(), autobudgetForecast = ?, statusAutobudgetForecast = ? WHERE cid = ?"
                   , $forecast->{sum}, $forecast->{status}, $row->{cid});

        my $autobudget_forecast_row = {
            cid => $row->{cid}
            , autobudgetForecastDate__dont_quote => 'now()'
            , autobudgetForecast => $forecast->{sum},
            , autobudgetForecastClicks =>  $forecast->{clicks},
            , statusAutobudgetForecast => $forecast->{status}
        };
        do_insert_into_table(PPC(shard => $SHARD), 'autobudget_forecast', $autobudget_forecast_row, on_duplicate_key_update => 1, key => ['cid']);

        # Отсылаем письмо агентству
        if ($forecast->{status} eq 'Wrong' && $row->{statusAutobudgetForecast} ne 'Wrong') {
            $row->{autobudgetForecast} = $forecast->{sum};
            $row->{autobudgetForecastClicks} = int($forecast->{clicks} || 0);
            my $user_props = get_user_data($row->{uid}, [qw/login email fio phone/]);
            $row->{"client_$_"} = $user_props->{$_} for keys %$user_props;

            if ($row->{AgencyUID}) {
                $row->{autobudget_strategy} = $row->{strategy}->{is_search_stop}
                    ? $row->{strategy}->{net}->{name} : $row->{strategy}->{search}->{name};
                $row->{is_different_places} = $row->{strategy}->{name} eq 'different_places';
                add_notification($rbac, 'autobudget_forecast_wrong', $row);
            }
        }
    }    

    $log->out({memory_usage => proc_memory()});

    $pm->finish(MAGIC_EXIT_CODE);
}

$pm->wait_all_children;                           

unless ($ONLY_CIDS) {
    juggler_ok();
}
$log->out("Let's finish..."); 


sub _autobudget_for_search {
    
    my $campaign = shift;
    my $strategy = $campaign->{strategy}->{search}->{name};
    
    my $currency = $campaign->{currency};
    die 'no currency given' unless $currency;
    my $max_bid_constant = get_currency_constant($currency, 'MAX_PRICE');

    my $projection;
    if ($strategy eq 'autobudget') {
        # просто автобюджет

        my $max_bid = $campaign->{strategy_decoded}->{bid} || $max_bid_constant;
        if ($max_bid > 0.1 * ($campaign->{strategy_decoded}->{sum} // 0)) {
            $max_bid = 0.1 * ($campaign->{strategy_decoded}->{sum} // 0);
        }

        my $forecast = eval { forecast_calc_camp($campaign->{cid}, $max_bid, {die_if_nobsdata => 1}) };
        unless ($@) {
            $log->out([$campaign->{cid}, $forecast]); 

            $projection = {

                sum => int(7 * get_data_by_place($forecast, PlacePrice::get_premium_entry_place())->{price} / 1e6 / 30 + 0.5),
            };
            $projection->{status} = $projection->{sum} < 0.95 * ($campaign->{strategy_decoded}->{sum} // 0) ? 'Wrong' : 'Valid';
        } else {
            $log->out(" Campaign: $campaign->{cid}, failed: $@"); 
        }
    } elsif ($strategy eq 'autobudget_week_bundle' && $campaign->{strategy_decoded}->{avg_bid}) {
        # недельный пакет кликов со средней ценой
        my $clicks = eval { calc_autobudget_clicks_by_avg_price($campaign->{cid}, $campaign->{strategy_decoded}->{avg_bid}, $currency) };
        unless ($@) {
            $log->out(" Campaign: $campaign->{cid}, result: $clicks");    

            $projection = {
                clicks => $clicks,
                status => $clicks < $campaign->{strategy_decoded}->{limit_clicks} ? 'Wrong' : 'Valid'
            }
        } else {
            $log->out(" Campaign: $campaign->{cid}, failed: $@"); 
        }
    } elsif ($strategy eq 'autobudget_week_bundle') {
        # недельный пакет кликов с максимальной ценой / или без ограничения на цену

        my $max_bid = $campaign->{strategy_decoded}->{bid} || $max_bid_constant;
        my $forecast = eval { forecast_calc_camp($campaign->{cid}, $max_bid, {die_if_nobsdata => 1}) };
        unless ($@) {
            $log->out([$campaign->{cid}, $forecast]); 

            $projection = {
                clicks => max(get_data_by_place($forecast, PlacePrice::get_guarantee_entry_place())->{clicks},
                              get_data_by_place($forecast, $PlacePrice::PLACES{PREMIUM1})->{clicks},
                              get_data_by_place($forecast, PlacePrice::get_premium_entry_place())->{clicks}),
            };
            $projection->{status} = $projection->{clicks} < $campaign->{strategy_decoded}->{limit_clicks} ? 'Wrong' : 'Valid';
        } else {
            $log->out(" Campaign: $campaign->{cid}, failed: $@"); 
        }
    }
    
    return $projection;
}

sub _autobudget_for_net {
    
    my $campaign = shift;

    my $currency = $campaign->{currency};
    die 'no currency given' unless $currency;
    my $max_bid_constant = get_currency_constant($currency, 'MAX_PRICE');

    my $projection;
    my $strategy = $campaign->{strategy}->{net}->{name};
    if ($strategy eq 'autobudget') {

        my $max_bid = min($campaign->{strategy_decoded}->{bid} || $max_bid_constant, 0.1 * ($campaign->{strategy_decoded}->{sum} // 0));
        my $forecast = eval {
            forecast_calc_camp($campaign->{cid}, $max_bid, {
                die_if_nobsdata => 1,
                for_net => 1
            })
        };

        unless ($@) {
            $log->out([$campaign->{cid}, $forecast]); 

            $projection = {
                sum => int(7 * $forecast->{sum} + 0.5),
            };
            $projection->{status} = $projection->{sum} < 0.95 * ($campaign->{strategy_decoded}->{sum} // 0) ? 'Wrong' : 'Valid';
        } else {
            $log->out(" Campaign: $campaign->{cid}, failed: $@"); 
        }
    } elsif ($strategy eq 'autobudget_week_bundle' && $campaign->{strategy_decoded}->{avg_bid}) {

        my $clicks = eval { 
            calc_autobudget_clicks_by_avg_price($campaign->{cid}, $campaign->{strategy_decoded}->{avg_bid}, 'context') 
        };
        unless ($@) {
            $log->out(" Campaign: $campaign->{cid}, result: $clicks");    

            $projection = {
                clicks => $clicks,
                status => $clicks < $campaign->{strategy_decoded}->{limit_clicks} ? 'Wrong' : 'Valid'
            }
        } else {
            $log->out(" Campaign: $campaign->{cid}, failed: $@"); 
        }
    } elsif ($strategy eq 'autobudget_week_bundle') {

        my $forecast = eval {
            forecast_calc_camp($campaign->{cid}, $campaign->{strategy_decoded}->{bid} || $max_bid_constant, {
                die_if_nobsdata => 1,
                for_net => 1
            })
        };

        unless ($@) {
            $log->out([$campaign->{cid}, $forecast]); 

            $projection = {clicks => int(7 * $forecast->{clicks} + 0.5)};
            $projection->{status} = $projection->{clicks} < $campaign->{strategy_decoded}->{limit_clicks} ? 'Wrong' : 'Valid';
        } else {
            $log->out(" Campaign: $campaign->{cid}, failed: $@"); 
        }
    }
    
    return $projection;
}
