#!/usr/bin/perl


=head1 METADATA

# ppcCalcAccountScoreFactors.pl: avg: 778 MB, max: 1767 MB
<crontab>
    env: YT_DIRECT_CLUSTER=prod
    time: 36 7-19 * * *
    params: --clear --download
    <switchman>
        group: scripts-other
        lockname: ppcCalcAccountScoreFactors.pl.prod.download
        <leases>
            mem: 1200
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    raw_events:     scripts.ppcCalcAccountScoreFactors.working.prod.download
    vars:           yt_cluster=hahn,arnold
    ttl:            36h
    tag:            direct_yt
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 DESCRIPTION

    Сохранение предрасчитанных данных по показателю качества аккаунта.
    --download - выгрузить данные из YT в MySQL
    --clear - удалить устаревшие данные в MySQL
    --force - игнорировать условия по свежести данных, выполнять работу
    --export-path - абсолютный путь до директории в YT с данными, без конечного слеша
        для бета-выгрузок будет выглядеть примерно так:
        //home/direct-dev/ppalex_development/import/account_score - Arnold
        //home/direct/tmp/ppalex_development/import/account_score - Hahn
        если не задан - используется продакшн-значение //home/direct/import/account_score

    На двух кластерах (каждом) в CalcFactorsJob:
    1. по таблице import/account_score/accounts_data смотрим, с какой (по дате загрузки) db/banners она сгенерирована
    2. если с текущей - генерацию не делаем
    3. если таблицы нет или даты отличаются:
        - генерируем таблицы YQL-запросом
        - сохраняем свежесть db/banners в аттрибутах (upload_time)

    На одном кластере (прод):
    1. получаем "свежесть" с таблицы import/account_score/accounts_data и из Property prev_account_score_calc_date
    2. если совпадает или в Property больше (так может быть при смене кластера YT) - сигналим в мониторинг, выходим
    3. проверяем, что за сегодня в базе еще нет данных (DIRECT-49506), если есть - сигналим и выходим
    4. чистим старые данные
    5. загружаем данные из YT в базу
    6. сохранем данные в Property

=cut

use strict;
use warnings;

use List::Util qw/maxstr sum/;
use List::MoreUtils;
use DateTime;

use my_inc '..';

use Yandex::Retry;
use Yandex::ListUtils;
use Yandex::YT::TableReader;
use Yandex::YT::Table;
use Yandex::DBTools;
use Yandex::DBShards;

my $YT_CLUSTER;
BEGIN {
    $YT_CLUSTER = $ENV{YT_DIRECT_CLUSTER} // 'prod';
}

use ScriptHelper get_file_lock => ['dont_die', "ppcCalcAccountScoreFactors.$YT_CLUSTER"];
use AccountScore;
use JSON;
use Property;

use Settings;
use Tools;
use EnvTools;
use ShardingTools;
use PrimitivesIds;


use constant BANNERS_UPLOAD_TIME_ATTR => 'upload_time';

my $export_path = '//home/direct/import/account_score';
my ($download, $clear, $FORCE);
extract_script_params(
    "download" => \$download,
    "clear" => \$clear,
    "force" => \$FORCE,
    "export-path=s" => \$export_path,
);
($download, $clear) = (1,1) if List::MoreUtils::all {!$_} ($download, $clear);

$log->msg_prefix("[$YT_CLUSTER]");
$log->out("start");

$log->out("setup YT environment");
Tools::force_set_yt_environment();

my $result_table = Yandex::YT::Table->new("$export_path/accounts_data");
if ($clear || $download) {
    my $prop = new Property('prev_account_score_calc_date') ;
    my $prev_account_score_calc_date = $prop->get();

    if (!$result_table->exists()) {
        $log->die("table with accounts_data does not exist");
    }
    my $yt_account_score_calc_date = $result_table->get_attribute(BANNERS_UPLOAD_TIME_ATTR());

    my $today = Yandex::DateTime->now()->strftime("%Y-%m-%d");
    my $has_today_data_in_db = get_one_field_sql(PPC(shard => 'all'), 'SELECT ClientID FROM account_score WHERE date = ? LIMIT 1', $today);

    if (!$FORCE && (!$prev_account_score_calc_date || !$yt_account_score_calc_date)) {
        $log->die("unknown state of account scores data in db and YT");
    } elsif (!$FORCE
        && $yt_account_score_calc_date
        && $prev_account_score_calc_date
        && $yt_account_score_calc_date le $prev_account_score_calc_date
    ) {
        if ($yt_account_score_calc_date lt Yandex::DateTime->now()->subtract(hours => 36)->strftime("%Y-%m-%dT%H:%M:%SZ")) {
            # показатели, рассчитанные по данным в YT уже загружены, но сами данные слишком старые
            juggler_warn(service_suffix => "$YT_CLUSTER.download",
                         description => "Account scores for $yt_account_score_calc_date already downloaded, but haven't been updated for more than 36 hours");
        }
        # показатели уже загружены одним из предыдущих запусков скрипта -- ничего делать не надо
        $log->out("Account scores was updated on $yt_account_score_calc_date. The previous success upload date is $prev_account_score_calc_date. Exiting");
        exit;
    } elsif ($has_today_data_in_db) {
        # В случае, если новые данные загрузятся в YT после того, как скрипт посчитает показатели "за сегодня" и загрузит данные в базу, последующие запуски этого скрипта по крону обнаружат, что у db/banners обновился upload_time. Но так как данные в базу уже загружены, ничего не делаем.
        $log->out("Account scores was updated, but data for $today are already uploaded to db, exiting");
        exit;
    }

    #Удаляем старые данные
    if ($clear) {
        clear_old_db_data();
    }

    # экспортируем данные в нащу БД
    if ($download) {
        my $status = export_yt_to_db();
        $prop->set($yt_account_score_calc_date);
        if ($status eq 'ok') {
            juggler_ok(service_suffix => "$YT_CLUSTER.download", description => "New account scores successfully downloaded");
        } else {
            juggler_warn(service_suffix => "$YT_CLUSTER.download", description => 'Possible error: Received too few records from YT');
        }
    }
}

$log->out("finish");

=head1 INTERNALS

=head2 export_yt_to_db

    экспортируем данные из YT в нащу БД

=cut
sub export_yt_to_db {

    my $profile = Yandex::Trace::new_profile('ppcCalcAccountScoreFactors:export_yt_to_db');
    $log->out("download started");
    my %to_insert;
    my $table = Yandex::YT::Table->new("$export_path/accounts_data");
    my $reader = $table->reader();
    while(my $r = $reader->next()) {
        my $ClientID = delete $r->{ClientID};
        my $score = delete $r->{score};
        $to_insert{$ClientID} = { factors_json => to_json($r), score => $score, type => 'client' };
    }

    $reader = Yandex::YT::TableReader->new("$export_path/avg_client_scores_by_agency");
    while (my $r = $reader->next()) {
        my $ClientID = $r->{id};
        my $score = $r->{score};
        $to_insert{$ClientID} = { factors_json => undef, score => $score, type => 'agency' };
    }

    $reader = Yandex::YT::TableReader->new("$export_path/avg_scores_by_manager");
    my %score_for_uid;
    while (my $r = $reader->next()) {
        my $uid = $r->{id};
        my $score = $r->{score};
        $score_for_uid{$uid} = $score;
    }
    my $ClientID_for_uid = get_uid2clientid(uid => [ keys %score_for_uid ]);
    for my $uid (keys %score_for_uid) {
        my $ClientID = $ClientID_for_uid->{ $uid };
        my $score = $score_for_uid{ $uid };
        if (!$ClientID) {
            my $msg = "Can't find ClientID for ManagerUID $uid";
            if (is_beta()) {
                $log->warn($msg);
                next;
            } else {
                $log->die($msg);
            }
        }
        $to_insert{$ClientID} = { factors_json => undef, score => $score, type => 'manager' };
    }

    my $counter = 0;
    my $today = Yandex::DateTime->now()->strftime("%Y-%m-%d");
    my $score_log = Yandex::Log->new(use_syslog => 1, log_file_name => 'account_score.log', no_log => 1);
    foreach my $chunk (sharded_chunks(ClientID => [keys %to_insert], 1_000)) {
        my @rows_to_insert;
        for my $client_id (@{$chunk->{ClientID}}) {
            my $row = [
                        $client_id,
                        $today,
                        $to_insert{$client_id}->{factors_json},
                        sprintf("%.2f", $to_insert{$client_id}->{score}),
                        $to_insert{$client_id}->{type},
                      ];
            push @rows_to_insert, $row;
        }
        $counter += do_mass_insert_sql(PPC(shard => $chunk->{shard}),
                                       "INSERT INTO account_score (ClientID, date, factors_json, score, type) VALUES %s",
                                       \@rows_to_insert);
        for my $row (@rows_to_insert) {
            my @fields = qw/ClientID date factors_json score type/;
            my @row_arr = @$row;
            my $log_record = { List::MoreUtils::zip(@fields, @row_arr) };
            $log_record->{factors} = $log_record->{factors_json} ? from_json($log_record->{factors_json}) : undef;
            delete $log_record->{factors_json};
            $score_log->out($log_record);
        }
    }

    my $status; # сейчас используется как сигнал о том, что записано слишком мало данных по сравнению с предыдущим запуском
    # Возможна ситуация, когда в результате выборки будут разные даты (за предыдущий запуск для какого-то шарда не было данных), поэтому выбираем максимальную
    my $prev_date = maxstr @{ get_one_column_sql(PPC(shard => "all"), ["SELECT MAX(date) FROM account_score", where => { date__lt => $today } ]) };
    my $prev_counter;
    if ($prev_date) {
        $prev_counter = sum @{ get_one_column_sql(PPC(shard => "all"), ["SELECT COUNT(ClientID) FROM account_score", where => { date => $prev_date } ]) };
    }
    if ($prev_date && $counter < $prev_counter / 2) {
        $status = 'fail';
    } else {
        $status = 'ok';
    }

    $log->out("inserted $counter records");
    $log->out("download finished");

    return $status;
}

=head2 clear_old_db_data

    Удаляем старые данные

=cut
sub clear_old_db_data {
    my $profile = Yandex::Trace::new_profile('ppcCalcAccountScoreFactors:clear_old_db_data');
    my $select_limit = 1_000_000;
    my $delete_limit = 1_000;
    my $date_border = Yandex::DateTime->now()->subtract(days => $AccountScore::HISTORY_DAYS)->strftime("%Y-%m-%d");
    for my $shard (ppc_shards()) {
        $log->out("clearing started, shard=$shard");
        my $min_client_id = 0;
        while(1) {
            # Для каждого клиента оставляем данные хотя бы за 2 дня: https://st.yandex-team.ru/DIRECT-35571

            my $data = get_hash_sql(PPC(shard => $shard), "SELECT ClientID, group_concat(date) AS dates
                                                            FROM account_score
                                                            GROUP BY ClientID
                                                            HAVING ClientID > $min_client_id
                                                            ORDER BY ClientID
                                                            LIMIT $select_limit");
            my %by_date;
            while (my ($client_id, $dates_str) = each %$data) {
                $min_client_id = $client_id if $client_id > $min_client_id;

                my @dates = sort split /,/, $dates_str;
                my @dates_to_delete = grep { $_ lt $date_border } @dates[0..$#dates-2];
                
                push @{ $by_date{ $_ } }, $client_id foreach @dates_to_delete;
            }

            while(my ($date, $client_ids) = each %by_date) {
                for my $client_ids_chunk (chunks $client_ids, $delete_limit) {
                    relaxed times => 2, sub {
                        do_delete_from_table(PPC(shard=>$shard), 'account_score', 
                                             where=>{date => $date, ClientID => $client_ids_chunk}
                            );
                    };
                }
            }
            
            last if scalar keys %$data != $select_limit;
        }
        $log->out("clearing finished shard=$shard");
    }
}

