#!/usr/bin/perl

use my_inc "..";


=head1 METADATA

<crontab>
    time: */59 */5 * * *
    sharded: 1
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    time: */59 */5 * * *
    sharded: 1
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl:     19h
    tag: direct_group_internal_systems
</juggler>

=cut


=head1 DESCRIPTION

    ppcCampCopyReportsDelete.pl
    Скрипт предназначен для удаления обработанных заданий очереди копирования кампаний с отчетом в таблице ppc.camp_copy_reports_queue
    Удаляются задания старше $STORAGE_INTERVAL месяцев со статусом 'Done' или 'Error'.

    Обязательные параметры:
        --shard-id N -- номер шарда, с которым работать
    Дополнительные параметры:
        --clientid N -- удалить отчеты только для указанного клиента
        --reportid N -- удалить только указанный отчет

=cut


use Direct::Modern;
use Direct::Storage;

use Yandex::Advmon;
use Yandex::DBTools;

use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Settings;

use Readonly;

# максимальное количество записей, которое пытаемся удалить за одну итерацию
Readonly my $MAX_RECORDS_LIMIT => 1000;
# удалять отчеты старше указанного количества месяцев
Readonly my $STORAGE_INTERVAL =>  3;
# список отчетов для удаления, задание удалится только если из MDS удалены все отчеты из этого списка
my @reports_to_delete = qw/
    banners_report_name
    phrases_report_name
    retargetings_report_name
    dynamic_conditions_report_name
    performance_filters_report_name
/;
# нужно для ограничения кол-во выполненных итераций
my $MAX_ITERATIONS = 10000;

my (@CLIENTIDS, @REPORTIDS);
extract_script_params(
    'clientid=i@' => \@CLIENTIDS,
    'reportid=i@' => \@REPORTIDS,
);

my $storage = Direct::Storage->new();
# считаем количество удаленных записей и пишем в Графит
my $delete_records_count = 0;

$log->out('START');

my %additional_conds;
# удаляем данные только по определенному клиенту
if (@CLIENTIDS) {
    $additional_conds{ClientID} = \@CLIENTIDS;
}
# удаляем данные только по определенному отчету
if (@REPORTIDS) {
    $additional_conds{id} = \@REPORTIDS;
}

# счётчик итераций; когда дойдёт до $MAX_ITERATIONS, скрипт завершается
my $iteration_counter = 0;

# отчеты которые не смогли удалить по какой либо причине
my @bad_records;
while (my @records = @{get_all_sql(PPC(shard => $SHARD),
                                  ['SELECT id, ClientID, queue_time, ready_time, statusReport,',
                                    join (', ', map { sql_quote_identifier($_) } @reports_to_delete),
                                   'FROM camp_copy_reports_queue',
                                    WHERE => {
                                        ready_time__le__dont_quote => "NOW() - INTERVAL $STORAGE_INTERVAL MONTH",
                                        statusReport => ['Done', 'Error'],
                                        id__not_in => \@bad_records,
                                        %additional_conds,
                                    },
                                   "LIMIT $MAX_RECORDS_LIMIT"])})
{
    $log->out('get '. scalar @records .' records to delete');

    my @delete_ok; # успешно удаленные из MDS записи
    foreach my $rec (@records) {
        $log->out($rec);
        my $is_all_reports_deleted = 1;
        foreach my $report (@reports_to_delete) {
            if ($rec->{$report}) {
                eval {
                    if ($storage->get_file('camp_copy_report', ClientID => $rec->{ClientID}, filename => $rec->{$report})) {
                        $storage->delete_file('camp_copy_report', ClientID => $rec->{ClientID}, filename => $rec->{$report});
                    }
                };
                if ($@) {
                    $log->out("error delete record id $rec->{id}, report name $rec->{$report}, from MDS: $@");
                    push @bad_records, $rec->{id};
                    $is_all_reports_deleted = 0;
                }
            }
        }
        # если все отчеты удалились (или таких файлов нет в MDS)
        if ($is_all_reports_deleted || $rec->{statusReport} eq 'Error') {
            push @delete_ok, $rec->{id};
        }
    }
    if (@delete_ok) {
        do_delete_from_table(PPC(shard => $SHARD), 'camp_copy_reports_queue', where => {id => \@delete_ok});
    }
    $delete_records_count += (scalar @delete_ok);

    last if should_quit(++$iteration_counter);
}

# пишем данные для мониторинга
local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
monitor_values({
    "flow.CampCopyReportsQueue.delete_records_count.shard_$SHARD" => $delete_records_count,
    "flow.CampCopyReportsQueue.bad_records_count.shard_$SHARD" => scalar @bad_records,
});

$log->out('Sending OK to juggler');
juggler_ok();

$log->out('FINISH');

=head2 should_quit

    перезапускаем процесс, если уже сделали достаточно итераций или получен SIGTERM/изменилась версия Директа
    На вход получает кол-во выполненных итераций
    На выходе отдает 0/1, где 1 означает необходимость перезапуска

=cut

sub should_quit {
    my $iteration_counter = shift;

    my $result = 0;
    if ($iteration_counter >= $MAX_ITERATIONS) {
        $log->out("done $MAX_ITERATIONS iterations, quitting");
        $result = 1;
    } elsif (my $reason = smart_check_stop_file()) {
        $log->out("$reason, quitting");
        $result = 1;
    }

    return $result;
}
