#!/usr/bin/perl

use my_inc "..";


=head1 METADATA

<crontab>
    time: */3 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 6000
        </leases>
    </switchman>
    package: scripts-switchman
    ulimit: -v 16000000
</crontab>
<crontab>
    time: */10 * * * *
    sharded: 1
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl:     1h30m
    tag: direct_group_internal_systems
</juggler>

=cut


=head1 DESCRIPTION

    ppcCampCopyReportsQueue.pl
    Скрипт предназначен для обработки очереди копирования кампаний с отчетом в таблице ppc.camp_copy_reports_queue
    Для выполнения выбираются наболее старое задание не в статусе 'Done' и 'Error'.

    Обязательные параметры:
        --shard-id N -- номер шарда, с которым работать

    Принимает опциональные параметры:
        --clientid CLIENTID — работать только с указанным клиентом, заказавшим отчёт

=cut


use Direct::Modern;

use Client::ConvertToRealMoney;
use Client;
use Direct::Storage;
use Notification qw/add_notification/;
use RBAC2::Extended;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Settings;
use User;

use Yandex::CSV;
use Yandex::DBTools;
use Yandex::ScalarUtils;

use JSON;

use Readonly;


# количество попыток сформировать отчет
Readonly my $TRY_COUNT  => 20;
# Максимальное количество заданий, обрабатываемых за один запуск скрипта
Readonly my $TASKS_LIMIT  => 100;

my @CLIENTIDS;
extract_script_params(
    'clientid=i@' => \@CLIENTIDS,
);


$log->out('START');

my $cond = {
    statusReport => ['New', 'Copied'],
};
if (@CLIENTIDS) {
    $log->out('Working only on ClientIDs:', \@CLIENTIDS);
    $cond->{ClientID} = \@CLIENTIDS;
}

my $tasks = get_all_sql(PPC(shard => $SHARD), ["SELECT id, ClientID, params_json, statusReport, queue_time, tries_count,
                                                  banners_report_name, phrases_report_name, retargetings_report_name, dynamic_conditions_report_name, performance_filters_report_name
                                                FROM camp_copy_reports_queue",
                                                WHERE => $cond,
                                               "ORDER BY queue_time LIMIT $TASKS_LIMIT"]);

$log->out('Got '. scalar @$tasks . ' tasks for processing');

my $storage = Direct::Storage->new();

for my $task (@$tasks) {
    $log->out("Working on task id $task->{id}, status: $task->{statusReport}, queue_time: $task->{queue_time}, params: $task->{params_json}");
    if ($task->{tries_count} >= $TRY_COUNT) {
        # поставить статус Error
        $log->out("Error during task $task->{id}: too many tries");
        do_update_table(PPC(shard => $SHARD), 'camp_copy_reports_queue', {statusReport => 'Error', ready_time__dont_quote => 'NOW()'},  where => {id => $task->{id}});
    } else {
        my $tries_count = ++$task->{tries_count};
        do_update_table(PPC(shard => $SHARD), 'camp_copy_reports_queue', {tries_count => $tries_count},  where => {id => $task->{id}});
        my $success = eval { _process_task($task, $storage); return 1; };
        if ($success) {
            $log->out("Done task $task->{id}");
        } else {
            $log->out("Error during task $task->{id}: $@");
        }
    }
}

$log->out('Sending OK to juggler');
juggler_ok();

$log->out('FINISH');


sub _process_task
{
    my ($task, $storage) = @_;

    my $msg_prefix_guard = $log->msg_prefix_guard("[$task->{id}]");

    my @new_cids;
    my $params = from_json($task->{params_json});
    my $report_name = {
        banners_report_name             => $task->{banners_report_name} // '',
        phrases_report_name             => $task->{phrases_report_name} // '',
        retargetings_report_name        => $task->{retargetings_report_name} // '',
        dynamic_conditions_report_name  => $task->{dynamic_conditions_report_name} // '',
        performance_filters_report_name => $task->{performance_filters_report_name} // '',
    };

    if ($task->{statusReport} eq 'New') {
        my $to_csv = {
            banners => {
                header_row => [
                    "CampaignId_$params->{old_login}",
                    "AdGroupId_$params->{old_login}",
                    "BannerId_$params->{old_login}",
                    "CampaignId_$params->{new_login}",
                    "AdGroupId_$params->{new_login}",
                    "BannerId_$params->{new_login}",
                ],
                data => [],
            },
            phrases => {
                header_row => [
                    "CampaignId_$params->{old_login}",
                    "AdGroupId_$params->{old_login}",
                    "KeywordId_$params->{old_login}",
                    "CampaignId_$params->{new_login}",
                    "AdGroupId_$params->{new_login}",
                    "KeywordId_$params->{new_login}",
                ],
                data => [],
            },
            retargetings => {
                header_row => [
                    "CampaignId_$params->{old_login}",
                    "AdGroupId_$params->{old_login}",
                    "RetargetingConditionId_$params->{old_login}",
                    "CampaignId_$params->{new_login}",
                    "AdGroupId_$params->{new_login}",
                    "RetargetingConditionId_$params->{new_login}",
                ],
                data => [],
            },
            dynamic_conditions => {
                header_row => [
                    "CampaignId_$params->{old_login}",
                    "AdGroupId_$params->{old_login}",
                    "WebpageId_$params->{old_login}",
                    "CampaignId_$params->{new_login}",
                    "AdGroupId_$params->{new_login}",
                    "WebpageId_$params->{new_login}",
                ],
                data => [],
            },
            performance_filters => {
                header_row => [
                    "CampaignId_$params->{old_login}",
                    "AdGroupId_$params->{old_login}",
                    "DynamicMediaAdFilterId_$params->{old_login}",
                    "CampaignId_$params->{new_login}",
                    "AdGroupId_$params->{new_login}",
                    "DynamicMediaAdFilterId_$params->{new_login}",
                ],
                data => [],
            },
        };
        my @ids_array_banners;
        my @ids_array_phrases;
        my $rbac = RBAC2::Extended->get_singleton( $params->{UID} );

        foreach my $old_cid (@{$params->{cids}}) {
            $log->out("Copy camp $old_cid");

            my $ids_mapping = {};
            my $new_cid = eval {
                Client::ConvertToRealMoney::copy_camp_converting_currency($rbac, $old_cid, $params->{new_uid}, $params->{manager_uid}, $params->{agency_uid},
                                                                          flags => $params->{flags}, override => $params->{override}, keep_empty => 1, ids_mapping => $ids_mapping,
                                                                          new_wallet_cid => $params->{new_wallet_cid}, UID => $params->{UID});
            };

            if (defined $new_cid) {
                $log->out("Successfully copied campaign $old_cid into campaign $new_cid");
                push @new_cids, $new_cid;
                # сохраняем полученные id-шники
                foreach my $old_pid (keys %{$ids_mapping->{pid_old2new}} ) {
                    my $new_pid = $ids_mapping->{pid_old2new}->{$old_pid};
                    my $banners_map = $to_csv->{banners}->{data};
                    foreach my $old_bid (@{$ids_mapping->{old_groups_banners}->{$old_pid}}) {
                        my $new_bid = $ids_mapping->{bid_old2new}->{$old_bid};
                        push @$banners_map, [$old_cid, $old_pid, $old_bid, $new_cid, $new_pid, $new_bid];
                    }
                    my $phrases_map = $to_csv->{phrases}->{data};
                    foreach my $old_phrase_id (@{$ids_mapping->{new_groups_phrases}->{$new_pid}}) {
                        my $new_phrase_id = $ids_mapping->{phrase_id_old2new}->{$old_phrase_id};
                        push @$phrases_map, [$old_cid, $old_pid, $old_phrase_id, $new_cid, $new_pid, $new_phrase_id];
                    }
                    my $retargetings_map = $to_csv->{retargetings}->{data};
                    foreach my $old_ret_cond_id (@{$ids_mapping->{old_pid2ret_cond_ids}->{$old_pid}}) {
                        my $new_ret_cond_id = $ids_mapping->{ret_cond_ids_old2new}->{$old_ret_cond_id};
                        push $retargetings_map, [$old_cid, $old_pid, $old_ret_cond_id, $new_cid, $new_pid, $new_ret_cond_id];
                    }
                    my $dynamic_conditions_map = $to_csv->{dynamic_conditions}->{data};
                    foreach my $old_dyn_cond_id (@{$ids_mapping->{old_gid2dyn_cond_ids}->{$old_pid}}) {
                        my $new_dyn_cond_id = $ids_mapping->{dyn_cond_ids_old2new}->{$old_dyn_cond_id};
                        push $dynamic_conditions_map, [$old_cid, $old_pid, $old_dyn_cond_id, $new_cid, $new_pid, $new_dyn_cond_id];
                    }
                    my $performance_filters_map = $to_csv->{performance_filters}->{data};
                    foreach my $old_perf_filter_id (@{$ids_mapping->{old_gid2perf_filter_ids}->{$old_pid}}) {
                        my $new_perf_filter_id = $ids_mapping->{perf_filter_ids_old2new}->{$old_perf_filter_id};
                        push $performance_filters_map, [$old_cid, $old_pid, $old_perf_filter_id, $new_cid, $new_pid, $new_perf_filter_id];
                    }
                }
            } else {
                my $msg = "error copying campaign $old_cid, skipping: $@";
                # флаг устанавливается при копировании кампаний на но
                if ($params->{flags}->{ignore_copy_errors}) {
                    $log->out($msg);
                } else {
                    $log->die($msg);
                }
            }
        }
        # гернерируем отчеты
        $log->out("Generate copy reports for task $task->{id}");
        foreach my $report (keys %$to_csv) {
            # формируем отчет только если есть данные
            if (@{$to_csv->{$report}->{data}}) {
                my $csv_report = data2csv([ sort { $a->[0] <=> $b->[0] || $a->[1] <=> $b->[1] || $a->[2] <=> $b->[2] } @{$to_csv->{$report}->{data}} ],
                                          { header_row => $to_csv->{$report}->{header_row} });

                my $name = "${report}_from_$params->{old_login}_to_$params->{new_login}_$task->{id}";
                $log->out("Upload report $name to MDS");
                # если файл уже существует в MDS, то нужно его удалить
                $storage->delete_file('camp_copy_report', ClientID => $task->{ClientID}, filename => $name) if $storage->get_file('camp_copy_report', ClientID => $task->{ClientID}, filename => $name);
                # заливаем в MDS
                $storage->save('camp_copy_report', \$csv_report, ClientID => $task->{ClientID}, filename => $name);
                $report_name->{"${report}_report_name"} = $name;
            }
        }

        # обновляем статусы на кампаниях
        do_in_transaction {
            # кампании обновляем в шарде того, кому скопировали
            do_update_table(PPC(uid => $params->{new_uid}), 'campaigns',
                                                  {
                                                      statusEmpty => 'No',
                                                      statusBsSynced => 'No',
                                                      LastChange__dont_quote => 'NOW()'
                                                  },
                                                  where => { cid => \@new_cids });
            do_update_table(PPC(shard => $SHARD), 'camp_copy_reports_queue',
                                                  {
                                                      statusReport => 'Copied',
                                                      %$report_name,
                                                  },
                                                  where => {id => $task->{id}});
        };
    }

    # отправляем email
    $log->out("Sending email notification to UID $params->{UID}");
    my $fio = get_user_data($params->{UID}, [qw/fio/]);
    my $user_vars = {
        uid => $params->{UID},
        client_id => $task->{ClientID},
        fio => $fio,
        new_login => $params->{new_login},
        old_login => $params->{old_login},
        %$report_name,
    };
    my $rbac = RBAC2::Extended->get_singleton(1);
    add_notification($rbac, 'camp_copy_report', $user_vars);
    do_update_table(PPC(shard => $SHARD), 'camp_copy_reports_queue', {statusReport => 'Done', ready_time__dont_quote => 'NOW()'},  where => {id => $task->{id}});
}
