#!/usr/bin/perl

use my_inc "..";

# $Id$

=head1 METADATA

<crontab>
    time: */29 * * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 12000
        </leases>
    </switchman>
    package: scripts-switchman
    ulimit: -v 40000000
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 6h
    tag: direct_group_internal_systems
</juggler>

<juggler_check>
    host:   checks_auto.direct.yandex.ru
    name:       direct.camp_metrika_goals.last_fetched_date
    raw_events: direct.camp_metrika_goals.last_fetched_date.production
    ttl:        12h
    tag: direct_group_internal_systems
    <notification>
        template: on_status_change
        status: OK
        status: CRIT
        method: telegram
        login: DISMonitoring
    </notification>
</juggler_check>

=cut

=head1 NAME

    ppcCampGetGoals.pl

=head1 DESCRIPTION

    Забираем из БК все цели по счетчикам метрики, которые привязаны к кампаниям
    и по которым были достигнуты цели, а также "виртуальные" ecommerce-цели.
    Данные забираем из БК каждые GOALS_REFRESH_INTERVAL часов.

    Опции командной строки:
    --days <days>
        за сколько последних дней втянуть данные по целям

    --date-from YYYY-MM-DD
    --date-to YYYY-MM-DD
        за какой период забирать статистику. Должны быть указаны оба параметра

    --force
        втягивать даже если сегодня уже загружали данные

    --skip-exists
        не затрагиваем в БД существующие записи

    --no-count
        не втягиваем в БД количества достижения целей (считаем их 0)

    --fake-bs-url
        использовать указанный url вместо ручки БК

    --chunk-size
        размер чанка сериализации строк данных

=cut

use Direct::Modern;

use List::MoreUtils qw/uniq all/;

use Yandex::Advmon;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ScalarUtils;
use Yandex::HashUtils;
use Yandex::TimeCommon qw/unix2human/;

use Settings;
use ScriptHelper 'Yandex::Log' => 'messages';

use BSStatImport;
use Campaign;
use Property;

use open ':std' => ':utf8';

# повторять запрос в БК
our $RETRY_COUNT //= 5;
# за сколько дней назад (исключая сегодняшний день) спрашивать цели
our $GET_GOALS_DAYS_AGO //= 28;
# сколько часов не обновляем данные по целям с момента последнего успешного обновления
our $GOALS_REFRESH_INTERVAL = 4;
# таймаут в секундах для запросов с данными
our $BS_GOALS_TIMEOUT //= 1800;

# если пришло больше 0.1% битых строк - умираем
our $INVALID_DATA_MAX_RATE = 0.001;

my ($days, $force, $skip_exists, $no_count, $date_from, $date_to, $FAKE_URL, $LINES_CHUNK_SIZE);
extract_script_params(
    'days=i' => \$days, # за сколько последних дней втянуть цели
    'force' => \$force,  # втягивать даже если сегодня уже загружали данные
    'skip-exists' => \$skip_exists,
    'no-count' => \$no_count,
    'date-from=s' => \$date_from,
    'date-to=s' => \$date_to,
    'fake-bs-url=s' => \$FAKE_URL,
    'chunk-size=i' => \$LINES_CHUNK_SIZE,
);
$GET_GOALS_DAYS_AGO = $days if defined $days;
my $juggler_description;

sub main
{

    my $current_timestamp = time();

    unless (defined $date_from && defined $date_to) {
        ($date_from, $date_to) = (unix2human(($current_timestamp - ($GET_GOALS_DAYS_AGO * 24 * 3600)), "%Y-%m-%d"), unix2human($current_timestamp, "%Y-%m-%d"));
    }

    local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};

    my $last_timestamp_stat_prop = new Property('last_camp_goals_timestamp_stat');
    my $last_timestamp_stat = $last_timestamp_stat_prop->get() || 0;

    # чтобы были данные по свежести, даже если падаем
    check_timestamp_for_juggler($last_timestamp_stat);

    # выходим если успешно загрузили
    if ($last_timestamp_stat > ($current_timestamp - ($GOALS_REFRESH_INTERVAL * 3600)) && !$force) {
        my $last_datetime_stat = unix2human($last_timestamp_stat);
        $juggler_description = "Already worked. Last fetched date is $last_datetime_stat";
        $log->out($juggler_description);
        return;
    }

    if ($force) {
        # если обновляем принудительно - дату обновления сегодняшних записей меняем на вчерашнюю
        $log->out('Updating (forced) stat_date=yesterday for today metrika goals');
        my $res = do_sql(PPC(shard => 'all'), "update camp_metrika_goals set stat_date = stat_date - INTERVAL 1 DAY where stat_date >= date(NOW())");
        $log->out("Updated $res rows in all shards");
    }

    my $campaigns_goals = query_to_bs($date_from, $date_to);

    if ($no_count) {
        foreach my $OrderID (keys %$campaigns_goals) {
            foreach my $goal_id (keys %{$campaigns_goals->{$OrderID}}) {
                $campaigns_goals->{$OrderID}->{$goal_id}->{$_} = 0 foreach (qw/total context/);
            }
        }
    }

    my $result = foreach_shard_parallel(OrderID => [keys %$campaigns_goals], with_undef_shard => 1, sub {
        my ($shard, $orders) = @_;

        if ($shard) {
            $log->msg_prefix("shard $shard:");
            Campaign::save_metrika_goals(hash_cut($campaigns_goals, $orders), skip_exists => $skip_exists, shard => $shard, log => $log);
        } else {
            # непонятно, куда сохранять данные (нет данных в метабазе?)
            # логгируем, для упрощения разбирательств
            $log->out({orders_without_shard => $orders});
        }
    });
    my @error_in_shards = grep { ! all { $_ } @{$result->{$_}} } keys %$result;
    if (@error_in_shards) {
        $log->out($result);
        $log->die("Errors in shards: " . join ', ', @error_in_shards);
    }

    # удаляем старые данные по целям (которые пришли более чем полгода назад)
    $log->out('Deleting old goals');
    my $res = do_sql(PPC(shard => 'all'), "
        DELETE FROM camp_metrika_goals
         WHERE DATE(stat_date) < DATE(NOW() - INTERVAL 6 MONTH)
    ");
    $log->out("Deleted $res rows in all shards");
    monitor_values({'flow.ppcCampGetGoals.deleted_goals_count' => $res});

    # обнуляем старые данные по целям (которые пришли не сегодня)
    unless ($skip_exists) {
        $log->out('Reset stats for old goals');
        my $res_upd = do_sql(PPC(shard => 'all'), "
            UPDATE camp_metrika_goals
               SET goals_count = 0
                 , context_goals_count = 0
             WHERE DATE(stat_date) < DATE(NOW())
        ");
        $log->out("Updated $res_upd rows in all shards");
        monitor_values({'flow.ppcCampGetGoals.updated_old_goals_count' => $res_upd});
    }

    $last_timestamp_stat_prop->set($current_timestamp);
    check_timestamp_for_juggler($current_timestamp);
}

=head2 check_timestamp_for_juggler($timestamp)

    Проверить последнюю дату, за которую есть данные и сообщить в мониторинг "ок" или "не ок"

=cut

sub check_timestamp_for_juggler {
    my $ts = shift;
    my $date = unix2human($ts);

    $log->out('send data to juggler');

    juggler_check(service => 'direct.camp_metrika_goals.last_fetched_date',
                  description => "свежесть (в часах) полученных из БК данных ($date)",
                  value => sprintf("%.1f", (time() - $ts)/3600),
                  warn => 8,
                  crit => 12,
    );
}

=head2 query_to_bs

    Достаем из БК все цели по кампаниям с кол-вом переходов по ним за период
    my $goals = query_to_bs('2011-04-27', '2011-05-25');
    В результате набор OrderID c AttributionType = 1, по каждому набор id целей с кол-вом переходов по каждой за период

    $goals = {
        43549 => {
            6 => {
                context => 209,
                total => 210
                  },
            4 => {
                context => 13,
                total => 20
            }
        },
        44764 => {
            1 => {
                context => 24,
                total => 42
            }
        }
    };

=cut

sub query_to_bs {

    my $date_from = shift;
    my $date_to = shift;

    $log->out("Request BS for orders for dates: $date_from - $date_to");
    my $t1 = Time::HiRes::time();
    my $bssi = BSStatImport->new(
        type => 'direct_goal_stat',
        params => {
            date_from => $date_from,
            date_to   => $date_to,
        },
        timeout => $BS_GOALS_TIMEOUT,
        separate_invalid_rows => 1,
        ($FAKE_URL ? (fake_url => $FAKE_URL): ()),
    );
    $log->out('URL: ' . $bssi->get_url());
    if (my $error = $bssi->get()) {
        $log->die($error);
    } else {
        $log->out("end request");
    }
    $log->out(sprintf("got goals stat from BS: %.2f sec, %d lines", Time::HiRes::time() - $t1, $bssi->get_rows_count()));

    my $rows_cnt = $bssi->get_rows_count();
    my $invalid_rows_cnt = $bssi->get_invalid_rows_count();
    my $invalid_data = $bssi->get_invalid_data();
    $log->out("Got $rows_cnt rows from BS with $invalid_rows_cnt invalid rows");

    if (@$invalid_data) {
        $log->out("logging invalid data");
        $log->bulk_out(invalid_data => $invalid_data);
    }
    if ($invalid_rows_cnt / $rows_cnt > $INVALID_DATA_MAX_RATE) {
        my $msg = "Got too many invalid lines from BS: $invalid_rows_cnt of $rows_cnt";
        juggler_crit(description => $msg);
        $log->die($msg);
    }

    my $chunks_iterator = $bssi->get_named_chunks_iterator($LINES_CHUNK_SIZE);
    my $chunks_count = 0;
    my $goals = {};
    while (my $stat_chunk = $chunks_iterator->()) {
        $log->out("goals stat chunk $chunks_count - processing");
        $chunks_count++;
        my %ecommerce_goals;
        for my $row (@$stat_chunk) {
            if ($row->{GoalID} > 0) {
                if ($row->{AttributionType} == 1) {
                    $goals->{$row->{OrderID}}->{$row->{GoalID}} = {
                        context => $row->{ContextGoalsCount} || 0,
                        total => ($row->{ContextGoalsCount} || 0) + ($row->{SearchGoalsCount} || 0)
                    };
                } else { # достижения по цели есть, по дефолтному типу отметим, что нули
                    $goals->{$row->{OrderID}}->{$row->{GoalID}} //= {
                        context => 0,
                        total => 0
                    }
                }
                if (($row->{GoalID} >= $Settings::ECOMMERCE_MIN_GOALID) && ($row->{GoalID} < $Settings::ECOMMERCE_MAX_GOALID)) {
                    my $name = $row->{GoalID} - $Settings::ECOMMERCE_MIN_GOALID;
                    $ecommerce_goals{$row->{GoalID}} //= [$row->{GoalID}, $name, 'ecommerce'];
                }
            }
        }
        if (%ecommerce_goals) {
            $log->out('got ' . (scalar keys %ecommerce_goals) . ' ecommerce-goals to insert');
            my $count = do_mass_insert_sql(PPCDICT, "INSERT IGNORE INTO metrika_goals (goal_id, name, goal_type) VALUES %s", [values %ecommerce_goals]);
            $log->out("inserted $count rows");
        }
    }
    return $goals;
}

$log->out('START');

main();

juggler_ok(description => $juggler_description);
$log->out('FINISH');
