#!/usr/bin/perl

use my_inc "..";



=head1 METADATA

# ppcCampQueue.pl: avg: 295 MB, max: 662 MB
<crontab>
   params: --par-id=copy:1
   time: */3 * * * *
   sharded: 1
   flock: 1
   <switchman>
       group: scripts-test
   </switchman>
   package: conf-test-scripts
</crontab>
<crontab>
   params: --par-id=copy:2
   time: */15 * * * *
   sharded: 1
   flock: 1
   <switchman>
       group: scripts-test
   </switchman>
   package: conf-test-scripts
</crontab>
<crontab>
   params: --par-id=copy:3
   time: */15 * * * *
   sharded: 1
   flock: 1
   <switchman>
       group: scripts-test
   </switchman>
   package: conf-test-scripts
</crontab>
<crontab>
   params: --par-id=rest
   time: */3 * * * *
   sharded: 1
   flock: 1
   <switchman>
       group: scripts-test
   </switchman>
   package: conf-test-scripts
</crontab>

<crontab>
    params: --par-id=copy:1
    time: */3 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 650
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
#Дополнительные обработчики очереди копирования - copy:1, copy:3
#Активируются через PpcProperty ppc_camp_queue_copy_additional_workers.
#Чтобы в нективированном состоянии давали меньшую нагрузку - запускаем их раз в 15 минут.
<crontab>
    params: --par-id=copy:2
    time: */15 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 650
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>

<crontab>
    params: --par-id=copy:3
    time: */15 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 650
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>

<crontab>
    params: --par-id=rest
    time: */3 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 650
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    raw_events:     scripts.ppcCampQueue.working.$par_id.shard_$shard
    sharded:        1
    vars:           par_id<ttl=25m>=copy_1
    vars:           par_id=rest
    <meta_url>
        title:  Документация
        url:    https://docs.yandex-team.ru/direct-dev/reference/perl-scripts/list/ppcCampQueue.pl
    </meta_url>
    tag: direct_group_internal_systems
</juggler>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.ppcCampQueue.working.additional
    raw_events:     scripts.ppcCampQueue.working.$par_id.shard_$shard
    ttl:            1500
    sharded:        1
    vars:           par_id=copy_2,copy_3
    tag: direct_group_internal_systems
</juggler>

<crontab>
    params: --par-id=copy:1
    time: */4 * * * *
    sharded: 1
    only_shards: 1
    package: scripts-sandbox
</crontab>
<crontab>
    params: --par-id=rest
    time: */4 * * * *
    sharded: 1
    only_shards: 1
    package: scripts-sandbox
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.ppcCampQueue.working.sandbox
    raw_host:       CGROUP%direct_sandbox
    raw_events:     scripts.ppcCampQueue.working.sandbox.$par_id.shard_$shard
    vars:           shard=1
    vars:           par_id=copy_1
    vars:           par_id=rest
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

    ppcCampQueue.pl - демон обработки фоновой очереди
                      архивации/разархивации/удаления/копирования кампаний

=cut

=head1 DESCRIPTION

    Опции командной строки:
    --help  - вывести справку
    --shard-id — номер шарда, с которым работать
    --par-id - тип очереди. Может принимать два значения:
        copy:N - обрабатывает команды копирования в потоке N, для простого ручного запуска можно использовать --copy:1
        rest - команды архивации, разархивации и удаления
    --once  - отработать одну итерацию и выйти
    --cid   - список номеров кампаний через запятую
    --login - логин пользователя, чьи кампании обрабатывать из очереди

=head1 RUNNING

    LOG_TEE=1 ./protected/ppcCampQueue.pl --once --cid 3660673 --shard-id 8 --par-id=copy:1
    LOG_TEE=1 ./protected/ppcCampQueue.pl --once --cid 3660673 --shard-id 8 --par-id=rest

    Пример для нескольких кампаний:
    LOG_TEE=1 ./protected/ppcCampQueue.pl --once --shard-id 11--cid 14683966 --cid 12985369 --par-id copy:1

=cut

use Direct::Modern;

use List::Util qw/min/;
use Try::Tiny qw/try catch/;

use Yandex::SendMail;
use Yandex::ScalarUtils;
use Yandex::Retry;

use Settings;
use ScriptHelper get_file_lock => undef,
                 script_timer => undef,
                 'Yandex::Log' => 'messages',
                 sharded => 1;
use Yandex::DBTools;
use Common qw/:subs/;
use Campaign;
use EnvTools;
use LockTools;
use Tools;
use PrimitivesIds;
use Property;
use Client;
use Client::ConvertToRealMoney;

use JSON;
use RBAC2::Extended;

# Пауза после пустой итерации
my $EMPTY_LOOP_SLEEP = $Settings::CAMP_QUEUE_SLEEP;

# Разгрузочный коэффциент для обработчиков очереди архивации/разархивации 
my $ARC_SLEEP_COEF_PROP  = new Property('camp_arc_queue_sleep_coef');

# Разгрузочный коэффциент для обработчиков очереди копирования
my $COPY_SLEEP_COEF_PROP = new Property('camp_copy_queue_sleep_coef');
# Активация и переопределение дополнительных воркеров очереди копирования
my $COPY_ADDITIONAL_WORKERS_CONF_PROP = new Property('camp_copy_queue_additional_workers');

# интервал сообщений в juggler.
my $FLUSH_JUGGLER_STATUS_INTERVAL = 60;
my $last_juggler_time = 0;

my ($PAR_ID, $ONCE, @cids, $login);
extract_script_params(
    'par-id=s' => \$PAR_ID,
    "once" => \$ONCE,
    'cid=s' => \@cids,
    'login=s' => \$login,
);

if (defined $PAR_ID) {
    die "Incorrect par-id: $PAR_ID" unless $PAR_ID =~ /^(copy(:\d+)?|rest)$/;
}
unless (defined $PAR_ID) {
    die "Usage: $0 [--par-id=(copy:N|rest)] [--shard-id=N] [--login=LOGIN] [--cid=N] [--once]";
}
my $SCRIPT = get_script_name();
#Запоминаем исходный шард, до переопределения
my $ORIGINAL_SHARD = $SHARD;

my $SQL_LOCK_NAME;
my ($THREAD_ID, $OPERATION);
if (_is_copy_worker()) {
    _apply_remaping_and_set_operation_with_thread_id();
    #Если $THREAD_ID - 0, значит это неактивированный дополнительный обработчик, выходим
    exit(0) unless $THREAD_ID;
    #В случае переопределения, sql lock берется на шарде, задания которого обрабатываем,
    #с соответствующим номером потока ("на того парня")
    $SQL_LOCK_NAME = "${SCRIPT}_${OPERATION}:${THREAD_ID}_LOCK";
    #Если это основной обработчик - проверяем, что нет лока установленного старой версией.
    #После выкладки https://st.yandex-team.ru/DIRECT-141900 можно вообще удалить использование файлового лока
    #т.к. с одной стороны логика switchman'а не должна запускать запуск дубликатов,
    #с другой стороны, у нас есть sql-lock который тоже не даст запуститься двум обработчикам одной пары шард:поток
    if ($THREAD_ID eq '1'){
         get_file_lock('dont_die', "${SCRIPT}.${OPERATION}.lock");
         release_file_lock();
    }
}
else {
    $OPERATION //= $PAR_ID;
    $THREAD_ID = 1;
    $SQL_LOCK_NAME = "${SCRIPT}_${PAR_ID}_LOCK";
}

my $previos_camp_copy_queue_additional_workers_value;
# файловый lock берется по параметрам запуска воркера, без учета переопределения ("на себя")
my $filename_safe_par_id = _get_safe_par_id(); 
get_file_lock('dont_die', "${SCRIPT}.${filename_safe_par_id}.lock");

#Если получилось взять файловый lock (нет запущеного с такими же параметрами воркера), берем sql-lock 
my $sql_lock = sql_lock_guard(PPC(shard => $SHARD), $SQL_LOCK_NAME, 1);


@cids = split qr/,/, join ',', @cids;

$log->msg_prefix("shard_$SHARD,par-id_".(_is_copy_worker() ? "${OPERATION}:${THREAD_ID}" : $PAR_ID));
$log->out("Start");
if (@cids) {
    $log->out('Working only on cids ' . join ',', @cids);
}
if ($login) {
    $log->out("Working only with login $login");
}

Tools::_save_vars(0); # для логгинга цен при копировании кампаний с условиями

my $continue = 1;
while($continue) {
    my $trace = Yandex::Trace->new(service => 'direct.script', method => get_script_name(short => 1), tags => "shard_$SHARD");

    if (my $reason = smart_check_stop_file()) {
        $log->out("$reason! Let's finish.");
        last;
    }

    my $join = '';
    my %where;
    $where{'oq.cid'} = \@cids if @cids;
    if ($login) {
        $join = 'JOIN campaigns c ON (c.cid = oq.cid) JOIN users u ON (u.uid = c.uid)';
        $where{'u.login'} = $login;
    }
    # в качестве временного решения, DIRECT-135731: тормозные запросы от ppcCampQueue
    my $q = 'set session range_optimizer_max_mem_size = 134217728';
    do_sql(PPC(shard => $SHARD), $q);
    do_sql(PPCDICT, $q);

    my $task;
    if ($OPERATION eq 'copy') {
        #Метим задание, которое собираемся обработать своим $THREAD_ID
        do_sql(PPC(shard => $SHARD), ['UPDATE camp_operations_queue_copy', SET => {par_id => $THREAD_ID},
            WHERE => {
                _OR => { par_id__is_null => 1, par_id => $THREAD_ID},
                $where{'oq.cid'} ? (cid => $where{'oq.cid'}) : (),
            },
            $login ? ('AND' => 'cid' => 'IN' => '(',
                    'SELECT cid FROM campaigns c JOIN users u on (c.uid = u.uid)',
                        WHERE => {'u.login' => $login}, ')') : (),
            ORDER => BY => 'par_id desc, effective_queue_time, queue_time',
            LIMIT => 1]);
        #Затем выбираем первое из помеченых
        $task = get_one_line_sql(PPC(shard => $SHARD), [
                        'SELECT oq.id, oq.cid, oq.queue_time, "copy" AS operation, oq.params FROM camp_operations_queue_copy oq',
                        $join, WHERE => {par_id => $THREAD_ID, %where},
                        'ORDER BY effective_queue_time, queue_time LIMIT 1']);

    } elsif ($OPERATION eq 'rest') {
        $task = get_one_line_sql(PPC(shard => $SHARD), [
                    'SELECT 0 as id, oq.cid, oq.queue_time, oq.operation, oq.params FROM camp_operations_queue oq',
                    $join, (%where ? (WHERE => \%where) : ()),
                    'ORDER BY queue_time LIMIT 1']);
    }
    if ($task) {
        $log->out("Get task \"$task->{operation}\" for camp $task->{cid}, queue_time: $task->{queue_time}, params: " . str($task->{params}));
        my $camp_info = get_camp_info($task->{cid}, undef, short => 1);
        my $uid = $camp_info->{uid};

        my $params;
        my $trace_comment_vars = {};
        if ($task->{params}) {
            $params = from_json $task->{params};
            if ($params->{UID}) {
                $trace_comment_vars->{operator} = $params->{UID};
            }
        }
        # добавляем UID к комментариям sql-запросов
        local $Yandex::DBTools::TRACE_COMMENT_VARS = sub {return $trace_comment_vars; };

        my $is_success = eval { retry tries => 5, pauses => [1, 30, 60, 180], sub {
            if ($task->{operation} eq 'arc') {
                my $sleep_coef = $ARC_SLEEP_COEF_PROP->get(60) // 0;
                relaxed times => $sleep_coef, sub {
                    my ($arc_result, $arc_error) = Common::_arc_camp($uid, $task->{cid}, force => 1);
                    $log->out({cid => $task->{cid}, arc_result => $arc_result, arc_error => $arc_error});
                };

            } elsif ($task->{operation} eq 'unarc') {
                my $sleep_coef = $ARC_SLEEP_COEF_PROP->get(60) // 0;
                relaxed times => $sleep_coef, sub {
                    unarc_camp($uid, $task->{cid}, force => 1);
                };

            } elsif ($task->{operation} eq 'del') {
                del_camp_data($task->{cid}, $uid);

            } elsif ($task->{operation} eq 'copy') {
                if ($params) {
                    my $rbac = RBAC2::Extended->get_singleton( $params->{UID} );

                    my $new_user_info = Primitives::get_user_info($params->{new_uid});
                    if (!$new_user_info) {
                        $log->out("Skip processing copy task for new_uid=$params->{new_uid}, cid=$task->{cid} : user was deleted");
                        return 1;
                    } elsif (($new_user_info->{statusBlocked} // '') eq 'Yes') {
                        $log->out("Skip processing copy task for new_uid=$params->{new_uid}, cid=$task->{cid} : user was blocked");
                        return 1;
                    }

                    # ещё раз проверяем, что у клиентов совпадают валюты
                    # при занесении в очередь мы уже проверяли, но с тех пор один из них мог перейти в другую валюту
                    my $new_client_id = get_clientid(uid => $params->{new_uid});
                    my $new_client_currencies = get_client_currencies($new_client_id, allow_initial_currency => 1, uid => $params->{new_uid});

                    my $time_before = time;
                    #Разгрузочный коэффициент берем для того шарда, с которым реально работаем
                    my $sleep_coef = _get_copy_sleep_coef($SHARD);
                    relaxed times => $sleep_coef, sub {
                        my $new_cid = eval {
                            Client::ConvertToRealMoney::copy_camp_converting_currency($rbac, $task->{cid}, $params->{new_uid}, $params->{manager_uid}, $params->{agency_uid}, flags => $params->{flags}, override => $params->{override}, campaign_name_prefix => $params->{campaign_name_prefix}, UID => $params->{UID});
                        };
                        if (defined $new_cid) {
                            $log->out("Successfully copied campaign $task->{cid} into campaign $new_cid");
                        } else {
                            my $msg = "error copying campaign $task->{cid}, skipping: $@";
                            $log->out($msg);
                            send_alert($msg, 'ppcCampQueue copy error');
                        }
                    };

                    update_effective_queue_time($params->{UID}, time - $time_before);

                } else {
                    my $msg = "no params given for background copy operation of campaign $task->{cid}, skipping";
                    $log->out($msg);
                    send_alert($msg);
                }
            } else {
                my $msg = "Unknown background operation $task->{operation} skipped";
                $log->out($msg);
                send_alert($msg);
            }
            return 1;
        } };
        if (!$is_success || $@) {
            $log->warn("ERROR processing task, dropping it: $@");
        }
        # не важно, прошла операция или нет - удаляем задачу из очереди
        # сравнение operation нужно, вдруг пока мы работали кто-то изменил задание?

        my $broken_sql_lock = Yandex::DBTools::has_broken_lock(PPC(shard => $SHARD));
        local $Yandex::DBTools::STRICT_LOCK_MODE = 0 if $broken_sql_lock;
        if ($task->{operation} eq 'copy') {
            do_delete_from_table(
                PPC(shard => $SHARD),
                'camp_operations_queue_copy',
                where => {id => $task->{id}}
            );
        } else {
            do_delete_from_table(
                PPC(shard => $SHARD),
                "camp_operations_queue",
                where => {cid => $task->{cid},  operation => $task->{operation}}
            );
        }
        $log->die("lost sql-lock, exitting") if $broken_sql_lock;
    } elsif ($ONCE) {
        $log->out("Finish");
        last;
    }

    # каждые N секунд сообщаем в juggler что еще работаем
    if (!$ONCE && !@cids && !$login && (time() - $last_juggler_time > $FLUSH_JUGGLER_STATUS_INTERVAL)) {
        $last_juggler_time = time();
        my $safe_par_id = _get_safe_par_id();
        juggler_ok(service_suffix => (EnvTools::is_sandbox() ? "sandbox.$safe_par_id" : $safe_par_id));
    }

    # делаем паузу, чтобы не положить базу
    if (!$task) {
        $log->out("No task found. Sleeping for $EMPTY_LOOP_SLEEP seconds.");
        sleep $EMPTY_LOOP_SLEEP;
    }

#Если воркер - дополнительный, проверяем, не изменилась ли для него конфигурация.
#Если изменилась - выходим, чтобы запуститься с новой.
   if ($THREAD_ID > 1) {
        my ($current_shard, $current_thread_id) = ($SHARD, $THREAD_ID);
        _apply_remaping_and_set_operation_with_thread_id();
        if ($current_shard != $SHARD || $current_thread_id != $THREAD_ID){
            $log->out("Worker ${ORIGINAL_SHARD}:${PAR_ID} configuration changed. Exiting.");
            $continue = 0;
        }
    }
}

if (!$ONCE && !@cids && !$login) {
    my $safe_par_id = _get_safe_par_id();
    juggler_ok(service_suffix => (EnvTools::is_sandbox() ? "sandbox.$safe_par_id" : $safe_par_id));
}
release_file_lock();
exit 0;

#Обновляем effective_queue_time кампаниям, поставленным на копирование тем же клиентом
sub update_effective_queue_time {
    my ($operator_uid, $copy_time) = @_;
    return unless $operator_uid;
   
    my $operator_client_id = get_clientid(uid => $operator_uid); 
    my $total_ela = Campaign::update_and_get_total_camp_copy_ela($operator_uid, $copy_time > 0 ? $copy_time : 1);
    if ($total_ela > 0) {
        do_sql(PPC(shard => $SHARD), 'UPDATE camp_operations_queue_copy SET effective_queue_time = TIMESTAMPADD(SECOND, ? ,queue_time)
                WHERE cid IN (SELECT cid FROM campaigns c WHERE c.ClientID = ?)', $total_ela, $operator_client_id);
    }
}

sub _is_copy_worker {
    return $PAR_ID =~ /^copy/;
}

#Пример конфигурации: {"1.2":"on", "1.3":"2.4", "2.2":"on", "2.3":"on"}
#Расшифровка:
#   worker copy:2 на первом шарде включен и обрабатывает свои задания (поток 2 на первого шарда),
#   worker copy:3 на первом шарда обрабатывае задания потока 4 второго шарда,
#   worker'ы copy:2 и copy:3 второго шарда обрабатывают свои задания  (потоки 2 и 3 второго шарда).
#   Т.е. в данном случае очередь первого шарда обрабатывается двумя воркерами (copy:1 и copy:2),
#   а очередь второго шарда - тремя воркерами (copy:2, copy:3 + copy:3 с первого шарда обрабатывает задания соседа).
sub _apply_remaping_and_set_operation_with_thread_id {
    my $thread_id;
    
    ($OPERATION, $thread_id) = split(/:/, $PAR_ID, 2);
    return unless _is_copy_worker();

    #Если это основной воркер - он не может быть ни выключен, ни переопределен
    return $THREAD_ID = 1 if $thread_id eq '1';

    #Чтобы при каждой операции копирования не делать проверки конфигурации,
    #сначала смотрим - менялась она или нет
    my $conf_json = $COPY_ADDITIONAL_WORKERS_CONF_PROP->get(120) // '';
    return $THREAD_ID //= 0 if $conf_json eq ($previos_camp_copy_queue_additional_workers_value // '');

    $previos_camp_copy_queue_additional_workers_value = $conf_json;

    #Если воркер дополнительный - он должен быть явно активирован.
    #По умолчанию - выключаем, задав $THREAD_ID = 0.
    $THREAD_ID = 0;

    my $conf;
    try {
        $conf = from_json($conf_json);
    } catch {
        my $e = shift;
        $log->out("property camp_copy_queue_additional_workers - json parsing error: $e");
    };

    return unless _is_valid_copy_configuration($conf);

    my $worker_id = "${ORIGINAL_SHARD}:${thread_id}";
    return unless exists $conf->{$worker_id};

    if ($conf->{$worker_id} eq 'on') {
        $THREAD_ID = $thread_id;
        $log->out("additional worker ${worker_id} enabled");
        return;
    }

    #Значение уже проверено в _is_valid_copy_configuration, поэтому сразу извлекаем шард и номер потока
    ($SHARD, $THREAD_ID) = split(/:/, $conf->{$worker_id});
    $log->out("remaping applied ${worker_id} -> ${SHARD}:${THREAD_ID}");
    
    return;
}

sub _is_valid_copy_configuration {
    my ($conf) = @_;

    return 0 unless defined $conf && ref $conf;

    unless (ref $conf eq 'HASH') {
        $log->out("property camp_copy_queue_additional_workers: invalid data");
        return 0;
    }

    my ($known_from, $known_to);
    my $no_errors = 1;

    #Проверяем, что в конфигурации дополнительных воркеров нет ошибок
    while (my ($worker_id, $state) = each %$conf) {
        my $is_valid_entry = _check($worker_id =~ /^\d+:(\d+)$/, "invalid worker_id '$worker_id'")
            && _check($1 > 1, "invalid worker_id '$worker_id' - main worker configuration couldn' be changed")
            && _check($known_from->{$worker_id}++ < 2, "duplicate worker_id '$worker_id'")
            && ($state eq 'on' || 
                    _check($state =~ /^\d+:(\d+)$/, "invalid state '$state' for worker '$worker_id'")
                 && _check($1 > 1, "invalid state '$state' for worker $worker_id - main worker remaping prohibited")
                 && _check($known_to->{$state}++ < 2, "duplicate state '$state' for worker '$worker_id'")
                 && _check(!exists $known_from->{$state}, "worker '$state' already enabled")
            );
        $no_errors = 0 unless $is_valid_entry;
    }
    
    return $no_errors;
}

sub _check {
    my ($is_valid, $message) = @_;
    $log->out("property camp_copy_queue_additional_workers error: $message") unless $is_valid;

    return $is_valid;
}

#Пример конфигурации: {"default":"0.02", "shard_1":"0.5"}
sub _get_copy_sleep_coef {
    my ($shard_id) = @_;

    my $default_sleep_coef = 0.01;
    my $conf_json = $COPY_SLEEP_COEF_PROP->get(120);
    
    return $default_sleep_coef unless defined $conf_json && $conf_json gt '';
    my $sleep_coef;
    try {
        my $conf = from_json($conf_json);
        $sleep_coef = $conf->{"shard_${shard_id}"} // $conf->{"default"} // $default_sleep_coef;
        die "invalid value ${sleep_coef}\n" unless $sleep_coef =~/^\d+(\.\d+)?$/
    }
    catch {
        my $e = shift;
        $log->out("Error while sleep coef defintion: $e");
    };

    return $sleep_coef // $default_sleep_coef;
}

sub _get_safe_par_id {
    return $PAR_ID =~ s/:/_/gr;
}
