#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: 24 1 * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 205
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 2d4h
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: 24 3 * * *
    package: scripts-sandbox
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    name:           scripts.ppcClearEventLog.working.sandbox
    raw_host:       CGROUP%direct_sandbox
    raw_events:     scripts.ppcClearEventLog.working.sandbox
    ttl: 28h
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

    ppcClearEventLog.pl

=head1 DESCRIPTION

    Чистит event-лог от старых событий. Время хранения событий задаётся в модуле EventLog.
    Время последней удачной очистки сохраняется в ppc_properties и используется для уменьшения выборки
    Параметры:
    --sleep-coef=1
    --days-ago - указание, сколько дней назад была последняя удачная очистка

=cut

use warnings;
use strict;

use Yandex::DBTools;
use Yandex::Retry;
use Yandex::DateTime qw/now duration date/;
use Yandex::ListUtils qw/chunks/;

use my_inc "..";

use EnvTools;
use Settings;
use ScriptHelper;
use ShardingTools;
use EventLog;
use Property;

use utf8;
use open ':std' => ':utf8';

=head2 $SELECT_BUNDLE_SIZE

    Количество записей, которые выбираются для удаления за один раз.

=cut

our $SELECT_BUNDLE_SIZE = 1_000_000;

=head2 $DELETE_BUNDLE_SIZE

    Количество записей, которые удаляются за один раз.

=cut

our $DELETE_BUNDLE_SIZE = 1_000;

$log->out('START');

my $DAYS_AGO;
my $SLEEP_COEF = 1;
extract_script_params(
    'days-ago=i' => \$DAYS_AGO,
    'sleep-coef=f' => \$SLEEP_COEF,
    );

my $today = Yandex::DateTime->now()->truncate(to => 'day');
my $prop = Property->new("ppcClearEventLog_last_success_time");
my $last_date = defined $DAYS_AGO ? $today-duration(days => $DAYS_AGO)
    : $prop->get() ? date($prop->get())
    : $today-duration(months => 1);
$log->out("last_date=$last_date");

# собираем условие для выбора записей, подпадающих под удаление, вида:
# (type IN (1,2,3,4,5,6,8) and eventtime < NOW() - INTERVAL 30 DAY) OR (type IN (7) and eventtime < NOW() - INTERVAL 30 DAY) OR [...]
my %days2types;
while (my($slug, $days_to_keep) = each %EventLog::CLEAR_EVENTS_AFTER_DAYS) {
    next unless defined $days_to_keep; # undef => храним вечно
    push @{$days2types{$days_to_keep}}, $EventLog::EVENTS{$slug}->{type};
}
my @conditions;
while (my($days, $types) = each %days2types) {
   push @conditions, 'OR' if @conditions;
   push @conditions, '(', {eventtime__between => [$last_date-duration(days=>$days), $today-duration(days=>$days)], type => $types}, ')';
}

for my $db (map {PPC(shard=>$_)} ppc_shards()) {
    $log->out("start process db ".join(',', dbnames($db)));
    my $ids_to_delete;
    do {
        $log->out('Selecting rows to be deleted');
    
        $ids_to_delete = get_one_column_sql($db, ['SELECT id FROM eventlog WHERE ', @conditions, ' LIMIT ?'], $SELECT_BUNDLE_SIZE);
        if (@$ids_to_delete) {
            $log->out('Selected ' . scalar(@$ids_to_delete) . ' rows to be deleted');
    
            for my $ids_bundle_to_delete (chunks($ids_to_delete, $DELETE_BUNDLE_SIZE)) {
                relaxed times => $SLEEP_COEF, sub {
                    my $deleted_rows_cnt = do_delete_from_table($db, 'eventlog', where => {id => $ids_bundle_to_delete});
                    $log->out('Deleted ' . int($deleted_rows_cnt) .' rows from eventlog');
                };
            }
        }
    } while (@$ids_to_delete == $SELECT_BUNDLE_SIZE);
}

$prop->set($today->ymd());

juggler_ok(service_suffix => (EnvTools::is_sandbox() ? 'sandbox' : undef));

$log->out('FINISH');
