#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: 2 2 * * *
    sharded: 1
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 2d
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: 2 2 * * *
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
    sharded: 1
</crontab>

=head1 DESCRIPTION

    Delete customers campaigns when _all_ of its campaigns:
    - are older than 365 days;
    - are unknown in BS and never were shown;
    - weren't paid;
    - weren't attempted to pay.
    - aren't connected to common wallet, that was paid

=head1 RUNNING

    ./protected/ppcClearOldUnpaidCampaigns.pl --shard-id 2 [--login somelogin]

    Optional args:

    --login somelogin - main representative login, whose campaigns should be cleaned

=cut

use my_inc "..";

use Direct::Modern;

use Yandex::Advmon;
use Yandex::DBTools;

use RBAC2::Extended;
use PrimitivesIds;

use Campaign qw/del_camp/;
use Campaign::Types qw(get_camp_kind_types);
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Settings;

my $DAYS_BEFORE_DELETE = 365;
my $USERS_CHUNK_SIZE = 100;
my $OPERATOR_UID = 1;

my @AGENCYIDS_TO_KEEP = (
    472729, # click-direct
    402819, # testagency
);

my $direct_campaign_types = get_camp_kind_types("under_wallet");
my $skip_auto_delete = get_camp_kind_types("skip_auto_archive_campaigns");

sub get_campaigns_chunk {
    my ($optional_cond) = @_;

    my $data_to_delete;
    do_in_transaction {
        do_sql(PPC(shard => $SHARD), 'SET GROUP_CONCAT_MAX_LEN=4294967295');
        $data_to_delete = get_all_sql(PPC(shard => $SHARD), ["
            SELECT STRAIGHT_JOIN c.uid, GROUP_CONCAT(c.cid) AS cids
             FROM campaigns c
               INNER JOIN users u USING(uid)
               LEFT JOIN camp_options co ON co.cid = c.cid
               LEFT JOIN campaigns wc ON c.uid=wc.uid AND wc.cid = c.wallet_cid
          ", WHERE => {
              'c.type' => $direct_campaign_types,
              'c.type__not_in' => $skip_auto_delete,
              'c.AgencyID__not_in' => \@AGENCYIDS_TO_KEEP,
              %$optional_cond,
          }, "
          GROUP BY c.uid
          HAVING COUNT(*) = COUNT(IF(
                   c.OrderID = 0
               AND c.sum = 0
               AND c.sum_to_pay = 0
               AND c.sum_spent = 0
               AND c.sum_last = 0
               AND IFNULL(c.shows, 0) = 0
               AND IFNULL(wc.sum, 0) = 0
               AND IFNULL(wc.sum_spent, 0) = 0
               AND IFNULL(wc.sum_last, 0) = 0
               AND u.hidden = 'No'
               AND (co.create_time IS NULL OR co.create_time < NOW() - INTERVAL $DAYS_BEFORE_DELETE DAY)
            , 1, NULL))
          LIMIT $USERS_CHUNK_SIZE",
        ]);
    };
    return $data_to_delete;
}


sub main {
    my @logins;
    extract_script_params(
        'login=s' => \@logins
    );
    $log->out('START');

    my %optional_cond;
    if (@logins) {
        $log->out("Script will clean only following logins: @logins");
        $optional_cond{'c.uid'} = get_uids(login => \@logins);
    }

    my $chunk_size;
    my $deleted_cnt = 0;
    my $delete_error_cnt = 0;
    do {
        $log->out('Fetching campaigns chunk to delete');
        my $chunk = get_campaigns_chunk(\%optional_cond);
        my $chunk_size = scalar(@$chunk);
        $log->out("Got new chunk to delete with $chunk_size campaigns");

        if ($chunk_size) {
            my $rbac = RBAC2::Extended->get_singleton($OPERATOR_UID);
            for my $item (@$chunk) {
                my $uid = $item->{uid};
                my $cids_str = $item->{cids};
                my @cids = split /,/, $cids_str;
                for my $cid (@cids) {
                    $log->out("Deleting campaign $cid of user with uid $uid");
                    my $camp_deleted = eval {
                        del_camp($rbac, $cid, $uid, dont_queue_deletion => 1, force_converted => 1);
                    };

                    if ($camp_deleted) {
                        $log->out("Succesfully deleted $cid");
                        $deleted_cnt++;
                    } else {
                        # Исключаем кампанию из списка
                        push @{ $optional_cond{'c.cid__not_in'} }, $cid;
                        if ($@) {
                            $log->out("Error when deleting $cid: $@")
                        } else {
                            $log->out("Campaign $cid cannot be deleted");
                        };
                        $delete_error_cnt++;
                    }
                }
            }
        }
        $log->out("Finished processing chunk");
    } while (($chunk_size // 0) == $USERS_CHUNK_SIZE);

    if (exists $optional_cond{'c.cid__not_in'}) {
        my $undeleted_cids = join(', ', @{$optional_cond{'c.cid__not_in'}});
        $log->out("Cannot delete following cids: $undeleted_cids")
    }

    $log->out("Writing monitoring: deleted_cnt=$deleted_cnt, delete_error_cnt=$delete_error_cnt");
    local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
    monitor_values({ flow => { ppcClearOldUnpaidCampaigns => {
        "deleted_cnt.shard_$SHARD" => $deleted_cnt,
        "delete_error_cnt.shard_$SHARD" => $delete_error_cnt,
    }}});

    $log->out('FINISH');

    juggler_ok();
}

main();
