#!/usr/bin/perl

=head1 METADATA

<crontab>
    params: --par-id=easy
    params_postfix: 2>&1 | tail -1000
    time: */5 * * * *
    sharded: 1
    package: scripts-switchman
    <switchman>
        group: scripts-other
        <leases>
            mem: 100
        </leases>
    </switchman>
</crontab>
<crontab>
    params: --par-id=heavy
    params_postfix: 2>&1 | tail -1000
    time: */5 * * * *
    sharded: 1
    ulimit: -v 10000000
    package: scripts-switchman
    <switchman>
        group: scripts-other
        <leases>
            mem: 1000
        </leases>
    </switchman>
</crontab>

<monrun>
    juggler_host:   checks_auto.direct.yandex.ru
    name: direct.client_potential_report.easy.max_age
    span: 2d
    warn: 21600
    sharded: 1
    expression: 'movingAverage(direct_one_min.db_configurations.production.flow.ClientPotentialReports.easy.max_age.shard_$shard, "5min")'
    tag: direct_group_internal_systems
    tag: direct_queues
</monrun>
<monrun>
    juggler_host:   checks_auto.direct.yandex.ru
    name: direct.client_potential_report.heavy.max_age
    span: 2d
    warn: 43200
    sharded: 1
    expression: 'movingAverage(direct_one_min.db_configurations.production.flow.ClientPotentialReports.heavy.max_age.shard_$shard, "5min")'
    tag: direct_queues
    tag: direct_group_internal_systems
</monrun>

<juggler>
    host:   checks_auto.direct.yandex.ru
    raw_events:     scripts.ppcClientPotentialReports.working.$par_id.shard_$shard
    sharded:        1
    vars:           par_id=easy
    vars:           par_id<ttl=1h>=heavy
    ttl:            30m
    tag: direct_group_internal_systems
</juggler>

=head1 NAME

    Построение отчётов по потенциалу клиентов

=head1 USAGE

    ./ppcClientPotentialReports.pl --par-id=(easy|heavy) --shard-id=1 [--login=report_owner_login1]
    
    NB! указывается шард оператора (того, кто заказывал отчет), а не клиента, по которому заказали отчет

=cut

use strict;
use warnings;

use my_inc "..";

use ScriptHelper 'Yandex::Log' => 'messages'
                    , get_file_lock => undef
                    , script_timer => undef
                    , sharded => 1;
use Settings;
use LockTools;
use Reports::Queue;
use Reports::ClientPotential;

use Yandex::Advmon;

use utf8;

local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};

my ($PAR_ID, $ONLY_LOGIN);
extract_script_params(
    'par-id=s' => \$PAR_ID,
    'login=s' => \$ONLY_LOGIN,
);

if (defined $PAR_ID) {
    die "Incorrect par-id: $PAR_ID" unless $PAR_ID =~ /^(easy|heavy)$/;
}

unless (defined $PAR_ID) {
    die "Usage: $0 [--par-id=(easy|heavy)] [--shard-id=N] [--login=LOGIN]";
}

my $SCRIPT_NAME = get_script_name();
get_file_lock('dont_die', "$SCRIPT_NAME.$PAR_ID.lock");

$ScriptHelper::trace = Yandex::Trace->new(service => 'direct.script', method => get_script_name(short => 1), tags => "shard_$SHARD,par-id_$PAR_ID");

Reports::ClientPotential::set_report_log($log);

$log->out("START");

unless ($ONLY_LOGIN) {
    $log->out("Collecting queue stats");
    my $queue = new Reports::Queue(type => 'client_potential', thread => $PAR_ID);
    my $oldest_report = $queue->get_oldest_report(shard => $SHARD);
    my $queue_length  = $queue->get_queue_length(shard => $SHARD);

    monitor_values({
        flow => { "ClientPotentialReports.$PAR_ID" => {
                                     queue_length => { "shard_$SHARD" => $queue_length },
                                     max_age      => { "shard_$SHARD" => ($oldest_report ? time() - $oldest_report->{create_time_ts} : 0) },
                                   },
                },
    });
}

$log->out("Processing reports");
my $error_stat = Reports::ClientPotential::make_reports($SHARD,
    thread => $PAR_ID,
    on_before_new_report => sub {
        my ($report) = @_;
        my $report_id = $report->{id};
        Yandex::Trace::restart(\$ScriptHelper::trace, tags => "$PAR_ID,$report_id");
    },
    ($ONLY_LOGIN ? (login => $ONLY_LOGIN) : ()),
);

$log->out("Writing monitoring data");
juggler_ok(service_suffix => $PAR_ID);

$log->out('FINISH');
