#!/usr/bin/perl

use my_inc '..';

=head1 METADATA

<crontab>
    time: */10 * * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 450
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    time: */10 * * * *
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl:            1h
    tag: direct_group_internal_systems
</juggler>

=head1 NAME

    ppcConvertMediaResources.pl

=head1 DESCRIPTION

    Скрипт проверяет готовность конвертации медиаресурсов, загруженных в видеоплатформу через внутренний отчет "Загрузка нового видео/аудио в видеоплатформу".
    Если в видеоплатформе имеются все нужные нам форматы, то создаем в БК автовидео-креатив.
    Скрипт заполняет таблицу ppcdict.media_resources

    По умолчанию скрипт запускается без опций командной строки, но в целях тестирования и отладки
    можно использовать следующие:
    --once
        выполнить одно задание и выйти
=cut


use Direct::Modern;
use ScriptHelper;
use Settings;

use JSON;
use List::MoreUtils qw/any/;

use Yandex::DBTools;
use Yandex::DBQueue;
use Yandex::HTTP qw/http_parallel_request/;
use Yandex::ListUtils qw/xminus/;
use Yandex::HashUtils qw/hash_cut hash_merge/;
use Yandex::Validate qw/is_valid_int/;

use Property;
use BS::URL;


# видеоплатформа должна вернуть все эти форматы, чтобы конвертация считалась завершенной
my $FORMATS = {
    'sq.mp4' => {
        id => '360p.mp4',
        type => 'video/mp4',
        width => 640,
        height => 360,
        delivery => 'progressive',
    },
    '480p.mp4' => {
        id => '480p.mp4',
        type => 'video/mp4',
        width => 854,
        height => 480,
        delivery => 'progressive',
    },
    '720p.mp4' => {
        id => '720p.mp4',
        type => 'video/mp4',
        width => 1280,
        height => 720,
        delivery => 'progressive',
    },
    'sq.webm' => {
        id => '360p.webm',
        type => 'video/webm',
        width => 640,
        height => 360,
        delivery => 'progressive',
    },
    '480p.webm' => {
        id => '480p.webm',
        type => 'video/webm',
        width => 854,
        height => 480,
        delivery => 'progressive',
    },
    '720p.webm' => {
        id => '720p.webm',
        type => 'video/webm',
        width => 1280,
        height => 720,
        delivery => 'progressive',
    },
};
# сколько раз пытаемся выполнить задание из DB::Queue
my $JOBS_TRYCOUNT = 5;
# сколько спим, если в очереди нет заданий
my $SLEEP_TIME = 90;

# сколько cекунд ожидаем пока ролик сконвертируется
my $WAIT_CONVERT_TIME = 30;
# сколько раз проверяем сконвертировался ли ролик
my $CONVERT_TRIES = 20;

# максимально допустимое значение ppcdict.media_resources.media_resource_id
# в БК media_resource_id отправляется как CreativeID
my $MAX_MEDIA_RESOURCE_ID = 50000;

# Отправляем в БК длительность видеоподложки в секундах независимо от реальной длительности видеоролика.
# В БК ролик будет обрезаться до этой длительности.
# Реальное значение может меняться через ppc_properties
my $DEFAULT_BS_VIDEO_AD_DURATION = 7;

# bs_video_ad_duration ttl. Используется при обращении к Property::get
my $BS_VIDEO_AD_DURATION_TTL = 300;

my ($ONCE, $video_api_url);
extract_script_params(
    'once' => \$ONCE,
);


$log->out("START");

$Yandex::DBQueue::LOG = $log;
my $JOB_QUEUE = Yandex::DBQueue->new(PPCDICT, 'convert_media_resources');


while (1) {
    if (my $reason = smart_check_stop_file()) {
        $log->out("$reason! Exiting.");
        last;
    }
    
    my $bs_ad_duration = _fetch_bs_ad_duration();
    if (my $job = $JOB_QUEUE->grab_job()) {

        if ($job->trycount > $JOBS_TRYCOUNT) {
            $job->mark_failed_permanently({ error => 'Too much tries' });
            $log->die("Job failed: too much tries");
        }

        my ($api_url, $type, $name, $colors, $categories) = @{ $job->args }{qw/api_url type name colors categories/};

        # Проверяем, что конвертация видео завершилась
        my $converted_videos = _check_media_converted($api_url);

        if (defined $converted_videos && @$converted_videos) {
            # media_resource_id не должен превышать $MAX_MEDIA_RESOURCE_ID
            my $max_media_resource_id = get_one_field_sql(PPCDICT, "SELECT IFNULL(max(media_resource_id), 0) FROM media_resources");
            if ($max_media_resource_id >= $MAX_MEDIA_RESOURCE_ID) {
                $job->mark_failed_once();
                $log->die("Current max media_resource_id = $max_media_resource_id is greater that allowed MAX_MEDIA_RESOURCE_ID = $MAX_MEDIA_RESOURCE_ID");
            }
            # получаем длительность ролика и катринки для превью
            my ($duration, $thumbnails) = _get_thumbnails_and_duration($api_url);
            # считаем средний битрейт
            foreach my $converted_video (@$converted_videos) {
                my $size = delete $converted_video->{size};
                # size в байтах, duration в секундах, битрейт считаем в килобитах в секунду
                hash_merge $converted_video, { bitrate => int(($size * 8)/($duration * 1024)) };
            }
            # Сохраняем результат
            my $media_resource_id = do_insert_into_table(
                PPCDICT, 'media_resources',
                {
                    name => $name,
                    yacontextCategories => $categories,
                    colors => $colors,
                    preview_url => encode_json($thumbnails),
                    resources_url => encode_json($converted_videos),
                }
            );

            # создаем в БК видеокреатив
            my $bs_success = _bs_create_autovideo_creative($media_resource_id, $bs_ad_duration, $converted_videos);
            if (!$bs_success) {
                $job->mark_failed_once();
                $log->die("Can't create media_resource_id = $media_resource_id in BS");
            }

            do_update_table(PPCDICT, 'media_resources',  {resource_type => $type}, where => {media_resource_id => $media_resource_id});

            $log->out("mark job as finished");
            my $marked_ok = $job->mark_finished({});
            if ($marked_ok) {
                juggler_ok(description => "Processed job $job->{job_id}");
            } else {
                $log->die("failed to mark job finished");
            }
        } else {
            $job->mark_failed_once();
            $log->die("convertation takes too long time");
        }
    } elsif ($ONCE) {
        $log->out("No jobs found, exiting after one iteration");
        last;
    } else {
        $log->out("No jobs found, sleep for $SLEEP_TIME seconds");
        juggler_ok(description => "No jobs found");
        sleep $SLEEP_TIME;
    }

    last if $ONCE;
}

$log->out("FINISH");


=head2 _check_media_converted

    Проверяем, что загруженный в видеоплатформу файл сконвертировался во все нужны форматы.
    Возвращает ссылку на массив хешей:
    $convert_result = [
        {
          'id' => '720p.mp4',
          'width' => 1280,
          'type' => 'video/mp4',
          'url' => 'https://storage.mdst.yandex.net/get-video-videodirekt/4857/158f7baada5/e94ef66676423fdb/720p.mp4?redirect=yes&sign=5aad8162c0d53559a5659a01d9c1b70089ef65eb7e1900b7fcce281933cc7aec&ts=6b1be2dd',
          'height' => 720,
          'delivery' => 'progressive',
          'bitrate' => 1654
        },
        ...
    ];

=cut

sub _check_media_converted {
    my $api_url = shift;

    my $convert_result;
    for (my $i = 0; $i < $CONVERT_TRIES; $i++) {
        # Ждем пока видео сконвертируется, переодически проверяя готовность
        $log->out("Wait for $WAIT_CONVERT_TIME seconds...");
        sleep $WAIT_CONVERT_TIME;
        $log->out("Check convertation result");
        my $api_params = {
            uid => $Settings::VIDEO_API_UID,
            format => 'json',
        };
        my $vfiles_url = Yandex::HTTP::make_url($api_url. 'vfiles', $api_params);
        my $resp = Yandex::HTTP::http_parallel_request(
            GET => { 0 => {url => $vfiles_url} },

            timeout => 10,
            soft_timeout => 0.5,
            num_attempts => 2,
        )->{0};
        if ($resp->{is_success}) {
            my $data = decode_json($resp->{content}) || {};
            # ответе должны быть все нужные нам форматы
            if (@{$data->{files}} && !@{xminus([ keys %$FORMATS ], [ map { $_->{name} } @{$data->{files}}])}) {
                $log->out("Media is converted in all formats");
                foreach my $file (@{$data->{files}}) {
                    if (any {$file->{name} eq $_} keys %$FORMATS) {
                        push @$convert_result, {
                            url => $file->{getUrl},
                            size => $file->{size},
                            %{hash_cut($FORMATS->{$file->{name}}, qw/id type width height delivery/)},
                        };
                    }
                }
                last;
            } else {
                $log->out("Convertation is not finished");
            }
        }
    }

    return $convert_result;
}

=head2 _get_thumbnails_and_duration

    Получить картинки для превью.
    Принимает api-ссылку видеоплатформы
    Возращает массив с длительностью ролика и массивом хешей с картинками для превью:
    $result = [
        30,
        [
            {
                height => 90,
                width  => 120,
                url    => "https://static.vd.yandex.net/get/videodirekt/m-4857-158f7b9cf96-e633368d57a9a6fe/120x90.jpg",
            },
            ...
        ]
    ];

=cut

sub _get_thumbnails_and_duration {
    my $api_url = shift;

    my ($duration, $thumbnails);

    my $resp = Yandex::HTTP::http_parallel_request(
        GET => { 0 => {url => Yandex::HTTP::make_url($api_url, {uid => $Settings::VIDEO_API_UID, format => 'json'})} },
        timeout => 10,
        soft_timeout => 0.5,
        num_attempts => 2,
    )->{0};

    if ($resp->{is_success}) {
        my $data = decode_json($resp->{content}) || {};
        $duration = $data->{duration};
        $thumbnails = $data->{thumbnails};
    } else {
        $log->out("Can't get humbnails and duration")
    }

    return ($duration, $thumbnails);
}

=head2 _bs_create_autovideo_creative

    Создание видеокреатива в БК

=cut

sub _bs_create_autovideo_creative {
     my ($creative_id, $duration, $media_files) = @_;

     my $result;

     my $url = BS::URL::get('import_auto_video_creative');
     my $resp = Yandex::HTTP::http_parallel_request(
        GET => {
            0 => {
                url => Yandex::HTTP::make_url($url, {request => encode_json({CreativeID => $creative_id, duration => $duration, theme => 'video-banner_theme_caucasus', mediaFiles => $media_files})})
            },
        },
        timeout => 10,
        soft_timeout => 0.5,
        num_attempts => 2,
     )->{0};

     if ($resp->{is_success}) {
         my $data = decode_json($resp->{content}) || {};
         $result = 1 if $data->{success};
     }

     return $result;
}


=head2 _fetch_bs_ad_duration

    Вытаскивает(из ppc_properties) значение длительности видеоподложек. 
    В случае, если такого значения нет в базе - используется значение по-умолчанию DEFAULT_BS_VIDEO_AD_DURATION. 
    Если есть, но не int или меньше 1 - завершает исполнение скрипта.
    Значение переменной меняется через внутренний отчет 'Изменение длительности видеоподложек '.

=cut

sub _fetch_bs_ad_duration {
    state $prop //= Property->new($Settings::BS_VIDEO_AD_DURATION_LIMIT_PROP);
    my $db_property_value = $prop->get($BS_VIDEO_AD_DURATION_TTL);
    if (is_valid_int($db_property_value, 1)) {
        return $db_property_value;
    } else {
        return $DEFAULT_BS_VIDEO_AD_DURATION; 
    }
}

