#!/usr/bin/perl

use my_inc '..';

=head1 METADATA

<crontab>
   time: 39 5 * * *
   package: scripts-switchman
   <switchman>
       group: scripts-other
   </switchman>
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 2d4h
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

    ppcCopySubclientsForceCurrency.pl

=head1 DESCRIPTION

    Размножает признак принятости оферты с агентства на всех его несвободных субклиентов.

=head1 RUNNING

    Можно запускать по конкретным агентствам:
        --login — логин агентства или одного из его представителей

    ./protected/ppcCopySubclientsForceCurrency.pl
    ./protected/ppcCopySubclientsForceCurrency.pl --login itagency

=cut

use Direct::Modern;

use Settings;
use ScriptHelper;

use List::MoreUtils qw(uniq);

use Client;
use PrimitivesIds;
use RBACDirect;
use RBACElementary;
use RBAC2::Extended;
use RBACDirect qw(rbac_get_all_agencies);
use Campaign::Types qw(get_camp_kind_types);

use Yandex::DBTools;
use Yandex::DBShards;

my $OPERATOR_UID = 1;
my $AGENCY_CHUNK_SIZE = 1_000;
my $SUBCLIENT_CHUNK_SIZE = 1_000;

my $logins = [];
extract_script_params(
    'login=s@' => $logins,
);

$log->out('START');

$log->out('Connecting to RBAC');
my $rbac = eval { RBAC2::Extended->get_singleton(1) } or $log->die("Can't connect to RBAC: $@");

my $agency_chief_uids;
if ($logins && @$logins) {
    $log->out('Working only on logins:', $logins);
    $agency_chief_uids = [ map { rbac_get_chief_rep_of_agency_rep($_) } @{ get_uids(login => $logins) } ];
} else {
    $log->out('Fetching agencies list');
    $agency_chief_uids = rbac_get_all_agencies_chief_reps($rbac);
    $log->out('Got ' . scalar(@$agency_chief_uids) . ' agency chief uids');
}

foreach_shard uid => $agency_chief_uids, chunk_size => $AGENCY_CHUNK_SIZE, sub {
    my ( $agency_shard, $agency_chief_uids_chunk ) = @_;

    $log->out('Fetching agencies ClientIDs');
    my $ag_chief_uid2clid = rbac_get_agencies_clientids_by_uids( $agency_chief_uids_chunk);
    my $agency_clids_chunk = [ uniq values %$ag_chief_uid2clid ];
    $log->out('Got ' . scalar(@$agency_clids_chunk) . ' agency ClientIDs');

    my $agid2chief_uid = { reverse %$ag_chief_uid2clid };
    undef $ag_chief_uid2clid;

    $log->out('Fetching agencies with accepted oferta');
    my $ag_clids2accepted_at = get_hash_sql(PPC(shard => $agency_shard), ['
        SELECT fcc.ClientID, fcc.accepted_at
        FROM force_currency_convert fcc
     ', WHERE => {
            'fcc.ClientID' => $agency_clids_chunk,
        },
    ]);
    $log->out('Got ' . scalar(keys %$ag_clids2accepted_at) . ' agencies with accepted oferta');
    return unless scalar(keys %$ag_clids2accepted_at);

    my @agency_clientids = keys %$ag_clids2accepted_at;

    for my $ag_clid (@agency_clientids) {
        my $msg_prefix_guard = $log->msg_prefix_guard("[AgencyID $ag_clid]");

        my $accepted_at = $ag_clids2accepted_at->{$ag_clid};
        my $chief_uid = $agid2chief_uid->{$ag_clid};
        if (!$chief_uid) {
            $log->out("Skipping AgencyID $ag_clid because of lack of chief uid");
            next;
        }
        $log->out("Fetching all subclients for uid $chief_uid");
        my $subclient_clids = rbac_get_subclients_clientids($rbac, $chief_uid);

        foreach_shard ClientID => $subclient_clids, chunk_size => $SUBCLIENT_CHUNK_SIZE, sub {
            my ( $client_shard, $subclient_clids_chunk ) = @_;

            # отсекаем свободных по галке и по наличию кампаний разных типов
            # также нас не интересуют валютные и те, у кого оферта уже принята
            $log->out('Fetching subclients to mark oferta accepted');
            my $clientids_to_enable = get_one_column_sql(PPC(shard => $client_shard), ['
                SELECT u.ClientID
                FROM users u
                LEFT JOIN clients cl ON u.ClientID = cl.ClientID
                LEFT JOIN force_currency_convert fcc ON u.ClientID = fcc.ClientID
                LEFT JOIN campaigns c ON u.uid = c.uid
             ', WHERE => {
                    'u.ClientID' => $subclient_clids_chunk,
                    _TEXT => q/
                            IFNULL(cl.allow_create_scamp_by_subclient, "No") = "No"
                        AND IFNULL(cl.work_currency, "YND_FIXED") = "YND_FIXED"
                    /,
                    'c.type' => get_camp_kind_types('currency_convert'),
                    'fcc.ClientID__is_null' => 1,
                },'
                GROUP BY u.ClientID
                HAVING COUNT(DISTINCT IFNULL(c.AgencyID,0)) <= 1
            ']);

            my @to_insert;
            for my $client_id (@$clientids_to_enable) {
                $log->out("Copying oferta acceptance for ClientID $client_id as agency ClientID $ag_clid accepted it $accepted_at");
                # ClientID, accepted_at
                push @to_insert, [$client_id, $accepted_at];
            }
            $log->out('Inserting oferta acceptance data');
            do_mass_insert_sql(PPC(shard => $client_shard), 'INSERT IGNORE INTO force_currency_convert (ClientID, accepted_at) VALUES %s', \@to_insert);
        };
    }
};

$log->out('Sending OK to juggler');
juggler_ok();

$log->out('FINISH');
