#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: */10 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host: checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 1h
    <notification>
        template: on_status_change
        status: CRIT
        status: OK
        method: telegram
        login: direct-bill-agg
    </notification>
</juggler>

=cut

=head1 NAME

    ppcCreateBillingAggregates.pl

=head1 DESCRIPTION

    Читает задание из ppc_properties PPC_CREATE_BILLING_AGGREGATES_SHARD_*,
    где указан список или диапазон клиентов, которые нужно проверить.
    Задания создаются через внутренний отчет
    https://direct.yandex.ru/registered/main.pl?cmd=internalReports&report_id=create_billing_aggregates_script_mgr

    Проверяет, есть ли у клиентов все биллинговые агрегаты, которые нужны
    для откруток кампаний клиента. Если нет, создает эти биллинговые агрегаты.

=head1 RUNNING

    ./protected/ppcCreateBillingAggregates.pl --shard-id 1

=cut


use Direct::Modern;

use JSON;
use List::MoreUtils qw/any uniq/;
use Readonly;

use Yandex::DBTools;
use Yandex::DBShards;

use my_inc "..";

use BillingAggregateTools;
use Campaign::Types;
use Direct::BillingAggregates;
use Direct::Feature;
use Direct::Wallets;
use Property;
use Rbac;
use Settings;
use Yandex::Clone;

use ScriptHelper sharded => 1;

my $task_prop = Property->new("PPC_CREATE_BILLING_AGGREGATES_SHARD_".$SHARD);
Readonly my $CLIENT_ID_CHUNK_SIZE => 1_000;
Readonly my $OPERATOR_UID => 1;

$log->out("START");

my $task_json = $task_prop->get();
if ($task_json && _task_not_finished($task_json)) {
    _process_task($task_json, $task_prop);
}

juggler_ok();

$log->out("FINISH");

sub _task_not_finished {
    my ($task_json) = @_;

    my $finished = 0;
    my $task = from_json($task_json);
    if ($task->{client_id_list} && ($task->{finished} // 0) == 1) {
        $finished = 1;
    } elsif ($task->{client_id_from} && $task->{client_id_from} >= $task->{client_id_to}) {
        $finished = 1;
    }

    return !$finished;
}

sub _process_task {
    my ($task_json, $task_prop) = @_;

    my $task = from_json($task_json);
    $log->out("processing task", $task);

    my $next_task = yclone($task);
    my @client_ids;
    if ($task->{client_id_list}) {
        @client_ids = @{$task->{client_id_list}};
        $next_task->{finished} = 1;
    } elsif ($task->{client_id_from})  {
        my @add_where;
        if ($task->{with_camp_types}) {
            push @add_where, (_TEXT => '
                EXISTS (
                    SELECT 1
                    FROM campaigns c
                    WHERE
                        c.ClientID = cl.ClientID
                        AND '.sql_condition(['c.type' => $task->{with_camp_types}]).'
                )'
            );
        }

        @client_ids = @{get_one_column_sql(PPC(shard => $SHARD), ["
            SELECT cl.ClientID
            FROM clients cl",
            WHERE => [
                'cl.ClientID__ge' => $task->{client_id_from},
                'cl.ClientID__lt' => $task->{client_id_to},
                @add_where,
            ], "
            ORDER BY cl.ClientID
            LIMIT $CLIENT_ID_CHUNK_SIZE"
        ])};

        if (scalar(@client_ids) < $CLIENT_ID_CHUNK_SIZE) {
            $next_task->{client_id_from} = $task->{client_id_to};
        } else {
            $next_task->{client_id_from} = $client_ids[-1] + 1;
        }
    }

    if (@client_ids) {
        _process_clients(\@client_ids);
    }

    $task_prop->cas(to_json($next_task, {canonical => 1}));
}

sub _process_clients {
    my ($client_ids) = @_;

    # массово выясняем, есть ли у клиетов доступ к фиче автосоздания, чтобы позже не ходить
    # 1000 раз в IntAPI
    my $feature_by_clients = _get_feature_by_clients($client_ids);

    # здесь выбираем из БД все кампании-общие счета клиентов, без проверки на то, включены они или нет.
    # Это можно тут делать, т.к. биллинговые агрегаты будут созданы под этими общими счетами только в случае,
    # если под ним есть кампании, для которых включено автосоздание БА.
    # т.е. проверка включенности ОС отложена до следующего шага
    my $wallets = Direct::Wallets->get_by(client_id => $client_ids,
        no_additional => 1,
        filter => {
            'c.currency__ne' => 'YND_FIXED',
        }
    )->items;
    my @wallet_ids = map { $_->id } @$wallets;
    my $existing_agg = Direct::BillingAggregates->get_by(wallet_id => \@wallet_ids)->items_by_wallet_and_product();

    my $camp_types_by_wallet = _get_camp_types_by_wallet(\@wallet_ids);

    for my $wallet (@$wallets) {
        my $client_id = $wallet->client_id;
        my $wallet_id = $wallet->id;
        my $camp_types = $camp_types_by_wallet->{$wallet_id};

        my $prefix_guard = $log->msg_prefix_guard("[shard_$SHARD, ClientID $client_id, wallet_cid $wallet_id]");
        BillingAggregateTools::check_and_add_new_aggregates($client_id, $wallet, $camp_types, $OPERATOR_UID
            , log => $log
            , preloaded_feature_by_clients => $feature_by_clients
            , preloaded_existing_aggs => $existing_agg
        );
    }
}

sub _get_feature_by_clients {
    my ($client_ids) = @_;

    my $client2info = Rbac::get_key2perminfo(ClientID => $client_ids);
    my @agency_ids = uniq map { $_->{agency_client_id} } grep { $_->{agency_client_id} } values %$client2info;

    my $feature_response = Direct::Feature::get_clients_uids_features([@$client_ids, @agency_ids], undef);
    if ($feature_response->{error}) {
        utf8::decode($feature_response->{error}) unless utf8::is_utf8($feature_response->{error});
        $log->die('error fetching clients features: '.$feature_response->{error});
    }

    my %result;
    for my $client_id (@$client_ids) {
        $result{$client_id} = 1;
        if (my $agency_id = $client2info->{$client_id}{agency_client_id}) {
            if (_extract_feature_from_response($feature_response, $agency_id, 'disable_billing_aggregates')) {
                $result{$client_id} = 0;
            }
        }
        if (_extract_feature_from_response($feature_response, $client_id, 'disable_billing_aggregates')) {
            $result{$client_id} = 0;
        }
    }

    return \%result;
}

sub _extract_feature_from_response {
    my ($feature_response, $client_id, $feature) = @_;

    my @client_features = @{$feature_response->{per_client_id_features}{$client_id} // []};
    return any { $_ eq $feature } @client_features;
}

sub _get_camp_types_by_wallet {
    my ($wallet_ids) = @_;

    my $rows = get_all_sql(PPC(cid => $wallet_ids), ["
        SELECT DISTINCT wallet_cid, type
        FROM campaigns",
        WHERE => [
            wallet_cid => SHARD_IDS,
            statusEmpty__ne => 'Yes',
            type__not_in => Campaign::Types::get_camp_kind_types('without_billing_aggregates'),
        ]
    ]);

    my %result;
    for (@$rows) {
        push @{$result{$_->{wallet_cid}}}, $_->{type};
    }

    return \%result;
}
