#!/usr/bin/perl

use my_inc "..";


=head1 METADATA

<crontab>
    time: */5 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 300
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 1h
    tag: direct_group_internal_systems
    <notification>
        template: on_status_change
        status: OK
        status: CRIT
        method: telegram
        login: DISMonitoring
    </notification>
</juggler>

# На ТС скрипт не запускается, конвертация проверяется фейковой ручкой, которая сама ставит и забирает задания и запускает код конвертации в апаче

=cut

=head1 DESCRIPTION

    Ставит заявки на переход в реальную валюту в очередь DBQueue,
    отслеживает состояние поставленных заявок и собирает данные
    для мониторинга.

    Раньше этот скрипт ставил задания в Gearman, поэтому
    вся логика по управлению очередью вынесена в воркеров (отмечают 
    начало/окончание выполнение задания, проверяют не выполнено ли 
    ещё задание), т.к. в Gearman очередь была не персистентна и не было 
    надёжной возможности контроллировать стоит ли сейчас определённая
    задача в очереди. Задачи ставятся в очередь по несколько раз.
    Уже поставленные задачи снова ставить в очередь начинаем только 
    через $TASK_REQUEUE_INTERVAL минут и только если в dbqueue нет задач, чтобы совсем не зафлудить очередь. 

    Параметры:

    --help, -h
        показать справку

    --once
        выполнить одну итерацию скрипта и завершить работу

    --clientid
        работать только с указанным ClientID

=head1 EXAMPLE

    LOG_TEE=1 ./protected/ppcCurrencyConvertMaster.pl
    LOG_TEE=1 ./protected/ppcCurrencyConvertMaster.pl --once --clientid 12345,67890 --clientid

=cut

use strict;
use warnings;

use List::MoreUtils qw/none/;

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use ScriptHelper get_file_lock => ['dont_die'],
                 sharded => 1;
use LockTools;
use Client;

use RBACElementary;
use RBAC2::Extended;
use Yandex::HashUtils;
use Client::ConvertToRealMoney;
use Yandex::Validate;
use Yandex::DBQueue;

use utf8;
use open ':std' => ':utf8';

=head2 $SLEEP_BETWEEN_ITERATIONS

    Сколько секунд спать между итерациями скрипта

=cut

my $SLEEP_BETWEEN_ITERATIONS = 10;

=head2 $TASKS_COUNT_LIMIT_FOR_ITERATION

    Количество задач, которые скрипт будет выбирать из БД 
    и пытаться поставить в очередь за одну итерацию

=cut

my $TASKS_COUNT_LIMIT_FOR_ITERATION = 50;

=head2 $TASK_REQUEUE_INTERVAL

    Количество минут, в течение которых не ставим повторно задачу в очередь.

    Если задача не завершится в течении этого времени, то в очереди будет несколько 
    копий задачи. Т.к. управление очередью происходит в воркерах, а воркеры 
    захватывают поклиентные SQL-локи, то это не страшно. Получив такое задание,
    воркер сразу же завершится или увидев, что всё уже выполнено, или если не 
    сможет получить лок.

    Если задача потеряется (gearmand перезапустится, например), то
    снова в очередь она попадёт не раньше, чем через этот интервал.

=cut

my $TASK_REQUEUE_INTERVAL = 60;

my $OPERATOR_UID = 1;

run() unless caller();

sub run {
    my $SQL_LOCK_NAME = "ppcCurrencyConvertMaster_LOCK";

    my ($ONCE, @CLIENT_IDS);
    extract_script_params(
        'once' => \$ONCE,
        'clientid=s' => \@CLIENT_IDS,
    );

    @CLIENT_IDS = split(/,+/, join(',', @CLIENT_IDS));
    for my $client_id (@CLIENT_IDS) {
        if (!is_valid_int($client_id, 0)) {
            $log->die("invalid clientid given: $client_id");
        }
    }

    # захватываем лок в БД, чтобы предотвратить одновременный запуск скрипта на нескольких машинах
    my $sql_lock = sql_lock_guard(PPC(shard => $SHARD), $SQL_LOCK_NAME, 1);

    $log->out('START');

    my %submited_tasks; # ${submited_tasks}->{<state>}->{ClientID} = <время добавления в очереди> чтобы выдерживать интервал $TASK_REQUEUE_INTERVAL
    while (1) {
        Yandex::Trace::restart(\$ScriptHelper::trace);

        if (my $reason = smart_check_stop_file()) {
            $log->out("$reason! Exiting.");
            exit 0;
        }

        $log->out('Start of new iteration');
        if (@CLIENT_IDS) {
            $log->out('Working only on following ClientIDs:', \@CLIENT_IDS);
        }
        my $need_another_iteration = one_currency_convert_iteration(\%submited_tasks, \@CLIENT_IDS);

        if (!$ONCE) {
            juggler_ok();
            my $sleep_time = $need_another_iteration ? 1 : $SLEEP_BETWEEN_ITERATIONS;
            $log->out("Sleeping for $sleep_time seconds");
            sleep $sleep_time;
        } else {
            $log->out('One iteration done. Exitting because of --once argument given.');
            last;
        }
    }

    $log->out('FINISH');

    release_file_lock();
}

=head2 one_currency_convert_iteration

    Результат:
        $need_another_iteration - 0/1 - нужна ли еще итерация сразу (1), или можно "поспать"

=cut

{
my ($queue, $rbac);
sub one_currency_convert_iteration {
    my ($submited_tasks, $client_ids) = @_;

    if (!defined $queue) {
        $log->out("Creating DBQueue object");
        $queue = Yandex::DBQueue->new(PPC(shard => $SHARD), 'convert_to_real_money');
    }

    if (!defined $rbac) {
        $rbac = RBAC2::Extended->get_singleton($OPERATOR_UID);
    }

    # удаляем просроченные записи о времени добавления в очередь, если нет заданий в dbqueue
    my $border_time = time() - $TASK_REQUEUE_INTERVAL*60;
    while (my ($state, $client_id_data) = each %$submited_tasks) {
        if ($client_id_data && %$client_id_data) {
            while (my($client_id, $last_queue_time) = each %$client_id_data) {
                if ($last_queue_time <= $border_time) {
                    my $jobs_for_client_limit = 1000;
                    my @jobs_for_client = @{ $queue->find_jobs(ClientID => $client_id, skip_archived => 1, limit => $jobs_for_client_limit + 1) };
                    $log->out("found " . scalar(@jobs_for_client) . " jobs in dbqueue for $client_id and state $state");
                    $log->warn("number of jobs for $client_id and state $state exceeds $jobs_for_client_limit") if (@jobs_for_client > $jobs_for_client_limit);
                    delete $client_id_data->{$client_id} if !grep { $_->{args}->{state} eq $state } @jobs_for_client;
                }
            }
        }
    }

    my @not_to_requeue_conds;
    while (my ($state, $client_id_data) = each %$submited_tasks) {
        push @not_to_requeue_conds, 'AND NOT (';
        push @not_to_requeue_conds, {state => $state, ClientID => [keys %$client_id_data]};
        push @not_to_requeue_conds, ')';
    }

    my @additinal_conds;
    if ($client_ids && @$client_ids) {
        push @additinal_conds, 'AND', {ClientID => $client_ids};
    }
    my $tasks_to_add = get_all_sql(PPC(shard => $SHARD), ['
        SELECT ClientID, uid, convert_type, state, new_currency, country_region_id, email, start_convert_at
        FROM currency_convert_queue
        WHERE state <> "DONE" AND (
            (start_convert_at < CURRENT_TIMESTAMP AND balance_convert_finished = 1) -- начинаем конвертацию только после окончания компенсаций по общим счетам в Балансе
            OR state = "NEW"    -- уведомляем Баланс сразу, не дожидаемся времени перехода
        )', @not_to_requeue_conds, @additinal_conds, '
        ORDER BY start_convert_at
        LIMIT ?'], $TASKS_COUNT_LIMIT_FOR_ITERATION);

    my $need_another_iteration = 0;
    if ($tasks_to_add && @$tasks_to_add) {
        $log->out('Processing ' . scalar(@$tasks_to_add) . ' tasks');
        if (scalar(@$tasks_to_add) == $TASKS_COUNT_LIMIT_FOR_ITERATION) {
            $need_another_iteration = 1;
        }
        my %tasks_to_submit;
        for my $task(@$tasks_to_add) {
            my $client_id = $task->{ClientID};

            # на всякий случай проверяем, что клиента действительно можно переводить
            # в состояниях NOTIFY и CONVERTING_DETAILED_STAT клиент уже по факту переведён и осталось только ему сообщить об этом
            if (none {$task->{state} eq $_} qw/NOTIFY CONVERTING_DETAILED_STAT FETCHING_BALANCE_DATA OVERDRAFT_WAITING/) {
                my $client_nds = get_client_NDS($client_id);
                my $client_currencies = get_client_currencies($client_id, allow_initial_currency => 0);
                my $client_chief_uid = rbac_get_chief_rep_of_client($client_id);
                my $error = Client::can_convert_to_real_money(
                    ClientID => $client_id,
                    NDS => $client_nds,
                    client_currencies => $client_currencies,
                    client_chief_uid => $client_chief_uid,
                    ignore_request_in_queue => 1,
                );
                if (defined $error) {
                    $log->warn("Cannot convert client with ClientID $client_id into real currency: $error");
                    next;
                }
            }

            $task->{action} = Client::ConvertToRealMoney::get_next_convert_task_name($task->{convert_type}, $task->{state});
            if (!$task->{action}) {
                $log->die("Cannot determine which action should be run for task in state $task->{state}");
            }

            my $job_id = get_new_id('job_id');
            my $job = $queue->insert_job({
                job_id => $job_id,
                ClientID => $task->{ClientID},
                args => $task,
            });
            $log->out("Adding job $job_id for action '$task->{action}' to DBQueue to convert currency for ClientID $task->{ClientID}");
            $tasks_to_submit{$task->{state}}->{$client_id} = time();
        }
        while (my($state, $clientid2time) = each %tasks_to_submit) {
            $submited_tasks->{$state} ||= {};
            hash_merge $submited_tasks->{$state}, $clientid2time;
        }
    } else {
        $log->out('No tasks found');
    }
    return $need_another_iteration;
}
}
