#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: */5 * * * *
    ulimit: -v 10000000
    package: scripts-switchman
    sharded: 1
    params: --uniq 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 1024
        </leases>
    </switchman>
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    raw_events:     scripts.ppcCurrencyConvertWorker.working.uniq_1.shard_$shard
    sharded:        1
    ttl:            12h
    tag: direct_group_internal_systems
    <notification>
        template: on_status_change
        status: OK
        status: CRIT
        method: telegram
        login: DISMonitoring
    </notification>
</juggler>

=cut

=head1 NAME

ppcCurrencyConvertWorker.pl

=head1 DESCRIPTION

Скрипт, обрабатывающий задания по конвертации из DBQueue.
Задание = этап конвертации = функция из Client::ConvertToRealMoneyTasks.
Если передан параметр --once, обработает одно задание и выйдет.

=cut

use Direct::Modern;
use my_inc '..';

use List::Util qw(max);
use Sys::Hostname qw/hostname/;
use Time::HiRes;

use Yandex::ProcInfo;
use Yandex::DBQueue;
use Yandex::Log;
use Yandex::Trace;

use Client::ConvertToRealMoneyTasks;
use LockTools;
use ScriptHelper sharded => 1, get_file_lock => undef, 'Yandex::Log' => 'messages';
use Settings;

# Таймаут на конвертирование клиента в реальную валюту (в секундах)
our $CONVERT_TO_REAL_MONEY_TIMEOUT = 10 * 60 * 60; # 10 часов

# Сколько минут ждать остановки кампании при конвертации копированием.
# Если за это время кампания не остановится в БК, то всё равно уносим
# с кампании все деньги.
our $TIME_TO_WAIT_FOR_CAMPAIGN_STOP = 40;

# Сколько минут ждать нотификации от Баланса с пересчитанной в валюту суммой овердрафта
# Если за это время нотификация не придёт, продолжим конвертировать клиента с нулевым овердрафтом.
our $TIME_TO_WAIT_FOR_OVERDRAFT = 40;

# максимальное количество попыток выполнить шаг конвертации
my $MAX_PROCESSING_TIMES = 5;

my $MAX_PROC_SIZE = 1.5 * 1024 * 1024 * 1024; # 1.5 ГБ

# максимальное количество итераций цикла "выбрать задачу - сделать задачу";
# итерации, в которых задачи не нашлось, считаются
my $MAX_ITERATIONS = 10000;
my $ITERATION_NUMBER = 0;

# Максимальное время жизни процесса (сек)
my $PROCESS_LIFETIME = 60*60;

# сколько спать при отсутствии отчёта (сек)
my $IDLE_SLEEP_TIME = 5;

my $PROC_FATHER_ID = hostname() or die("Can't get hostname");
# идентификация процесса в логах: "$PROC_FATHER_ID/$UNIQ/$$"
my $FULL_PROC_ID;

my $PROCESS_START_TIME = time();

my $JUGGLER_SERVICE_SUFFIX = '';

my $JOB_QUEUE;

# на сколько секунд процесс может захватить задание в очереди
my $GRAB_DURATION = 60 * 60 * 1.5;

my ($ONCE, $UNIQ);
run() unless caller();

sub run {
    extract_script_params(
        'uniq=s' => \$UNIQ,
        'once' => \$ONCE,
    );
    die '--uniq param is mandatory!' unless defined $UNIQ;

    if ( !get_file_lock( 'dont die', get_script_name() . ".$UNIQ" ) ) {
        exit 0;
    }
    $log->msg_prefix("[shard_$SHARD,uniq_$UNIQ]");

    $Client::ConvertToRealMoneyTasks::log = $log;
    $Yandex::DBQueue::LOG = $log;

    $JUGGLER_SERVICE_SUFFIX .= "uniq_$UNIQ";

    $FULL_PROC_ID = "$PROC_FATHER_ID/$UNIQ/$$";

    $log->out("start daemon $FULL_PROC_ID");

    $JOB_QUEUE = Yandex::DBQueue->new(PPC(shard => $SHARD), 'convert_to_real_money');

    while (1) {
        # дублирует проверку в should_quit() на случай, если до проверки в should_quit() не доберёмся (например, где-нибудь посередине появится next)
        last if ++$ITERATION_NUMBER > $MAX_ITERATIONS;
        my $idle = 0;

        if (my $job = $JOB_QUEUE->grab_job(grab_for => $GRAB_DURATION)) {
            work_on_job($job);
        } else {
            $idle = 1;
        }

        juggler_ok( service_suffix => $JUGGLER_SERVICE_SUFFIX );

        last if should_quit();
        idle() if $idle;

        restart_tracing('worker');
    }

    $log->out("end daemon $FULL_PROC_ID");

    exit 0;
}

sub work_on_job {
    my ($job) = @_;

    my $job_id = $job->job_id;
    my $ClientID = $job->ClientID;

    if ( $job->trycount > $MAX_PROCESSING_TIMES ) {
        $log->out({ error => "rank_exceed_limit", job_id => $job_id, trycount => $job->trycount });

        my $marked_ok = $job->mark_failed_permanently( { error => 'Unknown error' } );
        $log->out( "failed to mark $job_id failed (permanently) - another process handled it?") unless $marked_ok;
        return;
    }

    my $action = $job->args->{action};
    $log->out("$FULL_PROC_ID process start for job $job_id, action '$action'");

    my $starttime = Time::HiRes::time();

    my $error = '';
    my $ok = eval {
        local $SIG{ALRM} = sub { die "timeout" };
        alarm $CONVERT_TO_REAL_MONEY_TIMEOUT;
        $Client::ConvertToRealMoneyTasks::ACTION_MAP{$action}->($job->args);
        1;
    };
    alarm 0;
    $error = $@ if !$ok;

    my $convert_step_time = Time::HiRes::time() - $starttime;

    if ( $ok ) {
        my $marked_ok = $job->mark_finished( {
            time => $convert_step_time,
        } );

        $log->out( "failed to mark $job_id finished - another process handled it?") unless $marked_ok;
    } else {
        $error ||= 'Unknown error';
    	$log->out({ error => $error, job_id => $job_id, trycount => $job->trycount });

        if ( $job->trycount >= $MAX_PROCESSING_TIMES ) {
            $log->out({ error => "rank_exceed_limit", job_id => $job_id, trycount => $job->trycount });

            my $marked_ok = $job->mark_failed_permanently( { error => $error } );
            $log->out( "failed to mark $job_id failed (permanently) - another process handled it?") unless $marked_ok;
        } else {
            my $marked_ok = $job->mark_failed_once;
            $log->out( "failed to mark $job_id failed once - another process handled it?") unless $marked_ok;
        }
    }

    $log->out("process time for job $job_id: ".sprintf("%.3f", ( $convert_step_time ) )." sec, trycount = ".$job->trycount);
}

=head2 should_quit()

Проверки, что процесс должен завершиться, потому что:
    а) слишком много памяти ест
    б) слишком долго работает
    в) получил SIGTERM или версия Директа поменялась (использует стандартный smart_check_stop_file)
    г) сделали слишко много итераций
    д) нас просили сделать только одну итерацию

Возвращает 0 (надо работать дальше) или 1 (надо завершиться).

=cut

sub should_quit {
    return 1 if $ONCE;
    # перезапускаем процесс, если он начал занимать слишком много памяти 
    if ( proc_memory() > $MAX_PROC_SIZE ) {
        $log->out("$FULL_PROC_ID uses too much memory, quitting after $ITERATION_NUMBER iterations: " . proc_status_str());
        return 1
    # или процесс работает уже слишком долго (смерть от старости)
    } elsif ( Time::HiRes::time() - $PROCESS_START_TIME > $PROCESS_LIFETIME) {
        $log->out("$FULL_PROC_ID is too old (started at " . localtime($PROCESS_START_TIME) . "), quitting");
        return 1;
    # или уже сделали достаточно итераций
    } elsif ( $ITERATION_NUMBER >= $MAX_ITERATIONS ) {
        $log->out("$FULL_PROC_ID: done $MAX_ITERATIONS iterations, quitting");
        return 1;
    # или получен SIGTERM/изменилась версия Директа
    } elsif ( my $reason = smart_check_stop_file() ) {
        $log->out("$FULL_PROC_ID: $reason, quitting");
        return 1;
    }

    return 0;
}

sub idle {
    $log->out("No currency convert job selected, sleeping for $IDLE_SLEEP_TIME seconds");
    my $profile = Yandex::Trace::new_profile("ppcCurrencyConvertWorker:sleep");
    sleep $IDLE_SLEEP_TIME;
}

1;
