#!/usr/bin/env perl

use my_inc "..";

=head1 METADATA

<crontab>
    time: */5 * * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 60
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl:            1h
    tag: direct_group_internal_systems
</juggler>

<monrun>
    juggler_host: checks_auto.direct.yandex.ru
    tag: direct_group_internal_systems
    name: ExportDomainsToBalance.flow
    span: 3d
    warn: 0
    vars: MinDailyInc=1000
    expression: 'diffSeries(offset(keepLastValue(timeShift(direct_one_min.db_configurations.production.flow.ExportDomainsToBalance.last_syncid,"1d")),$MinDailyInc),keepLastValue(direct_one_min.db_configurations.production.flow.ExportDomainsToBalance.last_syncid))'
</monrun>

=cut

=head1 DESCRIPTION

    $Id$
    Синхронизация базы url в Балансе со списком доменов для клиентов Директа.
        (CRM забирает домены из Баланса)

=cut

use Direct::Modern;
use open ':std' => ':utf8';

use ScriptHelper 'get_file_lock' => [1800];

use Settings;
use Yandex::DBTools;
use Yandex::Advmon;
use Yandex::Balance;
use Yandex::ScalarUtils;
use Yandex::Overshard;
use Property;

my $BANNER_CNT_LIMIT = 50_000;

my $RPC_CHUNK_LENGTH = 200;
my $RPC_TIMEOUT = 600;

$log->out("START");

# Вспоминаем последний отправленный нами syncid
my $prop_last_syncid = new Property("ExportDomainsToBalance_last_syncid");
my $last_syncid = $prop_last_syncid->get();

unless (defined $last_syncid) {
    $log->out("No property last_syncid found. Exiting.");
    exit;
}

$log->out("Starting with last_syncid: $last_syncid");

# Получаем полный список URL и доменов из промодерированных баннеров в оплаченных кампаниях
my $records = overshard order => 'syncId', limit => $BANNER_CNT_LIMIT, get_all_sql(PPC(shard => 'all'), [q{
    SELECT
        sync_id AS syncId, record_id AS recordId, ClientID AS clientId, camp_type AS type, domain_id, removed,
        DATE_FORMAT(logtime, '%Y%m%dT%H:%i:%s') AS logDate
    FROM
        client_domains
    }, where => {
        sync_id__gt => $last_syncid,
    },
]);
enrich_data $records, using => 'domain_id', sub {
    my $ids = shift;
    return get_hashes_hash_sql(PPCDICT, [q{SELECT domain_id, domain FROM domains_dict}, where => {domain_id => $ids}]);
};

# Проверим и подготовим полученные записи
my @records_to_send;
for my $record (@$records) {
    if (!$record->{domain}) {
        $log->warn({"Found record without domain:" => $record});
        next;
    }
    delete $record->{domain_id};
    push @records_to_send, $record;
}

$log->out("Sending data to Balance");
while(my @chunk = splice @records_to_send, 0, $RPC_CHUNK_LENGTH) {
    my $res;
    eval {
        $log->out('try to send');
        $res = balance_add_client_domain(\@chunk, timeout => $RPC_TIMEOUT);
    };
    if ($@ || !$res) {
        $log->die("Sending data failed: " . str($@));
    } else {
        $last_syncid = $chunk[-1]->{syncId};
        $prop_last_syncid->set($last_syncid);
        $log->out("Successfully sent: updated last_syncid: $last_syncid");
    }
}

juggler_ok();

local $Yandex::Advmon::GRAPHITE_PREFIX = sub {[qw/direct_one_min db_configurations/, $Settings::CONFIGURATION]};
monitor_values({
    flow => {ExportDomainsToBalance => {last_syncid => $last_syncid}},
});

$log->out('FINISH');
