#!/usr/bin/perl

use my_inc '..';
use Direct::Modern;
use ScriptHelper 'Yandex::Log' => 'messages', sharded => 1, get_file_lock => undef;
use Settings;

use Encode;
use JSON;
use Time::HiRes qw//;
use Hash::Util qw/lock_hash/;
use List::Util qw/max/;
use List::MoreUtils qw/any none uniq/;

use LockTools qw(get_file_lock release_file_lock);
use Tools qw/encode_json_and_compress/;
use Direct::Encrypt qw/decrypt_text/;
use Direct::Validation::Feeds qw/is_valid_feed_type/;
use Direct::Feeds;


use Yandex::DBTools;
use Yandex::HashUtils qw/hash_cut hash_merge/;
use Yandex::HTTP qw/http_parallel_request/;
use Yandex::ListUtils qw/chunks xsort/;
use Yandex::ProcInfo qw/proc_memory/;
use Yandex::Trace;

=head1 DESCRIPTION

  Скрипт регулярно выбирает из feeds строки с update_status IN (New,Outdated,Updating), меняет статус на Updating и
  переобходит ручкой yml2directinf из BannerLand [Боря добавил туда ошибки/ворнинги, количество
  товаров, формирование справочников по указанному списку полей и количество товаров, описание в комментариях к
  DIRECT-43907 и:

  * при ошибках запроса BL (сетевых или парсинга ответа) - увеличиваем у фида fetch_errors_count на 1,
    если оно становится равно или превышает константу из get_bl_max_errors_count() - переводим update_status на Error
  * при ошибках BL (ErrNNNN) - fetch_errors_count тоже увеличиваем на 1 (по смыслу так делать не нужно, но сейчас на этот счетчик завязано отображение ошибок фида),
    сразу ставим update_status = Error
  * ВАЖНО, когда для ошибок BL вернут >1 числа попыток (возможно с заведением отдельной колонки-счетчика в БД)
    нужно не забыть вернуть логику перевода фидов по файлу (а не по урлу) в статус Error без учета числа попыток.

  * сохраняет результат парсинга для будущей истории в perf_feed_history: offers_count, parse_results_json_compressed
  * обновляет категории в perf_feed_categories. Исчезнувшие из фида категории помечает is_deleted = 1. Новые категории
      добавляются.
  * обновляет топ-1000 [константа] по числу товаров этого фида вендоров в perf_feed_vendors. Исчезнувших совсем или
      вылетвеших из топ'а удаляет.
  * обновляет в feeds: update_status=Done, fetch_errors_count = 0, last_refreshed=NOW(). В
      апдейте обязательно смотрит на то, что update_status по-прежнему Updating.
  TODO * если задан email, то в случае ошибок/ворнингов или невозможности выкачать данные пишет письмо [!!уточнить:
      сейчас уведомление написано только по галочке в кампании, к которой привязан фид!!]
  * Регулярно скидывает:
      update_status с Done на Outdated для тех фидов, где есть refresh_interval > 0 и с last_refreshed уже прошло как минимум refresh_interval секунд
      update_status с Error на Updated для тех фидов, где с LastChange уже прошло как минимум RECHECK_INTERVAL_ERROR_FEEDS дней

  Принимает параметры:
    --shard-id
    --once
    --clientid
    --feed-id
    --bannerland-url — возможность использовать произвольную ссылку как источник данных вместо $Settings::BANNER_LAND_URL.
    --max-feed-size <Mb> - максимальный размер фида в мегабайтах

=head1 METADATA

# большой ulimit (после выкидывания заливки фида в MDS, кажется уже не ?) нужен, т.к. фиды огромные
<crontab>
    time: */2 * * * *
    sharded: 1
    ulimit: -v 20000000
    package: scripts-switchman
    <switchman>
        <leases>
            mem: 16000
        </leases>
        group: scripts-other
    </switchman>
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl: 30m
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

ppcFeedToBannerLand.pl

=cut

=head1 SUBROUTINES/METHODS/VARIABLES

=cut

# ограничение по числу вендоров на фид, в таблице perf_feed_vendors
our $MAX_VENDORS_PER_FEED = 1000;

# ограничение по памяти для скрипта
our $MAX_MEMORY_LIMIT = 8 * 1024 * 1024 * 1024;

# максимальный размер файла в байтах по умолчанию
# используется если параметр не указан в лимитах клиента (client_limits.feed_max_file_size)
# или не передан в скрипту в параметре --max-feed-size
our $DEFAULT_MAX_FEED_SIZE = 512; # МБ

# таймаут в секундах запроса к Боре, большой -- для больших фидов
# может быть увеличен, если в чанке будут фиды размера больше, чем DEFAULT_MAX_FEED_SIZE
our $DEFAULT_BL_REQUEST_TIMEOUT = 30 * 60;

# время кеширования параметров работы, секунд
our $BL_PROPERTY_CACHE_TIME = 600;

# Пропертя send_new_feeds_in_bl_from_jobs
my $SEND_NEW_FEEDS_IN_BL_FROM_JOBS_PROP = Property->new('send_new_feeds_in_bl_from_jobs');

use constant {
    BL_ERROR_FETCH_FAILED => 1201,
    BL_ERROR_YML_NO_MODEL_OR_TAG_NAME => 1203,
    BL_ERROR_XML_FATAL => 1204,
    BL_ERROR_XML_NO_OFFERS => 1205,
    BL_ERROR_XML_NO_CATEGORIES => 1206,
    BL_ERROR_YML_NO_URL_TAG => 1207,
    BL_ERROR_EMPTY_FILE => 1211,
    BL_ERROR_FEED_TYPE_MISMATCH => 1212,
    BL_ERROR_FEED_DATA_TYPE_MISMATCH => 1213,
    BL_ERROR_BUSINESS_TYPE_MISMATCH => 1220,
    BL_ERROR_FILE_TOO_BIG => 1266,
    BL_ERROR_CATEGORY_NOT_POSITIVE_INTEGER => 1290,
    BL_ERROR_FATAL => 1812,
    YNDX_SMART_RECOMMEND_AGENCY_CLIENT_ID => 74171260, # ClientID агентства yndx-smart-recommend-agency (DIRECT-123740)
};

# ошибки которые мы скрываем от клиента (проблема на стороне BannerLand)
# после этих ошибок не добавляется запись об ошибке в таблицу perf_feed_history
# дальнейшая обработка фида должна быть прекращена
my @fatal_bl_errors = (
    BL_ERROR_FATAL,
);

# ошибки для клиента (может исправить их изменив фид)
# после этих ошибок в таблицу perf_feed_history добавляется запись об ошибке
# дальнейшая обработка фида должна быть прекращена
my @client_bl_errors = (
    BL_ERROR_FETCH_FAILED,
    BL_ERROR_EMPTY_FILE,
    BL_ERROR_FILE_TOO_BIG,
    BL_ERROR_YML_NO_MODEL_OR_TAG_NAME,
    BL_ERROR_XML_FATAL,
    BL_ERROR_XML_NO_OFFERS,
    BL_ERROR_XML_NO_CATEGORIES,
    BL_ERROR_YML_NO_URL_TAG,
    BL_ERROR_FEED_DATA_TYPE_MISMATCH,
    BL_ERROR_FEED_TYPE_MISMATCH,
    BL_ERROR_BUSINESS_TYPE_MISMATCH,
    BL_ERROR_CATEGORY_NOT_POSITIVE_INTEGER,
);

my %error_feed_update = (
    update_status => 'Error',
    LastChange => 'NOW()',
    offers_count => 0,
);
lock_hash(%error_feed_update);


my ($ONCE, @CLIENTIDS, @FEEDIDS, $FAKE_BANNER_LAND_URL, $MAX_FEED_SIZE);
extract_script_params(
    'once' => \$ONCE,
    'clientid=i@' => \@CLIENTIDS,
    'feed-id=i@' => \@FEEDIDS,
    'bannerland-url=s' => \$FAKE_BANNER_LAND_URL,
    'max-feed-size=i' => \$MAX_FEED_SIZE,
);

if (!@CLIENTIDS && !@FEEDIDS) {
    my $script_name = get_script_name();
    get_file_lock(undef, $script_name);
}

# максимальный размер фида в байтах
my $max_feed_file_size = ($MAX_FEED_SIZE || $DEFAULT_MAX_FEED_SIZE) * 1024 * 1024;
my $bannerland_url = $FAKE_BANNER_LAND_URL || $Settings::BANNER_LAND_URL;
my $json = JSON->new();

my $iterations_cnt = 0;

sub _log {
    my ($data, $suffix) = @_;
    my %predefined_data = (
        shard => $SHARD,
    );
    if (ref $data ne 'ARRAY'){
        $data = [$data];
    }
    $log->bulk_out($suffix => [map {hash_merge({}, $_, \%predefined_data)} @$data]);
};

sub _log_status_change {
    my ($status, $feed_id) = @_;
    _log({
        action    => "Feed status change",
        status    => $status,
        feedId    => $feed_id,
        timestamp => int(Time::HiRes::time()),
    });
};

sub _log_transport {
    return _log(shift, 'transport');
}

_log({
        action => "START"
    });

if (@CLIENTIDS) {
    _log({
            message => "Working only for ClientIDs",
            ClientID => \@CLIENTIDS,
        });
}
if (@FEEDIDS) {
    _log({
            message => "Working only for feed ids",
            feed_id => \@FEEDIDS,
        });
}

while (1) {
    _log({
            action => "Start iteration",
        });

    restart_tracing();

    if (my $reason = smart_check_stop_file()) {
        _log({
                action => "Exiting",
                reason => $reason,
            });
        exit 0;
    }

    my $start_time = Time::HiRes::time();

    _log({
            action => "Marking content as outdated",
        });
    update_last_refreshed(\@CLIENTIDS, \@FEEDIDS);

    _log({
            action => "Marking error feeds as updating",
        });
    update_error_feeds(\@CLIENTIDS, \@FEEDIDS);

    _log({
            action => "Fetching feeds to refresh",
        });

    my $BL_CHUNK_SIZE = Direct::Feeds::get_bl_chunk_size($BL_PROPERTY_CACHE_TIME);
    my $SELECT_CHUNK_SIZE = Direct::Feeds::get_bl_select_chunk_size($BL_PROPERTY_CACHE_TIME);
    _log({
            action => "Working with BL_CHUNK_SIZE = $BL_CHUNK_SIZE and SELECT_CHUNK_SIZE = $SELECT_CHUNK_SIZE",
        });
    my $send_new_feeds_in_bl_from_jobs = $SEND_NEW_FEEDS_IN_BL_FROM_JOBS_PROP->get(300) // 0;

    my $select_text = 'select
            f.feed_id, f.ClientID, f.url, f.fetch_errors_count,
            f.business_type, f.source, f.login, f.encrypted_password
          , cll.feed_max_file_size
          , f.feed_type
          , f.update_status
        from feeds f
        left join client_limits cll on f.ClientID = cll.ClientID';

    my @feed_statuses_to_send = qw/Outdated Updating/;

    if ( not $send_new_feeds_in_bl_from_jobs ) {
        push(@feed_statuses_to_send, 'New');
    }

    my $feeds_to_update = get_all_sql(PPC(shard => $SHARD), [ $select_text,
        where => {
            'f.update_status' => \@feed_statuses_to_send,
            (@CLIENTIDS) ? ('f.ClientID' => \@CLIENTIDS) : (),
            (@FEEDIDS) ? ('f.feed_id' => \@FEEDIDS) : (),
            '_NOT' => {
                'f.source' => 'site',
                'f.statusMbiSynced' => 'No',
            },
        },
        "order by update_status, last_refreshed",
        "limit $SELECT_CHUNK_SIZE"
    ]);

    _log({
            message => (scalar @$feeds_to_update)." feeds to update",
        });

    my $all_feeds_errors = {};
    for my $db_chunk (chunks($feeds_to_update, 1000)) {
        my @ids = map { $_->{feed_id} } @$db_chunk;
        for my $feed_id (@ids) {
            _log_status_change("Updating", $feed_id);
        }
        do_update_table(PPC(shard => $SHARD), 'feeds', { update_status => 'Updating' }, where => { feed_id => \@ids });

        # Фиды "Товарных рекомендаций" не отправляем в BL на валидацию. Проставляем им хардкоженные значния. DIRECT-123740
        my $recommendation_feed_ids = get_one_column_sql(PPC(shard => $SHARD), [
            'SELECT feed_id
             FROM feeds f
             JOIN clients cl ON cl.ClientID = f.ClientID',
             WHERE => {
                 'f.feed_id' => \@ids,
                 'cl.agency_client_id' => YNDX_SMART_RECOMMEND_AGENCY_CLIENT_ID
             }
        ]);
        if (scalar(@$recommendation_feed_ids)) {
            _log({
                action => "Autovalidate smart-recomendations feeds with ids",
                feed_id => $recommendation_feed_ids,
            });
            do_update_table(PPC(shard => $SHARD), 'feeds', {
                update_status => 'Done',
                feed_type => 'YandexMarket',
                offer_examples => undef,
                offers_count => 1,
                fetch_errors_count => 0,
                last_refreshed__dont_quote => 'NOW()',
                LastChange__dont_quote => 'NOW()',
                target_domain => undef,
            },
                where => {
                    feed_id => $recommendation_feed_ids,
                    update_status => 'Updating',
                }
            );
        }
        # Исключаем эти фиды из дальнейшей обработки
        my %in_recom_feed_ids = map {$_ => 1} @$recommendation_feed_ids;
        @ids = grep {not $in_recom_feed_ids{$_}} @ids;
        my @chunk = grep {not $in_recom_feed_ids{$_->{feed_id}}} @$db_chunk;

        # Определяем, нужно ли "заморозить" тип фида.
        # Тип фида замораживается, если фид используется в какой-то кампании.
        # Смена типа может быть опасна тем, что условия фильтров на старый фид не будут совместимы
        # с условиями фильтров на новый https://st.yandex-team.ru/DIRECT-76588
        my $need_to_freeze_feed_type = get_hash_sql(PPC(shard => $SHARD), [
            'SELECT feed_id, count(*) FROM (
                    SELECT ap.feed_id
                    FROM
                        adgroups_performance ap
                        JOIN phrases p ON p.pid = ap.pid
                        JOIN campaigns c ON c.cid = p.cid',
                    WHERE => {
                        'ap.feed_id' => \@ids,
                        'c.statusEmpty__ne' => 'Yes',
                    }, '
                    UNION
                    SELECT ad.feed_id
                    FROM
                        adgroups_dynamic ad
                        JOIN phrases p ON p.pid = ad.pid
                        JOIN campaigns c ON c.cid = p.cid',
                    WHERE => {
                        'ad.feed_id' => \@ids,
                        'c.statusEmpty__ne' => 'Yes',
                    }, '
                    UNION
                    SELECT ff.feed_id
                    FROM
                        filtered_feeds ff
                        JOIN adgroups_text at ON ff.filtered_feed_id = at.filtered_feed_id
                        JOIN phrases p ON p.pid = at.pid
                        JOIN campaigns c ON c.cid = p.cid',
                    WHERE => {
                        'ff.feed_id' => \@ids,
                        'c.statusEmpty__ne' => 'Yes',
                    }, '
                    UNION
                    SELECT at.feed_id
                    FROM
                        adgroups_text at
                        JOIN phrases p ON p.pid = at.pid
                        JOIN campaigns c ON c.cid = p.cid',
                    WHERE => {
                        'at.feed_id' => \@ids,
                        'c.statusEmpty__ne' => 'Yes',
                    }, '
                ) s
            GROUP BY feed_id'
        ]);

        my %req;
        my $max_allowed_feed_file_size_in_chunk;
        for my $row (@chunk) {
            my $allowed_feed_file_size = ($row->{feed_max_file_size} || $max_feed_file_size);
            my $url = Yandex::HTTP::make_url($bannerland_url, {
                cmd => 'yml2directinf',
                url => $row->{url},
                business_type => $row->{business_type},
                ($need_to_freeze_feed_type->{$row->{feed_id}} ? (last_valid_feed_type => $row->{feed_type}) : ()),
                max_file_size => $allowed_feed_file_size,
                max_file_size_type => 'bytes',
                status => $row->{update_status},
                feed_id => $row->{feed_id},
                with_previews => 1,
                ($row->{login} ? ( login => $row->{login} ) : () ),
                ($row->{encrypted_password} ? ( pass => decrypt_text($row->{encrypted_password}) ) : () ),
            });
            _log_transport({
                    feed_id => $row->{feed_id},
                    ClientID => $row->{ClientID},
                    request => $url,
                    stage => "request",
                });
            $req{$row->{feed_id}} = {
                url => $url,
            };
            $max_allowed_feed_file_size_in_chunk = max($max_allowed_feed_file_size_in_chunk, $allowed_feed_file_size);
        }

        _log({
                action => "Requesting BannerLand",
            });
        my $profile = Yandex::Trace::new_profile('bmapi:yml2directinf', obj_num => scalar keys %req);

        my $res = http_parallel_request(GET => \%req,
            max_req => $BL_CHUNK_SIZE,
            timeout => int($DEFAULT_BL_REQUEST_TIMEOUT * ($max_allowed_feed_file_size_in_chunk / ($DEFAULT_MAX_FEED_SIZE * 1024 * 1024))),
            content_ref => 1,
            handle_params => { keepalive => 1 },
            persistent => 0, # DIRECT-63220
            ipv6_prefer => 1,
        );
        undef $profile;

        _log({
                action => "Processing results",
            });

        for my $row (@chunk) {
            my $page = delete $res->{$row->{feed_id}};
            unless ($page) {
                my $msg = 'DIED! no data for feed';
                _log_transport({
                        message => $msg,
                        feed_id => $row->{feed_id},
                        ClientID => $row->{ClientID},
                        stage => "response",
                    });
                die $msg;
            }
            unless ($page->{is_success}) {
                _log_transport({
                        feed_id => $row->{feed_id},
                        ClientID => $row->{ClientID},
                        stage => "response",
                        transport_message => "request to BL failed",
                        url => $page->{headers}->{URL},
                        response_status => $page->{headers}->{Status},
                        response_reason => $page->{headers}->{Reason},
                    });
                my $feed_errors = update_errors_count($row);
                hash_merge $all_feeds_errors, $feed_errors;
                next;
            }
            my $content_ref = $page->{content_ref};
            my $bl_data = eval { $json->decode($$content_ref) };
            if ($@) {
                _log_transport({
                        feed_id => $row->{feed_id},
                        ClientID => $row->{ClientID},
                        stage => "response",
                        transport_message => "invalid json from BL",
                        url => $page->{headers}->{URL},
                        response_part => substr($$content_ref,0,1024),
                        parsing_error => $@,
                    });
                my $feed_errors = update_errors_count($row);
                hash_merge $all_feeds_errors, $feed_errors;
                next;
            }
            undef $content_ref;
            undef $page;
            my $target_domain = Direct::Feeds::extract_domain($bl_data);

            my %client_bl_errors;
            my %fatal_feed_errors;

            # проверяем фид на наличие ошибок с неизвестными кодами
            my @all_known_bl_errors = (@fatal_bl_errors, @client_bl_errors);
            my %unknown_feed_errors = %{ check_unknown_feed_errors($bl_data, $row, \@all_known_bl_errors) };
            if (!%unknown_feed_errors) {
                # проверяем фид на фатальные ошибки
                %fatal_feed_errors = %{ process_feed_errors($bl_data, $row, \@fatal_bl_errors) };
                if (!%fatal_feed_errors) {
                    # проверяем фид на клиентские ошибки
                    %client_bl_errors = %{ process_feed_errors($bl_data, $row, \@client_bl_errors) };
                    # записываем ответ (возможно с клиентскими ошибками) в perf_feed_history
                    my $to_insert = {
                        feed_id => $row->{feed_id},
                        offers_count => $bl_data->{all_elements_amount},
                    };
                    if (@{$bl_data->{errors}} || @{$bl_data->{warnings}}) {
                        $to_insert->{parse_results_json_compressed} = encode_json_and_compress(hash_cut $bl_data, qw/errors warnings/);
                    }
                    do_insert_into_table(PPC(shard => $SHARD), 'perf_feed_history', $to_insert);
                }
            }

            if (%unknown_feed_errors || %fatal_feed_errors || %client_bl_errors) {
                # прекращаем обработку фида если были ошибки
                hash_merge $all_feeds_errors, \%unknown_feed_errors, \%fatal_feed_errors, \%client_bl_errors;
                next;
            }

            if (!is_valid_feed_type($bl_data->{feed_type})){
                # прекращаем обработку фида если получили неизвестный тип
                _log_transport({
                    feed_id => $row->{feed_id},
                    ClientID => $row->{ClientID},
                    stage => "response",
                    transport_message => "validation error",
                    error => "unknown feed_type",
                    feed_type => $bl_data->{feed_type},
                });
                my $feed_type_error = mark_feed_as_failed($row);
                hash_merge $all_feeds_errors, $feed_type_error;
                next;
            }

            my $old_feed_type = $row->{feed_type};
            my $new_feed_type = $bl_data->{feed_type};
            my $is_feed_type_cannot_be_updated = $row->{source} !~ /^(url|site)$/ || $new_feed_type ne 'YandexMarket' || $old_feed_type ne undef;

            if ($need_to_freeze_feed_type->{$row->{feed_id}} && $old_feed_type ne $new_feed_type && $is_feed_type_cannot_be_updated) {
                # прекращаем обработку фида если его тип поменялся, кроме случая при изменении с null на YandexMarket
                _log_transport({
                    feed_id => $row->{feed_id},
                    ClientID => $row->{ClientID},
                    stage => "response",
                    transport_message => "validation error",
                    error => "attempted to change feed_type",
                    old_feed_type => $old_feed_type,
                    new_feed_type => $new_feed_type,
                });
                my $feed_type_error = mark_feed_as_failed($row);
                hash_merge $all_feeds_errors, $feed_type_error;
                next;
            }

            my $is_categs_changed = fix_categs_struct_inplace($bl_data->{categs});
            if ($is_categs_changed) {
                _log({
                    message => "fixed incorrect categories struct",
                    feed_id => $row->{feed_id},
                    ClientID => $row->{ClientID},
                });
            }
            # perf_feed_categories
            my @feed_cat_ids = map { $_->{id} } @{$bl_data->{categs}};
            # помечаем удаленными более не существующие категории
            do_update_table(PPC(shard => $SHARD), 'perf_feed_categories', { is_deleted => 1 }, where => {
                feed_id => $row->{feed_id},
                category_id__not_in => \@feed_cat_ids,
            });
            do_mass_insert_sql(PPC(shard => $SHARD),
                'insert into perf_feed_categories
                    (feed_id, category_id, parent_category_id, name, offers_count)
                    values %s
                    on duplicate key update
                        name = values(name),
                        parent_category_id = values(parent_category_id),
                        offers_count = values(offers_count),
                        is_deleted = 0',
                [ map { [$row->{feed_id}, $_->{id}, $_->{parentId} // 0, $_->{category}, $bl_data->{categoryId}->{$_->{id}}//0 ] }
                    grep { defined $_->{id} } @{$bl_data->{categs}} ]
            );

            # perf_feed_vendors
            # сортируем (desc) по количеству предложений для данного вендора
            my @vendors =
                xsort { \$_->{count} }
                map { { name => $_, count => $bl_data->{vendor}->{$_}  } }
                grep { length }
                keys %{$bl_data->{vendor}};

            if (@vendors > $MAX_VENDORS_PER_FEED) {
                # оставляем только топ-1000
                @vendors = @vendors[0 .. $MAX_VENDORS_PER_FEED-1];
            }
            do_in_transaction {
                do_delete_from_table(PPC(shard => $SHARD), 'perf_feed_vendors', where => {
                    feed_id => $row->{feed_id},
                    name__not_in => [ map { $_->{name} } @vendors ],
                });

                do_mass_insert_sql(PPC(shard => $SHARD), 'insert ignore into perf_feed_vendors (feed_id, name) values %s',
                    [ map { [ $row->{feed_id}, $_->{name} ] } @vendors ]
                );
            };

            my $offer_examples_json;
            if ($bl_data->{offer_examples}) {
                if (ref $bl_data->{offer_examples} eq 'HASH') {
                    $offer_examples_json = $json->encode($bl_data->{offer_examples});
                    if (length(Encode::encode('UTF-8', $offer_examples_json)) >= 2**24) {
                        undef $offer_examples_json;
                        _log({
                            message => "Offer examples is too long (more that 2^24 bytes)",
                            feed_id => $row->{feed_id},
                            ClientID => $row->{ClientID},
                        });
                    }
                } else {
                    _log({
                            message => "Got bad JSON format in offer_examples",
                            feed_id => $row->{feed_id},
                            ClientID => $row->{ClientID},
                            offer_examples => $bl_data->{offer_examples},
                        });
                }
            }

            do_update_table(PPC(shard => $SHARD), 'feeds', {
                    update_status => 'Done',
                    feed_type => $bl_data->{feed_type},
                    offer_examples => $offer_examples_json,
                    offers_count => $bl_data->{all_elements_amount},
                    fetch_errors_count => 0,
                    last_refreshed__dont_quote => 'NOW()',
                    LastChange__dont_quote => 'NOW()',
                    target_domain => $target_domain,
                },
                where => {
                    feed_id => $row->{feed_id},
                    update_status => 'Updating',
                }
            );
            _log_status_change("Done", $row->{feed_id});
        }
    }

    if (%$all_feeds_errors) {
        do_mass_update_sql(PPC(shard => $SHARD), 'feeds',
            feed_id => $all_feeds_errors,
            byfield_options => {
                fetch_errors_count => { dont_quote_value => 1 },
                LastChange => { dont_quote_value => 1 },
            },
        );
        for my $feed_id (keys %$all_feeds_errors) {
            _log_status_change("Error", $feed_id);
        }
    }

    if ($ONCE) {
        _log({
                message => "exit after first iteration",
            });
        last;
    }

    juggler_ok();

    $iterations_cnt++;
    if (proc_memory() > $MAX_MEMORY_LIMIT) {
        _log({
                message => "memory limit exceeded",
                proc_memory => proc_memory(),
                iterations_count => $iterations_cnt,

            });
        last;
    }

    my $iter_time = Time::HiRes::time() - $start_time;

    my $sleep_time = max(0, Direct::Feeds::get_max_sleep_time_seconds($BL_PROPERTY_CACHE_TIME) - $iter_time);
    _log({
            action => "sleep",
            sleep_time => $sleep_time,
            iter_time => $iter_time,
        });
    Time::HiRes::sleep($sleep_time);

    _log({
            action => "Finish iteration",
        });
}

if (!@CLIENTIDS && !@FEEDIDS) {
    release_file_lock();
}

_log({
        action => "FINISH",
    });

#######

=head2 fix_categs_struct_inplace

    Исправляем структуру описания категорий
    Иногда попадаются фиды, у которых вместо id написано Id.

    Переделываем in-place в массиве категорий ключи на правильные
    Возвращаем признак (0/1), что меняли ключи

=cut

sub fix_categs_struct_inplace {
    my %lkey2key = (
        'id' => 'id',
        'parentid' => 'parentId',
    );
    my %key2lkey = reverse %lkey2key;

    my ($categs) = @_;

    my $changed = 0;

    for my $categ (@$categs) {
        for my $key (keys %$categ) {
            if (!$key || exists $key2lkey{$key}) {
                next;
            }
            if (my $newkey = $lkey2key{ lc($key) }) {
                $changed ||= 1;
                $categ->{ $newkey } = delete $categ->{$key};
            }
        }
    }

    return $changed;
}

=head2 update_last_refreshed

Функция меняет статус фидов с Done на Outdated, если с момента последнего обновления
прошло более чем refresh_interval секунд

=cut

sub update_last_refreshed
{
    my ($only_clientids, $only_feedids) = @_;

    my @additional_cond;
    if ($only_clientids && @$only_clientids) {
        @additional_cond = ('AND', {ClientID => $only_clientids});
    }
    if ($only_feedids && @$only_feedids) {
        @additional_cond = ('AND', {feed_id => $only_feedids});
    }

    my $feeds_to_update = get_all_sql(PPC(shard => $SHARD), ["
        select feed_id
        from feeds
        where
            last_refreshed > 0
            and refresh_interval > 0
            and source = 'url'
            and last_refreshed < now() - interval refresh_interval second
            and update_status = 'Done'
    ", @additional_cond]);

    my @feed_ids_to_update = uniq map { $_->{feed_id} } @$feeds_to_update;
    my $need_to_update_last_refreshed = get_all_sql(PPC(shard => $SHARD), [
        'SELECT feed_id FROM (
                    SELECT ap.feed_id
                        FROM
                            adgroups_performance ap
                            JOIN phrases p ON p.pid = ap.pid
                            JOIN campaigns c ON c.cid = p.cid
                            JOIN camp_options co ON co.cid = c.cid',
                        WHERE => {
                            'ap.feed_id'        => \@feed_ids_to_update,
                            'c.statusEmpty__ne' => 'Yes',
                            'c.archived'        => 'No',
                            '_OR'               => {
                                'co.stopTime__ge__dont_quote' => 'DATE_SUB(NOW(), INTERVAL 7 DAY)',
                                'c.statusShow'                => 'Yes'
                            },
                        }, '
                    UNION SELECT ad.feed_id
                        FROM
                            adgroups_dynamic ad
                            JOIN phrases p ON p.pid = ad.pid
                            JOIN campaigns c ON c.cid = p.cid
                            JOIN camp_options co ON co.cid = c.cid',
                        WHERE => {
                            'ad.feed_id'        => \@feed_ids_to_update,
                            'c.statusEmpty__ne' => 'Yes',
                            'c.archived'        => 'No',
                            '_OR'               => {
                                'co.stopTime__ge__dont_quote' => 'DATE_SUB(NOW(), INTERVAL 7 DAY)',
                                'c.statusShow'                => 'Yes'
                            },
                        }, '
                    UNION SELECT ff.feed_id
                        FROM
                            filtered_feeds ff
                            JOIN adgroups_text at ON ff.filtered_feed_id = at.filtered_feed_id
                            JOIN phrases p ON p.pid = at.pid
                            JOIN campaigns c ON c.cid = p.cid
                            JOIN camp_options co ON co.cid = c.cid',
                        WHERE => {
                            'ff.feed_id'        => \@feed_ids_to_update,
                            'c.statusEmpty__ne' => 'Yes',
                            'c.archived'        => 'No',
                            '_OR'               => {
                                'co.stopTime__ge__dont_quote' => 'DATE_SUB(NOW(), INTERVAL 7 DAY)',
                                'c.statusShow'                => 'Yes'
                            },
                        }, '
                    UNION SELECT at.feed_id
                        FROM
                            adgroups_text at
                            JOIN phrases p ON p.pid = at.pid
                            JOIN campaigns c ON c.cid = p.cid
                            JOIN camp_options co ON co.cid = c.cid',
                        WHERE => {
                            'at.feed_id'        => \@feed_ids_to_update,
                            'c.statusEmpty__ne' => 'Yes',
                            'c.archived'        => 'No',
                            '_OR'               => {
                                'co.stopTime__ge__dont_quote' => 'DATE_SUB(NOW(), INTERVAL 7 DAY)',
                                'c.statusShow'                => 'Yes'
                            },
                        }, '
                ) s
            GROUP BY feed_id'
    ]);
    @feed_ids_to_update = map { $_->{feed_id} } @$need_to_update_last_refreshed;
    my $result = do_update_table(PPC(shard => $SHARD),
        'feeds',
        {update_status => 'Outdated'},
        where => {feed_id => \@feed_ids_to_update}
    );
    for my $feed_id (@feed_ids_to_update) {
        _log_status_change("Outdated", $feed_id);
    }
    _log({
            action => "feeds set as Outdated",
            feeds_count => $result,
        });
}

=head2 update_error_feeds

Функция меняет статус фидов с Error на Updating, если с момента последнего обновления
прошло более чем RECHECK_INTERVAL_ERROR_FEEDS дней

=cut

sub update_error_feeds
{
    my ($only_clientids, $only_feedids) = @_;

    my $RECHECK_INTERVAL_ERROR_FEEDS = Direct::Feeds::get_bl_recheck_interval_error($BL_PROPERTY_CACHE_TIME);
    my @additional_cond;
    if ($only_clientids && @$only_clientids) {
        @additional_cond = ('AND', {ClientID => $only_clientids});
    }
    if ($only_feedids && @$only_feedids) {
        @additional_cond = ('AND', {feed_id => $only_feedids});
    }

    my $count = int do_sql(PPC(shard => $SHARD), ["
        update feeds
        set update_status = 'Updating', fetch_errors_count = 0
        where
            LastChange < now() - interval $RECHECK_INTERVAL_ERROR_FEEDS day
            and update_status = 'Error'
    ", @additional_cond]);
    _log({
            action => "feeds set as Updating",
            feeds_count => $count,
        });
}

=head2 check_unknown_feed_errors

Найти неизвестные ошибки от BL, т. е. те, которых нет в указанном списке $errors_list

Возвращает ссылку на хеш
    $feed_id => {
        update_status => 'Error',
        fetch_errors_count => fetch_errors_count + 1
    }

Если хеш непустой, то дальнейшая обработка фида должна быть прекращена

=cut

sub check_unknown_feed_errors
{
    my ($bl_data, $row, $errors_list) = @_;
    my $errors_update = {};

    if ($bl_data->{errors} && @{$bl_data->{errors}}) {
        _log_transport([map{{
                feed_id => $row->{feed_id},
                ClientID => $row->{ClientID},
                stage => "response",
                transport_message => "fetch error",
                error => $_,
            }} @{$bl_data->{errors}}]);
        for my $err (@{$bl_data->{errors}}) {
            if (none { $err->{code} == $_ } @$errors_list) {
                $errors_update = mark_feed_as_failed($row);
                last;
            }
        }
    }
    return $errors_update;
}

=head2 process_feed_errors

Обработать ошибки от BL

Возвращает ссылку на хеш
    $feed_id => {
        update_status => 'Error',
        fetch_errors_count => fetch_errors_count + 1
    }

Если хеш непустой, то дальнейшая обработка фида должна быть прекращена

=cut

sub process_feed_errors
{
    my ($bl_data, $row, $errors_list) = @_;
    my $errors_update = {};

    if ($bl_data->{errors} && @{$bl_data->{errors}}) {
        _log_transport([map{{
                feed_id => $row->{feed_id},
                ClientID => $row->{ClientID},
                stage => "response",
                transport_message => "fetch error",
                error => $_,
            }} @{$bl_data->{errors}}]);
        for my $err (@{$bl_data->{errors}}) {
            if (any { $err->{code} == $_ } @$errors_list) {
                $errors_update = mark_feed_as_failed($row);
                last;
            }
        }
    }
    return $errors_update;
}

=head2 update_errors_count

    Увеличивает счетчик ошибок фида до достижения им $MAX_ERRORS_COUNT,
    после чего фид переходит в статус Error

=cut

sub update_errors_count
{
    my $row = shift;

    my $MAX_ERRORS_COUNT = Direct::Feeds::get_bl_max_errors_count($BL_PROPERTY_CACHE_TIME);
    my %update = (
        fetch_errors_count => 'fetch_errors_count + 1',
    );
    if ($row->{fetch_errors_count} >= $MAX_ERRORS_COUNT) {
        hash_merge \%update, \%error_feed_update;
    }

    return return { $row->{feed_id} => \%update };
}

=head2 mark_feed_as_failed

    Сразу выставить фиду статус Error.
    Также увеличивает fetch_errors_count на 1

=cut

sub mark_feed_as_failed {
    my $row = shift;
    return { $row->{feed_id} => { %error_feed_update , fetch_errors_count => 'fetch_errors_count + 1' } };
}
