#!/usr/bin/perl

use my_inc "..";


=head1 METADATA

<crontab>
    time: 47 2 * * *
    <switchman>
        group: scripts-other
        <leases>
            mem: 110
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl: 2d3h
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

    ppcFetchAgencyCountryCurrency.pl

=head1 DESCRIPTION

    Обновляет списки доступных уешным агентствам стран-валют из Баланса в таблице client_firm_country_currency.
    В дальнейшем такие агентства смогут создавать субклиентов в доступных им валютах.

=head1 RUNNING

    Можно запускать по конкретным агентствам:
        --login — логин одного из его представителей агентства, по которому надо получить данные
                  если не указан, получает по всем уешным

    ./protected/ppcFetchAgencyCountryCurrency.pl
    ./protected/ppcFetchAgencyCountryCurrency.pl --login itagency

=cut

use Direct::Modern;

use Settings;
use ScriptHelper;
use Primitives;
use RBAC2::Extended;
use RBACDirect;
use Client::CurrencyTeaserData;

use Yandex::DBTools;
use Yandex::DBShards;

use List::MoreUtils qw(all);

my $OPERATOR_UID = 1;
my $UIDS_CHUNK_SIZE = 100_000;

my $logins = [];

extract_script_params(
    'login=s' => $logins,
);

$log->out('START');

my $all_ag_uids;
if (!$logins || !@$logins) {
    $log->out('Connecting RBAC');

    my $rbac = eval { RBAC2::Extended->get_singleton($OPERATOR_UID) }
        or $log->die("Can't connect to RBAC: $@");

    $log->out('Fetching agencies to refresh');
    $all_ag_uids = rbac_get_all_agencies($rbac);
    $log->out('Got ' . scalar(@$all_ag_uids) . ' agency UIDs');
} else {
    $log->out('Using logins from commandline: ' . join(', ', @$logins));
    $all_ag_uids = [ map { get_uid_by_login2($_) } @$logins ];
}

$log->out('Starting per-shard processing');
my $result = foreach_shard_parallel(uid => $all_ag_uids, chunk_size => $UIDS_CHUNK_SIZE, sub {
    my ($shard, $uids) = @_;
    # Ловим ошибки из foreach_shard_parallel
    local $SIG{__DIE__} = sub { local $log->{msg_prefix} = undef; $log->die({ message => 'DIED!', error => $_[0], shard => $shard }); };

    $log->msg_prefix("[shard $shard]");

    $log->out('Fetching ClientIDs for non-multicurrency clients with uids:', $uids);
    my $clientids = get_one_column_sql(PPC(shard => $shard), [q(
        SELECT DISTINCT u.ClientID
        FROM users u
        LEFT JOIN clients cl ON u.ClientID = cl.ClientID
        WHERE
            IFNULL(cl.work_currency, "YND_FIXED") = "YND_FIXED"
        ),  'AND',  {
            'u.uid' => $uids,
    }]) || [];
    $log->out('Got ' . scalar(@$clientids) . ' clients to fetch data for');

    for my $client_id (@$clientids) {
        $log->out("Refreshing data for ClientID $client_id");
        my $success = eval {
            Client::CurrencyTeaserData::fetch_client_multicurrency_teaser_data($client_id, undef, balance_timeout => 320);
            return 1;
        };
        if (!$success) {
            $log->warn("Error refreshing data for ClientID $client_id, skipping:", $@);
        }
    }
});

my @error_in_shards = grep { ! all { $_ } @{$result->{$_}} } keys %$result;
if (@error_in_shards) {
    $log->die("Error in shards: " . join(', ', @error_in_shards), 'Result was: ', $result);
} else {
    $log->out('Done refreshing in all shards');
}

$log->out('Writing monitoring data');
juggler_ok();

$log->out('FINISH');
