#!/usr/bin/perl

use my_inc "..";


=head1 METADATA

<crontab>
    time: */30 * * * *
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 200
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>

<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded:        1
    ttl:            12h
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

    ppcFetchClientMulticurrencyTeaserData.pl

=head1 DESCRIPTION

    Получает от Баланса список стран и валют, которые могут быть у клиентов после перехода на реальную валюту.
    Полученный данные затем используются для показа тизера и на странице перехода, чтобы не ходить синхронно 
    в Баланс. Формирование этих данных может занимать десятки секунд по старым клиентам.

=head1 RUNNING

    Скрипт каждую ночь обновляет из Баланса список стран и валют, которые могут быть у клиентов после перехода на реальную валюту.
    Клиенты должны попадать под условия показа тизера.
    Данные обновляются в том числе и по тем клиентам, для которых данные уже есть, но которые ещё не перевелись.

    Можно запускать по конкретным клиентам:
        --client, --clientid, -c, --login — ClientID клиента или логин одного из его представителей, по которому надо получить данные
                                             если не указан, получает по всем, подходящим под условие показа тизера
        --shard-id — номер шарда, в котором искать клиентов (в том числе, указанных через --client*/--login)

    ./protected/ppcFetchClientMulticurrencyTeaserData.pl --shard-id=2
    ./protected/ppcFetchClientMulticurrencyTeaserData.pl --login holodilnikru --clientid 12345 --clientid 78910,111213 --shard-id=2

=cut

use warnings;
use strict;

use ScriptHelper
    sharded => 1,
    'Yandex::Log' => 'messages';

use Settings;

use Yandex::DBTools;
use Yandex::ListUtils qw(chunks);
use Yandex::Trace;

use Client::CurrencyTeaserData ();
use Client::ConvertToRealMoney ();
use Client::CurrencyTeaserData ();
use PrimitivesIds qw(get_clientids);
use Primitives ();
use Property ();

use Parallel::ForkManager;

use utf8;
use open ':std' => ':utf8';

=head1 SUBROUTINES/METHODS/VARIABLES

=cut

=head2 $CLIENTS_TO_FETCH_LIMIT

    Количество клиентов, данные которых обновлять за один раз.

=cut

my $CLIENTS_TO_FETCH_LIMIT = 50_000;

=head2 $CLIENTS_PER_FORK

    Количество клиентов, которых обрабатывать в одном отфорканном процессе

=cut

my $CLIENTS_PER_FORK = 250;

my $client_ids = [];
my $logins = [];

extract_script_params(
    'login=s' => $logins,
    'clientid=i' => $client_ids,
);

$log->out('START');

if ((!$client_ids || !@$client_ids) && (!$logins || !@$logins)) {
    $log->out("Fetching $CLIENTS_TO_FETCH_LIMIT existing clients to update");
    my $existing_client_ids = get_one_column_sql(PPC(shard => $SHARD), qq{
        SELECT fmt.ClientID
        FROM clients_to_force_multicurrency_teaser fmt
        LEFT JOIN clients cl ON fmt.ClientID = cl.ClientID
        LEFT JOIN client_teaser_data_lastupdate ctdl ON ctdl.ClientID = fmt.ClientID
        WHERE
            IFNULL(cl.work_currency, "YND_FIXED") = "YND_FIXED"
            AND IFNULL(ctdl.last_update, "2000-01-01") < NOW() - INTERVAL $Client::ConvertToRealMoney::COUNTRY_CURRENCY_LAST_UPDATE_INTERVAL_DAYS DAY
        ORDER BY ctdl.last_update
        LIMIT ?
    }, $CLIENTS_TO_FETCH_LIMIT);
    $log->out('Got ' . scalar(@$existing_client_ids) . ' existing clients to update:', $existing_client_ids);
    push @$client_ids, @$existing_client_ids;
} else {
    if (@$logins){
        push @$client_ids, @{get_clientids(login => $logins)};
    }

    $log->out('Using ClientIDs from commandline: ' . join(',', @$client_ids));
}

if (@$client_ids) {
    $log->out('Got ' . scalar(@$client_ids) . ' clients to fetch currencies for');

    my $parallel_level_property = Property->new($Client::CurrencyTeaserData::PARALLEL_LEVEL_PROPERTY_NAME);
    my $parallel_level = $parallel_level_property->get() || $Client::CurrencyTeaserData::DEFAULT_PARALLEL_LEVEL;
    $log->out("Parallel level is $parallel_level");
    my $pm = new Parallel::ForkManager($parallel_level);

    for my $client_ids_chunk (chunks($client_ids, $CLIENTS_PER_FORK)) {
        $log->out('Processing clients chunk in fork:', $client_ids_chunk);
        $pm->start and next;

        Yandex::Trace::restart(\$ScriptHelper::trace, tags => "shard_$SHARD,worker");

        $log->out('Fetching agencies for clients');
        my $clid2agid = Primitives::mass_get_client_first_agency($client_ids_chunk);

        for my $client_id (@$client_ids_chunk) {
            my $agency_id = $clid2agid->{$client_id};
            $log->out("Processing ClientID: $client_id and AgencyID: ". ($agency_id // 'undef'));
            my $success = eval {
                Client::CurrencyTeaserData::fetch_client_multicurrency_teaser_data($client_id, $agency_id, balance_timeout => 320);
                return 1;
            };
            if ($success) {
                $log->out("Successfully refreshed data for client $client_id");
            } else {
                $log->out("Error refreshing data for client $client_id: $@");
            }

            if (my $reason = smart_check_stop_file()) {
                $log->out("$reason. Exitinig!");
                exit(0);
            }
        }

        $pm->finish;
    }

    $pm->wait_all_children;
} else {
    $log->out('No clients found');
}

juggler_ok();

$log->out('FINISH');
