#!/usr/bin/perl

use my_inc "..";


=head1 METADATA

<crontab>
    time: 47 4 * * *
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    ttl:            2d4h
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: 27 6 * * *
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>
<crontab>
    env: SETTINGS_LOCAL_SUFFIX=DevTest
    time: 17 9 * * *
    package: conf-dev
</crontab>
<crontab>
    env: SETTINGS_LOCAL_SUFFIX=Dev7
    time: 27 9 * * *
    package: conf-dev
</crontab>

=cut

=head1 NAME

    ppcFetchCountryCurrencies.pl

=head1 DESCRIPTION

    Получает от Баланса список стран и возможных валют оплаты в этих странах.
    Полученные данные сохраняет в таблицу country_currencies в PPCDICT.

=cut

use warnings;
use strict;
use utf8;

use List::MoreUtils qw/part zip/;
use Readonly;

use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::ScalarUtils;

use BalanceWrapper;
use Currencies ();
use Settings;
use ScriptHelper;

Readonly::Array my @TABLE_FIELDS => qw/is_agency region_id currency firm_id/;

$log->out('START');

$log->out('Fetching Balance data');
my $country_currencies_array_new = BalanceWrapper::get_firm_country_currency();
$log->die('get_firm_country_currency() did not return an array ref') unless $country_currencies_array_new && ref($country_currencies_array_new) eq 'ARRAY';
$log->out('Got ' . scalar(@$country_currencies_array_new) . ' records from Balance');
$country_currencies_array_new = _rename_agency_field($country_currencies_array_new);
my $country_currencies_new = _array2hash($country_currencies_array_new);

$log->out('Fetching current table data');
my $country_currencies_array_old = get_all_sql(PPCDICT, ['SELECT', sql_fields(@TABLE_FIELDS),
                                                         'FROM country_currencies',
                                                         WHERE => {
                                                            # чтобы не удалять валюты, с которыми директ итак не работает
                                                            currency => [Currencies::get_currencies_list()],
                                                         },
                                               ]);
my $country_currencies_old = _array2hash($country_currencies_array_old);

my $country_currencies_diff = hash_diff($country_currencies_old, $country_currencies_new);
my ($added, $deleted) = part {defined $country_currencies_diff->{$_} ? 0 : 1} keys %$country_currencies_diff;

if ($deleted && @$deleted) {
    $log->out('Country/currency pairs to remove: ', $deleted);

    my @pairs;
    for my $entry (@$deleted) {
        my $row = _deserialize_country_currency_key($entry);
        push @pairs, '(' . join(',', map { sql_quote($_) } @{$row}{@TABLE_FIELDS}) . ')';
    }
    my $sql_pairs_cond = join ',', @pairs;
    my $sql_fields = sql_fields(@TABLE_FIELDS);

    my $deleted_rows_cnt = do_delete_from_table(PPCDICT, 'country_currencies', where => {_TEXT => "($sql_fields) IN ($sql_pairs_cond)"});
    $log->out("Deleted $deleted_rows_cnt country/currency pairs");
} else {
    $log->out('No records to remove');
}

if ($added && @$added) {
    $log->out('Country/currency pairs to add: ', $added);

    my @added_values;
    for my $entry (@$added) {
        my $row = _deserialize_country_currency_key($entry);
        push @added_values, [ @{$row}{@TABLE_FIELDS} ];
    }
    my $sql_fields = sql_fields(@TABLE_FIELDS);

    my $inserted_rows_cnt = do_mass_insert_sql(PPCDICT, "INSERT INTO country_currencies ($sql_fields) VALUES %s", \@added_values);
    $log->out("Inserted $inserted_rows_cnt new country/currency pairs");
} else {
    $log->out('No new records to add');
}

juggler_ok();

$log->out('FINISH');

=head2 _rename_agency_field

    Переименовывает в значениях хешей ключ agency -> is_agency

=cut

sub _rename_agency_field {
    my $array = shift;

    for my $entry (@$array) {
        $entry->{is_agency} = delete $entry->{agency};
    }

    return $array;
}

=head2 _serialize_country_currency_key

    Делает из {is_agency => 0, region_id => 123, currency => 'KZT', firm_id => 3} сериализованную версию вида '0:123:KZT:3'

=cut

sub _serialize_country_currency_key {
    my ($record) = @_;

    return join(':', map { str($record->{$_}) } @TABLE_FIELDS);
}

=head2 _deserialize_country_currency_key

    Делает из сериализованной версии вида '0:123:KZT:3' ссылку на хеш вида {is_agency => 0, region_id => 123, currency => 'KZT', firm_id => 3}

=cut

sub _deserialize_country_currency_key {
    my ($serkey) = @_;

    my @values = split ':', $serkey;

    return { zip(@TABLE_FIELDS, @values) };
}

=head2 _array2hash

    Преобразует массив с данными о стране/валюте в хеш вида {"is_agency:region_id:currency:firm_id"} = 1;

=cut

sub _array2hash {
    my ($country_currencies_array) = @_;

    $log->die('invalid _array2hash argument') unless $country_currencies_array && ref($country_currencies_array) eq 'ARRAY';

    my %currencies_by_country;
    for my $record (@$country_currencies_array) {
        $currencies_by_country{_serialize_country_currency_key($record)} = 1;
    }

    return \%currencies_by_country;
}
