#!/usr/bin/perl

use my_inc "..";

# $Id$

=head1 METADATA

# NB! запускать раньше 14 нежелательно - курс может мигать! https://st.yandex-team.ru/DIRECT-51823
<crontab>
    time: 13,43 14-23 * * *
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>
<crontab>
    time: 13,43 0-13 * * *
    params: --check-only
    <switchman>
        group: scripts-other
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host: checks_auto.direct.yandex.ru
    name: scripts.ppcFetchCurrencyRates.working
    ttl: 2h
    tag: direct_group_internal_systems
    <notification>
        template: on_status_change
        status: CRIT
        status: OK
        method: telegram
        login: DISMonitoring
    </notification>
</juggler>

<juggler_check>
    host:       checks.direct.yandex.ru
    name:       direct.fresh_currency_rates.ppcdict
    raw_events: direct.fresh_currency_rates.ppcdict.production
    ttl:        2h10m
    tag:        direct_group_internal_systems
    <notification>
        template: on_status_change
        status: CRIT
        status: WARN
        status: OK
        method: telegram
        login: DISMonitoring
    </notification>
</juggler_check>
<juggler_check>
    host:       checks.direct.yandex.ru
    name:       direct.fresh_currency_rates.ppc
    raw_events: direct.fresh_currency_rates.ppc_$shard.production
    sharded:    1
    ttl:        2h10m
    tag:        direct_group_internal_systems
    <notification>
        template: on_status_change
        status: CRIT
        status: WARN
        status: OK
        method: telegram
        login: DISMonitoring
    </notification>
</juggler_check>

<crontab>
    time: 13,43 8-23 * * *
    flock: 1
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>
<crontab>
    env: SETTINGS_LOCAL_SUFFIX=DevTest
    time: 13,43 8-23 * * *
    package: conf-dev
</crontab>
<crontab>
    env: SETTINGS_LOCAL_SUFFIX=Dev7
    time: 13,43 8-23 * * *
    package: conf-dev
</crontab>

=cut

=head1 NAME

    ppcFetchCurrencyRates.pl

=head1 DESCRIPTION

    Получает от Баланса курсы валют и складывает в PPCDICT, PPC.
    Курсы получаются по валютам, указанным в ключах BALANCE_CURRENCY_NAME хеша %Currencies::_CURRENCY_DESCRIPTION.

=head2 RUNNING

    Скрипт приспособлен к многократному запуску в течение дня. При запуске без параметров получет курсы
    валют начиная с последней даты, на которую удалось получить курсы всех валют, (не включительно) по 
    завтрашний день (включительно).
    Это поведение можно переопределить параметрами:
        --date — дата, на которую необходимо получить курсы валют. Данные будут получены только на эту дату.
                 Если курс на эту дату уже есть, он будет перезаписан "свежим". Это может понадобиться, 
                 если задним числом изменится уже полученный курс или будет "дырка" в датах, за которые есть курсы.

    ./ppcFetchCurrencyRates.pl
    ./ppcFetchCurrencyRates.pl --date 2011-09-25

=cut

use warnings;
use strict;
use utf8;

use Carp qw/longmess/;
use List::MoreUtils qw/firstval/;
use List::Util qw/min/;

use Yandex::Advmon;
use Yandex::Balance qw/balance_get_currency_rate/;
use Yandex::DBTools;
use Yandex::SendMail qw/send_alert/;
use Yandex::TimeCommon;
use Yandex::Validate qw/is_valid_date/;

use Settings;
use ScriptHelper;

use Currencies;
use Property;
use ShardingTools qw/ppc_shards/;

use open ':std' => ':utf8';

=head3 %dbs

    Данные про базы, в которые надо записать курсы.

        db -- БД, в которой существует таблицы currency_rates

=cut

my @dbs = (
    {
        db => PPCDICT,
        dbname_for_log => 'ppcdict',
        property => Property->new('FETCH_CURRENCY_RATE_LAST_SUCCESS_DATE'),
        monitor_filename => 'currency_rates_last_fetched.date',
    },
    (map { {
        db => PPC(shard => $_),
        dbname_for_log => "ppc:$_",
        property => Property->new('FETCH_CURRENCY_RATE_PPC_SHARD'.$_.'_LAST_SUCCESS_DATE'),
        monitor_filename => 'currency_rates_ppc_shard'.$_.'_last_fetched.date',
    }} ppc_shards()),
);
$log->out('Started');

my $fetch_for_date;
my $CHECK_ONLY;
extract_script_params(
    'date=s' => \$fetch_for_date,
    'check-only' => \$CHECK_ONLY,
);

if ($CHECK_ONLY) {
    check_rates_dates();
    $log->out("skip fetching rates due to --check-only param");
    juggler_ok(description => 'Выполнена только проверка актуальности курсов');
    exit;
}

my @currencies_to_fetch = grep {$_} map { $_->{BALANCE_CURRENCY_NAME} } values %Currencies::_CURRENCY_DESCRIPTION;
$log->die('no currencies to fetch') unless @currencies_to_fetch;

my @dates;
if ($fetch_for_date) {
    die 'ошибочный формат даты' unless $fetch_for_date && is_valid_date($fetch_for_date);
    @dates = ($fetch_for_date);
} else {
    for my $db_data (@dbs) {
        my $property = $db_data->{property};
        my $last_date_with_rate = $property->get();
        $db_data->{last_time_with_rate} = mysql2unix($last_date_with_rate) if $last_date_with_rate;
    }

    # если для какой-то базы это первое получение курсов, то забираем с первой даты, на которую есть курсы всех валют в другой базе, чтобы данные в обеих таблицах совпадали
    # при этом в базу, в которой уже есть курсы, полученные курсы также запишутся
    my $db_with_data = firstval {defined $_->{last_time_with_rate}} @dbs;
    if ($db_with_data) {
        for my $db (grep {!defined $_->{last_time_with_rate}} @dbs) {
            my $first_date_with_rate = get_one_field_sql($db_with_data->{db}, ['SELECT min(date) FROM (SELECT date FROM currency_rates', WHERE => {currency => \@currencies_to_fetch}, 'GROUP BY date HAVING COUNT(DISTINCT currency) = ?) x'], scalar(@currencies_to_fetch));
            $db->{last_time_with_rate} = mysql2unix($first_date_with_rate);
            $log->out("DB $db->{dbname_for_log} have no currency rates yet. Fetching from $first_date_with_rate as other tables have all rates for this date");
        }
    }

    # получаем данные, начиная с минимальной даты, на которую есть курсы во всех таблицах
    my $min_last_time_with_rate = min map {$_->{last_time_with_rate}} @dbs;
    if ($min_last_time_with_rate != mysql2unix(tomorrow())) {
        @dates = get_distinct_dates(unix2mysql($min_last_time_with_rate), tomorrow());
    }
}

if (@dates) {
    my $some_dates_skipped = 0;
    for my $date(@dates) {
        my @data_to_insert;
        my $all_currencies_fetched = 1;
        for my $currency(@currencies_to_fetch) {
            my $rate = eval { balance_get_currency_rate($currency, $date); };
            if (defined $rate) {
                if ($rate > 0) {
                    $log->out("Rate for currency $currency for date $date is $rate");
                    push @data_to_insert, [$currency, $date, $rate];
                } else {
                    # ошибок не случилось, но курс не найден (например, ещё слишком рано спрашивать курс на завтра)
                    $log->out("No rate for currency $currency for date $date found");
                    $all_currencies_fetched = 0;
                }
            } else {
                $log->warn("Error getting data for currency $currency: $@");
                $all_currencies_fetched = 0;
            }
        }
        if (@data_to_insert) {
            # записываем полученный курс во все таблицы (даже если по last_time_with_rate понятно, что в этой таблице уже есть курс), чтобы не было расхождений
            for my $db_data (@dbs) {
                $log->out("Writing fetched data to $db_data->{dbname_for_log}");
                my $table_prefix = $db_data->{table_prefix} || '';
                my $count = eval { do_mass_insert_sql($db_data->{db}, "INSERT INTO ${table_prefix}currency_rates (currency, date, rate) VALUES %s ON DUPLICATE KEY UPDATE rate = VALUES(rate)", \@data_to_insert) };
                if (!defined $count) {
                    my $msg = "Error writing to $db_data->{dbname_for_log}: $@";
                    $log->warn($msg);
                    send_alert(Carp::longmess($msg), 'ppcFetchCurrencyRates error');
                    $db_data->{some_dates_skipped} = 1;
                }
            }
        }
        if ($all_currencies_fetched) {
            $log->out("Currency rates for date $date have been successfully fetched");
            for my $db_data (@dbs) {
                if (!$db_data->{some_dates_skipped} && !$fetch_for_date) {
                    $db_data->{property}->set($date);
                    monitor_values({"direct.$db_data->{monitor_filename}" => mysql2unix($date)}); # XXX сложная проверка direct.currency_rates_expire_in,expr `cat /var/www/ppc.yandex.ru/protected/monitor/currency_rates_last_fetched.date` + 86400 - `date +%s`
                }
            }
        } else {
            for my $db_data (@dbs) {
                $db_data->{some_dates_skipped} = 1;
            }
        }
    }
} else {
    $log->out('Rate data is already fresh');
}

check_rates_dates();
juggler_ok(description => 'Выполнено получение новых данных и проверка актуальности курсов');

$log->out('Finished');
exit;

use constant {
    WARN_BORDER => 51500,   # 14:50
    CRIT_BORDER => 58800,   # 16:20
};

sub check_rates_dates {
    $log->out('checking rate dates for juggler');
    for my $db_data (@dbs) {
        my $db_name = ($db_data->{dbname_for_log} =~ s/:/_/gr);
        my $last_time_with_rate = mysql2unix($db_data->{property}->get() // 0);
        juggler_check(service => "direct.fresh_currency_rates.$db_name",
                      description => "Возраст (в секундах) последних курсов валют (отрицательный - отлично, есть курс на завтра; положительный - чем больше, тем хуже, ближе к 86400 - совсем плохо - нет курсов на завтра)",
                      value => (time() - $last_time_with_rate),
                      warn => WARN_BORDER,
                      crit => CRIT_BORDER,
                      );
    }
}

