#!/usr/bin/perl

=head1 METADATA

<crontab>
    time: */5 * * * *
    params: --first-aid
    sharded: 1
    <switchman>
        group: scripts-other
        <leases>
            mem: 260
        </leases>
    </switchman>
    package: scripts-switchman
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    raw_events: scripts.ppcFirstAidSend.working.$type.shard_$shard
    sharded:    1
    vars:       type=first_aid
    ttl:        1h
    tag: direct_group_internal_systems
</juggler>

<crontab>
    time: */10 * * * *
    params: --first-aid
    sharded: 1
    <switchman>
        group: scripts-test
    </switchman>
    package: conf-test-scripts
</crontab>

=cut

=head1 NAME

$Id$

=head1 DESCRIPTION

    Нужен для МКС -- это способ менеджерам-саппортам узнавать о новых пользователях.
    
    Скрипт, собирает из camp_payments_info данные о зачислениях всех пользователей со времени предыдущего запуска.
        Если пользователь не был на "Первой помощи", то отправляет.

    Нужно, чтобы не загружать дополнительной логикой BalanceClient::NotifyOrder2.

    Отправка кампаний клиентов на ПП требуется после положительной модерации кампании и после поступления денег на кампанию.
    
    Отклоненные кампании по-прежнему отправляются в moderateClientNew.pl

    Условия отправки заявки на первую и вторую помощи для кампании:
       https://wiki.yandex-team.ru/EkaterinaBoguslavskaya/fareasons

    5 марта 2014: DIRECT-29627 - Отключена генерация заявок на вторую помощь (возможно временно).

=cut

use Direct::Modern;

use Yandex::DBTools;
use Yandex::ListUtils;
use Yandex::TimeCommon;

use my_inc "..";

use Campaign::Types;
use Campaign;
use Common qw/:globals :subs/;
use Mediaplan qw/add_optimizing_request is_second_aid/;
use Notification;
use Property;
use RBAC2::Extended;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';
use Settings;

=head1 SUBROUTINES/METHODS/VARIABLES

=cut

my ($first_aid, $second_aid, $once);

extract_script_params(
    "first-aid" => \$first_aid,
    "second-aid" => \$second_aid,
    "once" => \$once,
);
my $rbac = RBAC2::Extended->get_singleton(1);

if ($first_aid) {
    sendFirstAid();
    juggler_ok(service_suffix => 'first_aid');
} 

exit;

=head2 sendFirstAid

    Создает автоматические заявки на Первую Помощь.

=cut
sub sendFirstAid {

    $log->out("Start (First Aids)");
    my $max_current_time = 0;

    my $check_prop = new Property("last_check_first_aid_$SHARD");
    my $prev_time = $check_prop->get() || (time() - 24 * 3600);

    my $data = get_all_sql(PPC(shard => $SHARD), "SELECT c.uid, c.cid, c.type, unix_timestamp(cpi.last_payment_time) as ulogtime
                                                   FROM camp_payments_info cpi 
                                                        JOIN campaigns c using (cid)
                                                  WHERE cpi.last_payment_time > ?", unix2mysql($prev_time));
    if (scalar(@$data)) {
        $max_current_time = [sort {$b->{ulogtime} <=> $a->{ulogtime}} @{$data}]->[0]->{ulogtime};
        # Если клиент с лёгким интерфейсом уже давно пользуется кампанией и уже делал несколько платежей, 
        # то создавать ему заявку на ПП не нужно.
        my $cids_with_one_payment = Campaign::filter_campaigns_for_first_aid([map {$_->{cid}} @$data]);
        $log->out("cids_with_one_payment: ".join(",", nsort @{$cids_with_one_payment || []}));
        my %cids_with_one_payment = map {$_ => 1} @$cids_with_one_payment;
        my %PAYMENT_CAMP_TYPES_FOR_FA = map {$_ => 1} @{ get_camp_kind_types('payment_on_types_for_fa') };

        @$data = grep {
            $cids_with_one_payment{$_->{cid}}
            && exists $PAYMENT_CAMP_TYPES_FOR_FA{$_->{type}}
        } @$data;
    }

    foreach my $l (@$data) {

        # выбираем главного представителя и все его кампании
        my $campaigns = get_user_camps($l->{uid})->{campaigns};

        my ($camp_info) = grep {$_->{cid} == $l->{cid}} @$campaigns;

        next if ! defined $camp_info && $l->{type} ne 'wallet';

        # проверяем возможность предоставляения услуги "Первой помощи"
        my $vars = {campaigns=>$campaigns};
        Common::set_optimize_camps_vars($l->{uid}, $vars, automatic=>1);
        if ($vars->{optimize_camp}->{no_enable_optimize_camps}) {
            $log->out("can't optimize, skip");
            $log->out($vars->{optimize_camp});
            next;
        }

        # если оптимизация возможна - то отправляем
        # для кошелька проверяем все кампании под ним
        my $opt_camps = $l->{type} eq 'wallet' ? [ grep { $_->{wallet_cid} == $l->{cid} } @$campaigns  ] : [ $camp_info ];
        for my $camp (@$opt_camps) {
            if ($camp->{may_be_optimized}
                && !(get_one_field_sql(PPC(shard => $SHARD), "select 1 from optimizing_campaign_requests where cid = ? limit 1", $camp->{cid}))) {

                $log->out(sprintf("FA request was added for - uid: %d; cid: %d; ", $l->{uid}, $camp->{cid}));
                add_optimizing_request($camp->{cid}, {is_automatic => 1});
                send_notification($camp->{cid});
            }
        }
    }
    $check_prop->set($max_current_time) if ($max_current_time); 
    $log->out("Finish (First Aids)");

}

sub send_notification {
    my $cid = shift;
    my $mail_vars = get_one_line_sql(PPC(cid => $cid), "select u.login, u.FIO as fio, u.email, c.name as name_camp, c.cid, co.auto_optimize_request, u.uid, o.req_type
                                           from campaigns c 
                                                left join camp_options co using(cid)
                                                left join optimizing_campaign_requests o using(cid)
                                                join users u on u.uid=c.uid 
                                           where o.status='New' and c.cid=?", $cid);
    $mail_vars->{is_second_aid} = Mediaplan::is_second_aid($mail_vars->{req_type});    
    add_notification($rbac, 'optimization_request', $mail_vars);

}
