#!/usr/bin/perl

use my_inc "..";

=head1 METADATA

<crontab>
   time: */59 * * * *
   sharded: 1
   package: scripts-switchman
   <switchman>
       group: scripts-other
       <leases>
           mem: 50
       </leases>
   </switchman>
</crontab>
<juggler>
    host:   checks_auto.direct.yandex.ru
    sharded: 1
    ttl:            2h
    tag: direct_group_internal_systems
</juggler>

=cut

=head1 NAME

    ppcForceCurrencyConvert.pl

=head1 DESCRIPTION

    Ставит в очередь на конвертацию клиентов с запланированной датой
    принудительной конвертации в валюту

=head1 RUNNING

    Можно запускать по конкретным клиентам:
        --login — логин клиента или одного из его представителей

    ./protected/ppcForceCurrencyConvert.pl --shard-id 1
    ./protected/ppcForceCurrencyConvert.pl --shard-id=2 --login holodilnikru

=cut

use Direct::Modern;

use Settings;
use ScriptHelper sharded => 1, 'Yandex::Log' => 'messages';

use Client;
use Primitives;
use PrimitivesIds;
use RBACElementary;
use RBAC2::Extended;
use Client::ConvertToRealMoney ();
use Property;

use Yandex::TimeCommon;
use Yandex::DBTools;

my $OPERATOR_UID = 1;

=head1 SUBROUTINES/METHODS/VARIABLES

=head2 $CONVERT_QUEUE_INTERVAL

    За какой интервал до начала даты конвертации ставить клиента в очередь на конвертацию

=cut

my $CONVERT_QUEUE_INTERVAL = '24 HOUR';

my $LASTRUN_PROPERTY_NAME = "ppcForceCurrencyConvert_last_run_date_shard$SHARD";

my $logins = [];
extract_script_params(
    'login=s@' => $logins,
);

$log->out('START');

my (@additional_conds, $prop);
my $today = today();
if ($logins && @$logins) {
    $log->out('Working only on logins:', $logins);
    my @only_client_ids = map { get_clientid(uid => get_uid_by_login2($_)) } @$logins;
    @additional_conds = ('AND', {'fcc.ClientID' => \@only_client_ids});
} else {
    $prop = Property->new($LASTRUN_PROPERTY_NAME);
    my $last_date = $prop->get() // '';
    if ($last_date eq $today) {
        $log->out("Last run date according to $LASTRUN_PROPERTY_NAME property is $last_date, exitting");
        juggler_ok(description => 'Script has already done its work today');
        exit 0;
    }
}

$log->out('Connecting to RBAC');
my $rbac = eval { RBAC2::Extended->get_singleton(1) } or $log->die("Can't connect to RBAC: $@");

$log->out('Fetching clients to queue force currency convert');
my $clients = get_all_sql(PPC(shard => $SHARD), ["
        SELECT fcc.ClientID
             , fcc.currency
             , fcc.country_region_id
        FROM force_currency_convert fcc
        INNER JOIN clients_to_force_multicurrency_teaser t ON fcc.ClientID = t.ClientID
        INNER JOIN client_teaser_data_lastupdate ctdl ON ctdl.ClientID = fcc.ClientID
        LEFT JOIN currency_convert_queue q ON fcc.ClientID = q.ClientID
        WHERE
            fcc.convert_date IS NOT NULL
            AND q.ClientID IS NULL
            AND NOW() + INTERVAL $CONVERT_QUEUE_INTERVAL >= fcc.convert_date
            AND ctdl.last_update > NOW() - INTERVAL $Client::ConvertToRealMoney::COUNTRY_CURRENCY_LAST_UPDATE_INTERVAL_QUEUE_DAYS DAY
    ", @additional_conds, '
        ORDER BY fcc.convert_date
    ']);
if (@$clients) {
    my $clients_cnt = scalar @$clients;
    $log->out("Got $clients_cnt ClientID's to queue currency convert");

    my $client_ids = [ map { $_->{ClientID} } @$clients ];

    my $daily_client_limit = Client::ConvertToRealMoney::get_force_convert_daily_queue_client_limit();

    my $clients_nds = mass_get_client_NDS($client_ids, fetch_for_ynd_fixed_too => 1);
    my $clients_currencies = mass_get_client_currencies($client_ids);
    my $clients_chief_rep = rbac_get_chief_reps_of_clients($client_ids);

    my $clid2agid = Primitives::mass_get_client_first_agency($client_ids);

    my $clients_queued_cnt = 0;
    for my $client (@$clients) {
        my $client_id = $client->{ClientID};
        my $country = $client->{country_region_id};
        my $currency = $client->{currency};
        my $client_nds = $clients_nds->{$client_id};

        my @country_currencies;
        my $error = Client::can_convert_to_real_money(
            ClientID => $client_id,
            NDS => $client_nds,
            client_currencies => $clients_currencies->{$client_id},
            client_chief_uid => $clients_chief_rep->{$client_id},
            country_currency_ref => \@country_currencies,
            agency_id => $clid2agid->{$client_id},
        );
        if ($error) {
            $log->out("Got error for ClientID $client_id from can_convert_to_real_money, skipping", $error);
            next;
        }

        my $start_convert_date = tomorrow();
        my $convert_type = Client::ConvertToRealMoney::get_convert_type($client_id, $currency, $client_nds, \@country_currencies);

        if ($convert_type eq 'MODIFY') {
            $clients_queued_cnt++;
            if ($clients_queued_cnt > $daily_client_limit) {
                $log->out("Limit of $daily_client_limit MODIFY clients reached, stopping loop");
                last;
            }
        }

        $log->out("Queuing $convert_type convert of ClientID $client_id for $start_convert_date into currency $currency with country $country");
        do_in_transaction {
            # ставим заявку на переход в очередь
            do_insert_into_table(PPC(ClientID => $client_id), 'currency_convert_queue', {
                ClientID => $client_id,
                uid => $OPERATOR_UID,
                convert_type => $convert_type,
                new_currency => $currency,
                country_region_id => $country,
                email => undef,
                start_convert_at => $start_convert_date,
                state => 'NEW',
                in_state_since__dont_quote => 'NOW()',
            });

            # записываем дату перехода
            do_insert_into_table(PPC(ClientID => $client_id), 'client_currency_changes', {
                ClientID => $client_id,
                currency_from => 'YND_FIXED',
                currency_to => $currency,
                date => $start_convert_date,
            });
        };
    }
} else {
        $log->out('No clients found to queue force currency convert');
}

if ($prop) {
    $log->out("Writing date $today to property");
    $prop->set($today);
}

$log->out('Sending OK to juggler');
juggler_ok();

$log->out('FINISH');
